(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  TransisE, (Version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(**************************************************************)
(* SMQ.ML : Simple queue implementation *)
(* Author: Ohad Rodeh, 12/96 *)
(* Based on code by: Mark Hayden - from iq.ml.*)
(**************************************************************)
open Printf
(**************************************************************)
let name = "SMQ"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type 'a opt = 
  | Full of 'a
  | Unset					
  | Reset					

type 'a t = {
  mutable head   : int ; (* relative to offset *)
  mutable tail 	 : int ; (* relative to offset *)
  mutable read	 : int ; (* relative to offset *)

  mutable offset : int ;
  mutable max	 : int ; (* Not relative *)
  mutable array  : 'a opt array
}

(**************************************************************)
(* Debugging printf.
*)
let debug = true

(**************************************************************)
(* The resize function realignes the array, and increases its
   size, if need be. 
*)

let fix_max iq =
  iq.max <- Array.length iq.array - iq.offset

let resize iq size =
  let align iq =
    let size = Array.length iq.array
    and array = iq.array in
    let first =
      let rec find_first i =
      	if i >= size then 
	  match array.(i) with 
      	  | Reset -> find_first (succ i)
	  | _ -> i
	else 0
      in find_first 0
    in
    if first > 0 then (
      Array.blit iq.array first iq.array 0 (size-first) ;
      Array.fill iq.array (size-first) first Unset ;
      iq.offset <- iq.offset - first ;
      fix_max iq
   )
  in
  
  align iq ;
  let size = size + iq.offset in
  if size * 2 + 10 >= Array.length iq.array then (
    let add_on = array_create name (size*3+10) Unset in
    iq.array <- Array.append iq.array add_on ;
    fix_max iq
  )

(**************************************************************)

let nth iq i =
  let i = i + iq.offset in
  let array = iq.array in
  if i < 0 or i >= Array.length array then 
    None
  else let v = array.(i) in
  match v with
  | Reset | Unset -> None
  | Full v -> Some v


let nth_set iq i v =
  if i >= iq.max then
    resize iq (succ i) ;
  let i = i + iq.offset in
  iq.tail <- max iq.tail i+1;
  let array = iq.array in
  if i>=0 then (
    match array.(i) with 
    | Reset | Unset -> (array.(i) <- Full v ; true)
    | Full _ -> false)
  else false

let clear iq ed =
  let ed = ed + iq.offset in
   if ed >= 0 & ed < iq.read then (
     for i=iq.head to ed do
       iq.array.(i) <- Reset
     done;
     iq.head <- ed)

let advance_tail iq i = 
  if i >= iq.max then 
    resize iq (succ i);
  let i = i + iq.offset in
  iq.tail <- max iq.tail (succ i)

(**************************************************************)

let empty () = 
  let iq = {
    array 	= array_create name 4 Unset ;
    head 	= 0 ;
    tail 	= 0 ;
    read  	= 0 ;
    offset 	= 0 ;
    max		= 0
  } in
  fix_max iq ;
  iq

(**************************************************************)
(* READ_HOLE: Returns the index of the start and end of any
 * hole at the beginning of the "window."  Used for NAKs...
 *)

let read_hole iq =
  let rec advance_read () = 
    match nth iq iq.read with 
    | Some m -> 
    	iq.read <- (succ iq.read);
   	advance_read ()
    | None -> ()
  in
   
  advance_read ();
  if iq.tail <= iq.read then (
    None
    ) else (
    let first_set =
      let rec loop i =
	if i >= iq.tail then i
	else (
	  match nth iq i with
	  | Some m -> i
	  | None -> loop (succ i)
	)
      in loop iq.read
    in Some (iq.read, pred first_set)
  )

(**************************************************************)

let list_of_smq_interval (head,tail) iq f =
  let rec loop i =
    if i > tail then []
    else match nth iq i with
    | Some v -> (let res = f v in 
      match res with 
      |	Some r -> (i,r)::(loop (succ i))
      |	None -> loop (succ i))
    | None -> loop (succ i)
  in loop head
   


