(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  TransisE, (Version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(* EVS.ML : Extended Virtual Synchrony *)
(* Author: Ohad Rodeh, 12/96 *)
(**************************************************************)
(*

EVS - Extended Virtual Synchrony
----------------------------------
 Delay the EView event, ECast DoneView. Wait until all members in
Old_view\failures cast DoneView. ECast transitional view, and then,
the delayed regular view. 

*)
(**************************************************************)
open Event
open Util
open Layer
open Trans
open View
(**************************************************************)
let name = Trace.source_file "EVS"
(**************************************************************)

type header =
| NoHdr
| DoneView

type state = {
    failed	       : bool array; 
    done_view  	       : bool array;
    mutable dl_dn_view : dn option
  }

(**************************************************************)

let is_some = function 
  | None -> false
  | Some a -> true

let dump (ls,vs) s =
  eprintf "EVS:dump:%s\n" ls.name;
  eprintf "  failed    =%s\n" (string_of_bool_array s.failed);
  eprintf "  done_view =%s\n" (string_of_bool_array s.done_view);
  eprintf "  dl_dn_view=%b\n" (is_some s.dl_dn_view)

(**************************************************************)

let init () (ls,vs) = {
  failed      	= array_create name ls.nmembers false;
  done_view     = array_create name ls.nmembers false;
  dl_dn_view    = None
}

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let assert a s = if not a then failwith s in
  let log = Trace.log name ls.name in
  
    (* Check if we're done.
       *)
  let check_ok () =
    if is_some s.dl_dn_view then (
    let all_true = ref true in
    for i = 0 to pred (ls.nmembers) do
      if not (s.done_view.(i) || s.failed.(i)) then
	all_true := false
    done ;
      if !all_true then (
	log (fun () -> sprintf "Transitional view=%s" 
	  (string_of_bool_array s.done_view)); 
    	dnnm (create name ETransView[(Transitional s.done_view)]); 
      	dnnm (some_of s.dl_dn_view);
	s.dl_dn_view <- None
      )
    )
  in

  let up_hdlr ev abv hdr = match getType ev, hdr with
  | _,NoHdr -> up ev abv
  | _ -> failwith "bad up event"

  and uplm_hdlr ev hdr = match getType ev,hdr with
  | ECast, DoneView -> 
      free name ev;
      s.done_view.((getOrigin ev)) <- true;
      check_ok ()
  | _ -> failwith "bad up event"


  and upnm_hdlr ev = match getType ev with

  (* EFail: Mark some members as being failed.  Check if
   * we're done blocking. 
   *)
  | EFail ->
      let ranks = getFailures ev in
      List.iter (fun rank ->
        s.failed.(rank) <- true
      ) ranks ;
      upnm ev ;
      check_ok ()

  (* Caught my own event
   *)
  | ETransView -> free name ev
  | EDump -> (dump (ls,vs) s; upnm ev)
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv NoHdr

  and dnnm_hdlr ev = match getType ev with

    (* EView - send DoneView, and wait for others.
       *)
  | EView -> 
      s.dl_dn_view  <- Some ev;
      s.done_view.(ls.rank) <- true;
      if ls.nmembers > 1 then dnlm (create name ECast[]) DoneView;
      check_ok ()

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr_state init hdlrs None NoOpt args vs

let _ = Layer.install name (Layer.init l)

(**************************************************************)





