(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  transise, (version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(* DAG.ML : directed acyclic graph handling *)
(* Author: Ohad Rodeh, 12/96 *)
(* Based on code by: Mark Hayden *)
(**************************************************************)
(**************************************************************)
open Trans 
open Smq
open Printf
(**************************************************************)
(*  The record kept at each node includes - 
*)

type cmsg_id = {
    rnk  : rank;
    seq  : seqno
  }

type 'a cmsg = {
    data             : 'a;                 (* The acutal information *)
    ack_list         : cmsg_id list;       (* dag acks *)
    mutable ack_vct  : seqno option array; (* vector of infered acks *)
    mutable pending  : bool                (* This msg arrived out of order *)
  } 

type 'a cdag = {
    qs                   : 'a cmsg Smq.t array;
    dlvr_vct             : seqno option array; (* Delivery vector *)
    mutable dlvr_roots   : cmsg_id list;       (* Deliverable roots *)
    mutable min_vct      : seqno option array;
    holes                : (seqno * seqno) option array ;
    nmemb                : int;
    mutable pending_cnt  : int
  } 

(**************************************************************)

(* Auxilary functions
*)
val get_min        : 'a cdag -> seqno array
val get_tail       : 'a cdag -> seqno array
val get_dlvr_vct   : 'a cdag -> seqno option array
val get_dlvr_roots : 'a cdag -> cmsg_id list

val advance_tail   : 'a cdag -> seqno array -> unit

val create : int -> 'a cdag
(* [create n] Creates a new dag, with [n] iq's.
*)

val gc     : 'a cdag -> unit
(* [gc dag] does garbage collection. Compute a vector of stable
   messages - min_vct, and clear the dag below that vector. Update
   dag.min_vct.
*)

val insert_my_msg : 'a cdag -> rank -> 'a 
  -> (rank * seqno * 'a * seqno option array) * (cmsg_id list)
(* [insert_my_msg dag r msg]. Inserts a [msg] into the [dag],
   to iq [r]. returns the computed ack vector, and ack list
   of [msg].
*)

val insert : 'a cdag -> rank -> seqno -> 'a -> cmsg_id list
  -> (rank * seqno * 'a * seqno option array) list
(* [insert dag r seq msg ack_list]. Inserts a [msg] into the [dag],
   to iq [r], sequence number [seq], with ack_list [ack_list]. It
   returns a list of deliverable messages.
*)

val list_of_dag_interval : 'a cdag -> rank -> (seqno * seqno) 
  -> (seqno * ('a * cmsg_id list)) list
(* [list_of_dag_interval dag r (lo,hi)] Computes an interval of a dag.
   *)

val stat          : 'a cdag -> unit
(* [stat dag] prints the [dag]'s state. Used for debugging.
   *)

val compute_holes : 'a cdag -> (rank * seqno * seqno) list
(* [compute_holes dag] Computes the closest hole to head at each of
   the queues.  
*)

val missing       : 'a cdag -> cmsg_id list -> (rank * seqno * seqno ) list
(* [missing dag ack_list] computes the list of missing messages out
   of an ack_list.
   *)
