(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  transise, (version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(* DAG.ML : directed acyclic graph handling *)
(* Author: Ohad Rodeh, 12/96 *)
(**************************************************************)
(* 
Overview
--------
  This file implements a dag with holes. This information structure
keeps a queue of messages for each member [array qs]. Messages
causally depend on each other [msg.ack_list]. For each message, we
compute its ack_vct - the complete vector of messages which it
follows.  A msg always follows itself.

  The dag's state includes [dlvr_vct] - the vector of deliverable
messages, [dlvr_roots] - the deliverable roots of the dag, [min_vct] -
the vector of stable messages, [holes] - list of holes, nearest to
dlvr_vct, and nmemb - the number of different processes logged in the dag. 
  
  Pending_cnt counts the number of messages, that arrived out of order, 
and are still awaiting delivery. It is used for optimization. 
*)
(**************************************************************)
(* Example 
   -------
This is an example dag, with 3 processors {A,B,C}. It has
 3 Smq.t queues -
  A:[1A]      [2A]  [3] [4A]      
        \         \    /    \
  B:     [1B]      [2B][3- 6][7B] 
             \    /    \
  C:          [1C]      [2C]       

    Messages which were received are denoted by [1A][2B] .. Messages
  with are missing are denoted by [3][3- 6] ..

  The message information includes - 
       acklist  ack_vector  pending          
  7B's [4B]                 t
  4A's [2B]                 t
  2B's [2A,1C]  [2,2,1]     f         
  1C's [1B]     [1,1,1]     f         

The dag information includes - {
  qs         = The three queues {A,B,C};
  dlvr_vct   = [2,2,2];
  dlvr_roots = [2B,2C];
  min_vct    = [1,0,0];
  holes      = [|Some(3,3) ; Some(3,6), None|]
  nmemb      = 3
  pending_cnt= 2
}
*) 
(**************************************************************)
open Trans 
open Smq
open Printf
open Util
(**************************************************************)
type cmsg_id = {
    rnk  : rank;
    seq  : seqno
  }

type 'a cmsg = {
    data             : 'a;                 (* The acutal information *)
    ack_list         : cmsg_id list;       (* dag acks *)
    mutable ack_vct  : seqno option array; (* vector of infered acks *)
    mutable pending  : bool                (* This msg arrived out of order *)
  } 

type 'a cdag = {
    qs                   : 'a cmsg Smq.t array;(* Message queues *)
    dlvr_vct             : seqno option array; (* Delivery vector *)
    mutable dlvr_roots   : cmsg_id list;       (* Deliverable roots *)
    mutable min_vct      : seqno option array; (* Stable vector *)
    holes                : (seqno * seqno) option array ;
    nmemb                : int;                (* Number of members *)
    mutable pending_cnt  : int                 (* #messages out of order *)
  } 

(**************************************************************)

(* MH: no longer in Util module. (Version in Lset module also
 * orders the elements for efficiency.)
 *)

let subtract f = function
  | [] -> f
  | e ->
     let rec subtract_e = function
       | [] -> []
       | elem::l -> if List.mem elem e then subtract_e l else elem :: subtract_e l
     in subtract_e f

(**************************************************************)
(* Auxliray functions 
*)
let get_min  dag = Array.map (fun i -> int_of_some i) dag.min_vct
let get_tail dag = Array.map (fun iq -> pred iq.tail) dag.qs
let get_dlvr_vct dag = dag.dlvr_vct
let get_dlvr_roots dag = dag.dlvr_roots

let stat dag = 
  printf "Dag state= \n high=[|";
  Array.iter (fun i -> printf "%d|" i) (get_tail dag);
  printf "] \n dlvr=[|";
  Array.iter (fun i -> print_some i) dag.dlvr_vct;
  printf "] \n roots= [|";
  List.iter (fun {rnk=r;seq=s} -> 
    printf "(r=%d,s=%d)" r s) dag.dlvr_roots;
  printf "] \n min_vct= [|";
  Array.iter (fun i -> print_some i) dag.min_vct;
  printf "] \n holes= [|";
  Array.iter (function 
    | Some(lo,hi) -> printf "lo=%d,hi=%d|" lo hi
    | None -> printf "N|") dag.holes;
  printf "]\n";
  printf "pending_cnt=%d\n" dag.pending_cnt

let assert dag x y = if not x then (stat dag;failwith y)

let some_dec x = 
  match x with
  | None -> None
  | Some z -> if z=0 then None 
      else (Some (pred z))

let advance_tail dag vct = 
  Util.for_array (fun i v -> advance_tail dag.qs.(i) v) vct

let nth dag r s = Smq.nth dag.qs.(r) s


(**************************************************************)
(* Insert functions
   *)

(* Set a node in the dag
   *)
let fix dag org seq cm = 
  List.iter (fun {rnk=r; seq=s} -> 
    let msg = nth dag r s in
    if int_of_some dag.dlvr_vct.(r) <  s then (
      match msg with
      | None -> Smq.advance_tail dag.qs.(r) s
      | Some _ -> ())
      ) cm.ack_list

(* Is this message deliverable?
   *)
let deliverable dag org seq ack_list= 
  List.for_all (fun {rnk=r;seq=s} -> int_of_some dag.dlvr_vct.(r) >= s
    ) ack_list

let compute_ack_vct dag org seq cm  = 
    List.iter (fun {rnk=r;seq=s} -> 
      match nth dag r s with
      | None -> stat dag; failwith (sprintf "Deliverable message (%d,%d), is missing predecesor" r s)
      | Some msg -> max_vct cm.ack_vct msg.ack_vct
	  ) cm.ack_list;
     cm.ack_vct.(org) <- Some(seq)

(* Compute msg [cm]'s ack vector, update dlvr_vct+dlvr_roots.
*)
let deliver dag org seq cm = (
  compute_ack_vct dag org seq cm;
  dag.dlvr_vct.(org) <- Some seq;
  dag.dlvr_roots <- {rnk=org;seq=seq} :: (subtract dag.dlvr_roots cm.ack_list)
)

(* check the dag, row by row for any pending messages, which became 
   deliverable.
   *)
let check dag org seq cm = 
  let l = ref [(org,seq,cm.data,cm.ack_vct)] 
  and adv = ref true in 
  while dag.pending_cnt <> 0 && !adv do 
    let dlvr_plus = Array.map (fun i -> some_inc i) dag.dlvr_vct in 
    adv := array_exists (fun r s -> 
      match (nth dag r s) with 
      | None -> false
      | Some m -> 
	  if deliverable dag r s m.ack_list then (
	    assert dag m.pending "Delivering a non-pending msg";
	    l := !l @ [(r,s,m.data,m.ack_vct)];
	    m.pending <- false;
	    dag.pending_cnt <- pred dag.pending_cnt;
	    deliver dag r s m;
	    true) else false
	    ) dlvr_plus
  done;
  !l

(* Insert a message into the dag. Return a list of deliverable messages.
   *)
let insert dag org seq data ackl = 
  if int_of_some dag.min_vct.(org) < seq then 
    begin
      let cm = nth dag org seq in 
      match cm with 
      | None -> 
	  let acks = ref ackl 
	  and pend = ref false in
	  if seq> 0 then acks := {rnk=org;seq=(pred seq)} :: ackl;
	  if not (deliverable dag org seq !acks) then pend := true;
	  let msg = {
	  data     = data;
          ack_list = !acks;
	  ack_vct  = array_create name dag.nmemb None;
	  pending  = !pend
	  } in 
	  assert dag (Smq.nth_set dag.qs.(org) seq msg) "Cannot nth_set";
	  fix dag org seq msg;
	  if msg.pending then (
	  dag.pending_cnt <- succ dag.pending_cnt;
	  [])
	  else (
	  deliver dag org seq msg;
	  check dag org seq msg 
 	 )
	 | _ -> []
    end
  else []

let insert_my_msg dag org data = 
  let myseq = (some_inc dag.dlvr_vct.(org)) in
  let ack_list = subtract dag.dlvr_roots [{rnk=org;seq=(pred myseq)}] in
  let l = insert dag org myseq data dag.dlvr_roots in 
  assert dag (List.length l = 1) "Sanity:DAG:Could not insert my own message";
  ((List.hd l), ack_list)
         	   
(**************************************************************)

let create n = 
  {
  qs           = Array.map (fun () -> Smq.empty ()) (array_create name n ());
  dlvr_vct     = array_create name n None;
  dlvr_roots   = [];
  min_vct      = array_create name n None;
  holes        = array_create name n None;
  nmemb        = n;
  pending_cnt  = 0
} 

(**************************************************************)
(* Compute an interval of a dag.
   *)
let list_of_dag_interval dag org (head ,tail) =
  Smq.list_of_smq_interval (head,tail) dag.qs.(org) 
  (fun msg -> Some(msg.data, msg.ack_list))

(**************************************************************)
(* Clear the dag, below a certain vector
*)
let clear dag = 
  for i=0 to dag.nmemb-1 do
    match dag.min_vct.(i) with 
    | None -> ()
    | Some z -> Smq.clear dag.qs.(i) z
  done

(* garbage collection.  Compute a vector of stable messages -
   dag.min_vct, and clear the dag below that vector.  
   *)
let gc dag = (
  let mins = Array.copy dag.dlvr_vct in 
  try 
    for r1=0 to dag.nmemb-1 do 
      let s = dag.dlvr_vct.(r1) in 
      match s with
      | None -> failwith "NO"
      | Some s -> let top_msg = (nth dag r1 s) in 
	  match top_msg with 
	  | None -> stat dag; failwith "Sanity - deliverable message is Null"
	  | Some msg -> 
	      for r2=0 to dag.nmemb-1 do
      	      	mins.(r2)<- some_min msg.ack_vct.(r2) mins.(r2)
	      done
    done;
    dag.min_vct <- Array.map (fun i -> some_dec i) mins;
    clear dag 
  with (Failure "NO") -> ()
)
(**************************************************************)
(* This function computes the closest hole to head at each of the
   fifo queues.  
   *)
let compute_holes dag =
  let holes = ref [] in
  for i=0 to dag.nmemb-1 do
    match (Smq.read_hole dag.qs.(i)) with
    | Some (lo,hi) -> holes := !holes @[(i,lo,hi)]
    | None -> ()
  done;
  !holes

let missing dag ack_list = 
  let miss = ref [] in
  List.iter (function {rnk=r;seq=s} -> 
    if is_none (nth dag r s) then miss := !miss @ [(r,s,s)]) ack_list;
  !miss
(**************************************************************)



