(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  TransisE, (Version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(* AGREE.ML : Lansis protocol for Total order *)
(* Author: Ohad Rodeh, 12/96 *)
(**************************************************************)
(* 
General Description - First implementation
------------------------------------------
  We create a cut of agree messages through the dag, using a 
queue. 
*)
(**************************************************************)
open Trans
open Layer
open Event
open Util
open View
(**************************************************************)
let name = Trace.source_file "AGREE"
let failwith = make_failwith name
(**************************************************************)
type header = NoHdr

type ('a,'b) msg = 
  | Agree of up * ('a,'b)Layer.msg
  | ImAlive of up
  | Null of up

type ('a,'b) opt = 
  | Agr of seqno * ('a,'b) msg
  | Causal 
  | None

type ('a,'b) state = {
    mutable cut        : ('a,'b) opt array;
    mutable agr_q      : ('a,'b) msg Queue.t;
    mutable got_trans  : bool;
    mutable got_view   : bool
  } 
    
(**************************************************************)

type ('a,'b) t = (header,'a,'b,('a,'b) state) Layer.t

(**************************************************************)
let init () (ls,vs) =
  let n      = ls.nmembers in {
  cut        = array_create name n None;
  agr_q      = Queue.create ();
  got_trans  = false;
  got_view   = false
} 

(**************************************************************)

let opt_is_none = function 
  | None  -> true
  | _ -> false
      
let ev_of_msg = function
  | Agree(ev,abv) -> ev
  | ImAlive(ev) -> ev
  | Null(ev) -> ev

let seq_of_msg msg = 
  let ev = ev_of_msg msg in 
  let acks = getAckVct (ev) in
  some_of acks.((getOrigin ev))

let dump (ls,vs) s =
  eprintf " AGREE:dump:%s\n" ls.name;
  eprintf "Cut=[";
  Array.iter (function
    | Agr(seq, _)->   eprintf "Agr(%d)|" seq
    | None    ->   eprintf "N|"
    | Causal  ->   eprintf "C|") s.cut;
  eprintf "]\n Queue=";
  Queue.iter (function msg -> 
    eprintf "(%d,%d,%d)" (ev_of_msg msg).origin (seq_of_msg msg) (ev_of_msg msg).flags) s.agr_q;
  eprintf "\n got_trans=%b got_view=%b\n" s.got_trans s.got_view
  
(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let log = Trace.log name ls.name in
  let assert a m = if not a then (dump (ls,vs) s ;failwith m) in

  (* Check if message follows the head of the queue.
     *)
  let follows_cut msg = 
    let ev = ev_of_msg msg in
    let acks = getAckVct ev in
    let v = ref false in 
    for r=0 to pred ls.nmembers do 
      match s.cut.(r) with 
      | Agr(seq, _) -> if (int_of_some acks.(r)) >= seq then v:=true
      | _ -> ()
    done;
    !v 

  and mlt_up = function
    | Agree(ev,abv) -> up ev abv
    | ImAlive(ev)  -> upnm ev
    | Null(ev)     -> ()

  (* Check if the cut is full.
     *)
  and check_done () =
    if array_exists (fun i v -> match v with
    | Agr _ -> true
    | _ -> false) s.cut
    then (
      if s.got_view then 
	begin
	  let l = Array.map (function 
	    | Agr _ | Causal -> true
	    | None -> false) s.cut in 
	    log (fun () -> sprintf "[%d]%s" ls.rank (string_of_bool_array l))
	end;
      s.got_view
      or array_for_all (function 
	| Agr _ | Causal -> true
	| None -> false) s.cut
      )
    else false
  in

  (* Insert a message to queue [some_que].
     *)
  let insert_que msg some_que =
    let ev = ev_of_msg msg in 
    let org = (getOrigin ev) in 
    if (follows_cut msg) then (
	(match s.cut.(org) with 
	| None -> s.cut.(org) <- Causal;
      	| _ -> ());
      Queue.add msg some_que)
    else
	(match s.cut.(org) with
	| None -> (match msg with
	  | Agree _ -> s.cut.(org) <- Agr(seq_of_msg msg, msg)
	  | _ -> mlt_up msg)
	| _ -> failwith "SANITY-Que")
  in      
  (* Check if the cut is deliverable, ECast it, in lexicographic 
     order. Filter the rest of the messages, creating a new cut and
     queue.
     *)
  let rec check_ok () = 
    if check_done () then (
      Array.iter(function 
      	| Agr(_, Agree(ev,abv)) -> mlt_up (Agree(ev,abv))
      	| _ -> ()) s.cut;
      	s.cut <- array_create name ls.nmembers None;
      let tmp_q = Queue.create () in 
      Queue.iter (function msg -> insert_que (Queue.take s.agr_q) tmp_q
	) s.agr_q;  
      s.agr_q <- tmp_q;
      check_ok () 
      )
  in

  (* Catch all messages passing through tcausal, thus
     creating agreement.
     *)
  let up_hdlr ev abv hdr = match getType ev with 
  | ECast when (ev.flags land xxx_causal) <> 0 
    && not (getNoTotal ev) -> 
      insert_que (Agree(ev,abv)) s.agr_q;
      check_ok ()

      (* Let no_total messages through *)
  | ECast when (ev.flags land xxx_causal) <> 0 
    && not (getNoTotal ev) -> 
      up ev abv;
      insert_que (Null ev) s.agr_q;
      check_ok ()

  | _ -> up ev abv

  and uplm_hdlr ev hdr = failwith "unknown local message"

  and upnm_hdlr ev = match getType ev with
  | EAlive -> 
      insert_que (ImAlive ev) s.agr_q;
      check_ok () 

 | ETransView ->   
     s.got_trans <- true;
     upnm ev
      
  | EView -> 
      s.got_view <- true;
      check_ok ();
      assert (Queue.length s.agr_q = 0) "Queue <> empty, after EView";
      upnm ev

  | EDump -> ( dump (ls,vs) s ; upnm ev)
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv NoHdr

  and dnnm_hdlr = dnnm

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

(**************************************************************)



