(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* STABLE.ML : stability detection protocol *)
(* Author: Mark Hayden, 12/95 *)
(* Based on code by: Robbert vanRenesse *)
(* Note: Sequence numbers begin at 0 in each epoch.
 * Acknowlegdements give the number of messages acknowledged
 * so far in the epoch.  Thus the first messge is
 * acknowledged with '1'.  Similarly, the stability numbers
 * correspond to the number of messages that are stable in
 * the current epoch. *)
(**************************************************************)
open Trans
open Layer
open View
open Event
open Util
(**************************************************************)
let name = Trace.source_file "STABLE"
(**************************************************************)

type header = Gossip of (seqno array) * (bool array)

type state = {
  explicitack	   : bool ;
  sweep		   : Time.t ;
  acks             : seqno array array ;
  failed           : bool array ;
  mutable block_ok : bool ;
  mutable next_gossip : Time.t ;
  mutable blocking : bool ;

  (* BUG: actually, dbg_maxs is used in protocol *)
  mutable dbg_maxs : seqno array ;
  mutable dbg_mins : seqno array
}

(**************************************************************)

let dump (ls,vs) s =
  eprintf "STABLE:dump:%s\n" ls.name ;
  eprintf "  failed   =%s\n" (string_of_bool_array s.failed) ;
  eprintf "  dbg_mins =%s\n" (string_of_int_array s.dbg_mins) ;
  eprintf "  dbg_maxs =%s\n" (string_of_int_array s.dbg_maxs) ;
  for i = 0 to pred ls.nmembers do
    eprintf "  acks(%d)=%s\n" i (string_of_int_array s.acks.(i))
  done

(**************************************************************)

(* Note: the my_row field is set to point to one of the
 * subvectors in the acknowledgement matrix.  These are
 * defined separately as an optimization to prevent vector
 * lookups.  *)

let init () (ls,vs) = {
  explicitack	= Param.bool vs.params "stable_explicit_ack" ;
  sweep	        = Param.time vs.params "stable_sweep" ;

  failed        = array_create name ls.nmembers false ;
  acks          = Array.create_matrix ls.nmembers ls.nmembers 0 ; (* matrix (from,to) *)
  next_gossip   = Time.invalid ;
  blocking	= false ;
  block_ok      = false ;

  dbg_mins	= array_create name ls.nmembers 0 ;
  dbg_maxs  	= array_create name ls.nmembers 0
}

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let log = Trace.log name ls.name in
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in

  (*
   * DO_STABLE: Recalculate stability.  Send down a EStable event.
   *)
  let do_stable () =
    let mins = Array.copy s.acks.(ls.rank) in
    let maxs = array_create name ls.nmembers 0 in

    (* For each member, update ack entry and find new stability.
     *)
    for i = 0 to pred ls.nmembers do
      if not s.failed.(i) then (
        let row = s.acks.(i) in
        for j = 0 to pred ls.nmembers do
	  if row.(j) >| maxs.(j) then maxs.(j) <- row.(j) ;
	  if row.(j) <| mins.(j) then mins.(j) <- row.(j)
	done
      )
    done ;

    s.dbg_mins <- Array.copy mins ;
    s.dbg_maxs <- Array.copy maxs ;

    (* Send dn EStable event.
     *)
    dnnm (create name EStable[
      Stability mins ;
      NumCasts maxs
    ]) ;
    
    if s.block_ok && mins = maxs then (
      log (fun () -> sprintf "EBlockOk:release") ;
      s.block_ok <- false ;
      upnm (create name EBlockOk[]) ;
    ) ;
  in

  let up_hdlr ev abv () = up ev abv
  and uplm_hdlr ev hdr = match getType ev,hdr with

    (* Gossip Message: if from a live member, copy into
     * the origins row in my acknowledgement matrix.
     *)
  | (ECast|ESend), Gossip(remote,failed) ->
      let origin = getOrigin ev in
      if (not failed.(ls.rank))		(* BUG: could auto-fail him *)
      && not s.failed.(origin)
      then (
	let local = s.acks.(origin) in
	for i = 0 to pred ls.nmembers do
	  if remote.(i) >| local.(i) then
	    local.(i) <- remote.(i)
	done ;

	(* If I'm blocking and this gossip helps, try
	 * sending up stability array.  
         *)
(*
	if s.blocking 
        && local = s.dbg_maxs 
        then
	  dnnm (create name EStableReq[]) ;
*)
      ) ;
      free name ev

  | _ -> failwith "unknown local message"
   
  and upnm_hdlr ev = match getType ev with
      
    (* EInit: request a first timer alarm to bootstrap things.
     *)
  | EInit ->
      dnnm (timerAlarm name Time.zero) ;
      upnm ev

    (* EBlock: before passing event on, gossip about my
     * row in the stability matrix (this is an optimization).
     *)
  | EBlock ->
      s.blocking <- true ;
      dnnm (create name EStableReq[]) ;
      upnm ev

    (* EFail: mark the failed members and check if any
     * messages are now stable.
     *)
  | EFail ->
      List.iter (fun rank ->
      	s.failed.(rank) <- true
      ) (getFailures ev) ;

      (* Optimization, try to do a last minute GC.
       *)
      dnnm (create name EStableReq[]) ;
      upnm ev

  | EBlockOk ->
      if s.block_ok then
        failwith "2nd EBlockOk" ;
      if ls.nmembers = 1 (*|| Arge.get Arge.hack_no_sync(*HACK*)*) then (
        upnm ev
      ) else (
        log (fun () -> sprintf "EBlockOk:capture") ;
        s.block_ok <- true ;
        free name ev ;
      )

    (* ETimer: every so often:
     *   1. recalculate stability and deliver EStable event
     *   2. gossip about my row in the stability matrix
     *)
  | ETimer ->
      let time = getTime ev in
      (*if !verbose then dump (ls,vs) s ;*)
      if time >= s.next_gossip then (
      	let old = s.next_gossip in
	s.next_gossip <- Time.add time s.sweep ;
	dnnm (timerAlarm name s.next_gossip) ; (* request next gossip *)

      	if old <> Time.invalid then
          dnnm (create name EStableReq[])
      ) ;
      upnm ev

    (* Got reply to my stability request.
     *)
  | EStableReq ->
      let casts = getNumCasts ev in
      for i = 0 to pred ls.nmembers do
        s.acks.(ls.rank).(i) <- casts.(i)
      done ;
      do_stable () ;
      begin
  	let ev = create name ECast[Unreliable] in
	let my_row = Array.copy s.acks.(ls.rank) in
	let failed = Array.copy s.failed in
	dnlm ev (Gossip(my_row,failed))
      end ;
      upnm ev

  | EDump -> dump (ls,vs) s ; upnm ev
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv ()

  and dnnm_hdlr ev = match getType ev with

    (* EAck: update my entry in my row of the acknowledgement
     * matrix.
     *)
  | EAck ->
      begin
	match getAck ev with
	| RankSeqno(rank,seqno) -> (
	    if seqno > s.acks.(ls.rank).(rank) then (
	      s.acks.(ls.rank).(rank) <- seqno ;

	      (* If I'm blocking and everything is
	       * now stable, then gossip.
	       *)
  () (*
	      if s.blocking then (
		s.my_row.(ls.rank) <- s.ncasts ;
		if s.my_row = s.dbg_maxs then
		  dnlm (castEv *watch-out* name) (my_gossip s)
	      )
  *)
	    )
	  )
        | _ -> failwith "got unknown type of ack"
      end ;
      free name ev

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr_state init hdlrs None (FullNoHdr ()) args vs

let _ = 
  Param.default "stable_sweep" (Param.Time (Time.of_float 1.0)) ;
  Param.default "stable_explicit_ack" (Param.Bool false) ;
  Layer.install name (Layer.init l)

(**************************************************************)
