(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* FRAG.ML *)
(* Author: Mark Hayden, 6/95 *)
(**************************************************************)
open Trans
open Layer
open View
open Event
open Util
(**************************************************************)
let name = Trace.source_file "FRAG"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type frag = {
  mutable iov : Iovecl.t ;
  mutable i : int
}

(* The last fragment also has the protocol headers
 * for the layers above us.
 *)
type header = NoHdr 
  | Frag of seqno * seqno		(* ith of n fragments *)
  | Last of seqno			(* num of fragments *)

type state = {
  max_len	: int ;			(* maximum transmisstion size *)
  cast		: frag array ;		(* recv fragments for Cast's *)
  send		: frag array ;		(* recv fragments for Send's *)
  local         : bool array		(* which members are local to process *)
}

let dump (ls,vs) s =
  eprintf "FRAG:dump:%s\n" ls.name ;
  eprintf "  my_rank=%d, nmembers=%d\n" ls.rank ls.nmembers ;
  eprintf "  view =%s\n" (View.to_string vs.view) ;
  eprintf "  local=%s\n" (string_of_bool_array s.local)

let init () (ls,vs) = 
  let addr = ls.addr in
  let local = Array.map (Addr.same_process addr) vs.address in
  let log = Trace.log name ls.name in
  log (fun () -> sprintf "local=%s" (string_of_bool_array local)) ;
  { max_len = Param.int vs.params "frag_max_len" ;
    cast = array_createf ls.nmembers (fun _ -> {iov=[||];i=0}) ;
    send = array_createf ls.nmembers (fun _ -> {iov=[||];i=0}) ;
    local = local
  }

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let ack = make_acker name dnnm in
  let log = Trace.log name ls.name in

  let all_local ranks = 
    List.for_all (fun rank -> s.local.(rank)) ranks
  in

  (* Handle the fragments as they arrive.
   *)
  let handle_frag ev i n abv =
    (* First, flatten the iovec.  Normally, the iovec
     * should have only one element anyway.  
     *)
    let iov = Iovecl.flatten name (getIov ev) in
    
    (* Select the fragment info to use.
     *)
    let frag = 
      if getType ev = ECast
      then s.cast.((getOrigin ev))
      else s.send.((getOrigin ev))
    in

    (* Check for out-of-order fragments: could send up lost
     * message when they come out of order.  
     *)
    if frag.i <> i then (
      log (fun () -> sprintf "expect=%d/%d, got=%d\n" 
        frag.i (Array.length frag.iov) i) ;
      failwith "fragment arrived out-of-order" ;
    ) ;
    
    (* On first fragment, allocate the iovec array where
     * we will put the rest of the entries.
     *)
    if i = 0 then (
      if frag.iov <> [||] then failwith "sanity" ;
      frag.iov <- array_create name n (Iovec.empty name)
    ) ;
    if Array.length frag.iov <> n then
      failwith "bad frag array" ;
    
    (* Add fragment into array.
     *)
    if i >= Array.length frag.iov then
      failwith "frag index out of bounds" ;
    frag.iov.(i) <- iov ;
    frag.i <- succ i ;
    
    (* On last fragment, send it up.  Note that the ack on
     * this should ack all previous fragments.
     *)
    match abv with
    | None ->
	if i = pred n then failwith "sanity" ;
      	ack ev ; free name ev
    | Some abv ->
    	if i <> pred n then failwith "sanity" ;
	let iov = frag.iov in
      	frag.iov <- [||] ;
      	frag.i <- 0 ;
      	up (set name ev [Iov iov]) abv
  in

  let up_hdlr ev abv hdr = match getType ev, hdr with
  | (ECast | ESend), Last(n) ->
      handle_frag ev (pred n) n (Some abv)
  | _, NoHdr -> up ev abv
  | _, _     -> failwith "bad up event"

  and uplm_hdlr ev hdr = match getType ev, hdr with
  | (ECast | ESend), Frag(i,n) ->
      handle_frag ev i n None
  | _ -> failwith "unknown local message"
  and upnm_hdlr = upnm
  
  and dn_hdlr ev abv = match getType ev with
  | ECast | ESend ->
      (* Messages are not fragmented if either:
       * 1) They are small.
       * 2) They are tagged as unreliable. (BUG?)
       * 3) All their destinations are within this process.
       *)
      let iovl = getIov ev in
      let lenl = Iovecl.len name iovl in
      let ranks = getRanks ev in
      let typ = getType ev in
      if lenl <= s.max_len 
      || getUnreliable ev
      || (typ = ESend && all_local ranks)
      then (
      	dn ev abv NoHdr
      ) else (
(*
        log (fun () -> sprintf "fragmenting") ;
*)
	(* Fragment the message.
	 *)
	let frags = Iovecl.fragment name s.max_len (getIov ev) in
	let nfrags = Array.length frags in
	if nfrags < 2 then failwith "sanity" ;
	  
	for i = 0 to nfrags - 2 do
	  dnlm (set name ev [Iov frags.(i)]) (Frag(i,nfrags)) ;
	done ;

	(* The last fragment has the header above us.
         *)
	dn (set name ev [Iov frags.(pred nfrags)]) abv (Last(nfrags)) ;
	free name ev
      )
  | _ -> dn ev abv NoHdr

  and dnnm_hdlr = dnnm

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr init hdlrs None (FullNoHdr NoHdr) args vs

let _ = 
  let frag_len =(*min*) (Hsys.max_msg_len () - 2000) (*3000*) in
  Param.default "frag_max_len" (Param.Int frag_len) ;
  Layer.install name (Layer.init l)
    
(**************************************************************)
