(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* BOTTOM.ML : bottom protocol layer *)
(* Author: Mark Hayden, 4/95 *)
(* Based on code by: Robbert vanRenesse *)
(* With structural changes sugested by Scott Stoller *)
(**************************************************************)
open Layer
open Trans
open View
open Event
open Util
(**************************************************************)
let name = Trace.source_file "BOTTOM"
(**************************************************************)

type header = NoHdr
  | MergeRequest
  | MergeGranted
  | MergeDenied

(**************************************************************)

type state = {
  alarm			: Alarm.t ;
  alive			: bool array ;
  mutable enabled 	: bool
}

(**************************************************************)

let init () (ls,vs) = {
  alarm	  	= Alarm.get () ;	(* Hack! *)
  alive	  	= array_create name ls.nmembers true ;
  enabled	= true
}

let dump (ls,vs) s =
  eprintf "BOTTOM:dump:%s\n" ls.name ;
  eprintf "  enabled=%b\n" s.enabled

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let log = Trace.log name ls.name in

  let got_merge ev typ abv =
    if true || s.enabled then (
      let time = Alarm.gettime s.alarm in
      up (set name ev[Type typ;Time time]) abv
    ) else (
      free name ev
    )
  in

  let up_hdlr ev abv hdr = match getType ev, hdr with
  | (ECast|ESend), NoHdr ->
      let origin = getOrigin ev in
      if s.alive.(origin) then 
	up ev abv
      else
	free name ev

  | _, MergeRequest -> got_merge ev EMergeRequest abv
  | _, MergeGranted -> got_merge ev EMergeGranted abv
  | _, MergeDenied  -> got_merge ev EMergeDenied abv
  | _ -> failwith "bad up event"

  and uplm_hdlr _ _ = failwith "got uplm event"
  and upnm_hdlr ev = match getType ev with
  | EInit ->
      if s.enabled then
	upnm (set name ev [Time (Alarm.gettime s.alarm)])
      else 
	free name ev
  | ETimer | EAsync | EGossipExt -> 
      if s.enabled then upnm ev else free name ev
  | _ -> failwith "bad upnm event"

  and dn_hdlr ev abv = 
    if s.enabled then (
      match getType ev with
      | ECast         -> dn ev abv NoHdr
      | ESend         -> dn ev abv NoHdr
      | EMergeRequest -> dn ev abv MergeRequest
      | EMergeGranted -> dn ev abv MergeGranted
      | EMergeDenied  -> dn ev abv MergeDenied
      | _ -> failwith "bad down event[1]"
    ) else (
      free name ev
    )
	
  and dnnm_hdlr ev = match getType ev with
  | EGossipExt -> 
      dnnm ev

  (* ETimer: request a timeout callback from the transport.
   *)
  | ETimer ->
      let alarm = getAlarm ev in
      if alarm = Time.zero then
	upnm (create name ETimer[Time (Alarm.gettime s.alarm)])
      else
      	dnnm ev

  (* EFail: mark the members as failed and bounce the event.
   *)
  | EFail ->
      let failed = getFailures ev in
      List.iter (fun i ->	
	s.alive.(i) <- false
      ) failed ;
      upnm ev

  (* EExit: disable the transport and send up an EExit event.
   *)
  | EExit ->
      if s.enabled then (
	(* Mark myself and all members as disabled.
	 *)
	s.enabled <- false ;
	for i = 0 to pred ls.nmembers do
	  s.alive.(i) <- false
	done ;

	(* Pass event down.
	 *)
	dnnm ev ;
	let time = Alarm.gettime s.alarm in
	upnm (create name EExit[Time time]) ;
      ) else (
	log (fun () -> "2nd Exit (ok in some cases)") ;
	free name ev
      )

  (* All of these we just bounce up as-is.
   *)
  | ESuspect
  | EXferDone
  | ERekey
  | EPrompt
  | EProtocol
  | EMigrate
  | EView
  | EElect
  | EStable
  | EBlock
  | ELeave
  | EBlockOk
  | EAccount 
  | EInfo
  | EStableReq ->
      upnm ev
  | EDump -> 
      dump (ls,vs) s ;
      upnm ev

  | _ -> failwith "bad down event[2]"

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr_state init hdlrs None (FullNoHdr NoHdr) args vs

let _ = 
  Layer.install name (Layer.init l)

(**************************************************************)
