(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(****************************************************************)
(* SEQUENCER.ML                                                 *)
(* Author: Roy Friedman, 3/96                                   *)
(* Bug fixes: Mark Hayden, 10/96				*)
(* Based on the C implementation of dynseq but without rotating	*)
(* the sequencer.					        *)
(*                                                              *)
(* The coordinator of the group acts as a sequencer; in order   *)
(* to send messages, processes must send the messages pt2pt to  *)
(* the sequencer, who then broadcasts these messages to         *)
(* the rest of the group.                                       *)
(****************************************************************)
open Layer
open View
open Event
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "SEQUENCER"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
(* Headers

 * NoHdr: all non-totally ordered cast messages.

 * Ordered(rank): deliver this message as though from member
 * 'rank'.

 * Unordered: (ie., not ordered by a token holder) Sent at
 * the end of the view.  All members deliver unordered
 * messages in fifo order in order of the sending members
 * rank.
 *)

type header = NoHdr
  | ToSeq
  | Ordered of rank
  | Unordered of seqno

(**************************************************************)

type 'abv item =
  | Msg of 'abv * Iovecl.t
  | Unset				(* for iq *)
  | Reset				(* for iq *)

type ('abv,'b) state = {
  seq_rank		: int ;         (* who is the sequencer *)
  ordered		: int array ;
  mutable got_view	: bool ;	(* have we got an EView? *)
  mutable blocking 	: bool ;	(* are we blocking? *)

  (* buffered send messages (waiting for delivery from coordinator) *)
  casts 		: 'abv item Iq.t ;

  (* buffered recv messages waiting for the end of the view *)
  up_unord 		: (seqno * 'abv * Iovecl.t)  Queue.t array
}

(**************************************************************)

let dump (ls,vs) s =
  eprintf "SEQUENCER:dump:%s\n" ls.name ;
  eprintf " blocking=%b \n" s.blocking

(**************************************************************)

let init () (ls,vs) = {
  seq_rank      = 0 ;			(* always rank 0 *)
  ordered	= array_create name ls.nmembers 0 ;
  blocking 	= false ;
  casts 	= Iq.empty Unset Reset ;
  got_view	= false ;
  up_unord 	=
    Array.map (fun _ ->
      Queue.create ()
    ) vs.view
}

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let ack = make_acker name dnnm in

  (* Called when blocking is about to begin.  Send any
   * messages we have buffered as 'unordered.'
   *)
  let now_blocking () =
    if not s.blocking then (
      s.blocking <- true ;
      List.iter (function (seqno,Msg(abv,iov)) ->
	dn (castIov name iov) abv (Unordered seqno) ;
	Queue.add (seqno,abv,iov) s.up_unord.(ls.rank)
      | _ -> failwith "buffer sanity"
      ) (Iq.list_of_iq s.casts)
    )
  in

  let up_hdlr ev abv hdr = match getType ev, hdr with
    | ESend, ToSeq ->
	if (ls.rank <> s.seq_rank) then
	  failwith "send when not sequencer" ;

	(* Bounce the message back out and deliver locally. 
	 *)
	if not s.blocking then (
	  up (castOriginIov name (getOrigin ev) (getIov ev)) abv ;
	  if ls.nmembers > 1 then
	    dn (castIov name (getIov ev)) abv (Ordered (getOrigin ev)) ;
	  s.ordered.((getOrigin ev)) <- succ s.ordered.((getOrigin ev))
	) ;
	ack ev ; free name ev

    | ECast, Ordered(rank) ->
      	if s.got_view then
	  failwith "got ordered after view" ;
	  
      	(* If it's from me, then remove from my iq.
	 *)
	if rank = ls.rank then
	  Iq.advance_head s.casts s.ordered.(rank) ;

      	(* Deliver ordered messages.  Keep track of how many
	 * I've got from each member.
       	 *)
	s.ordered.(rank) <- succ s.ordered.(rank) ;
	up (set name ev[Origin rank]) abv

    | ECast, Unordered(seqno) ->
      	(* Buffer unordered messages until the view.
	 *)
	Queue.add (seqno,abv,(getIov ev)) s.up_unord.((getOrigin ev)) ;
	ack ev ; free name ev
    | _, NoHdr      -> up ev abv (* all other events have NoHdr *)
    | _             -> failwith "non-NoHdr on non ECast"

  and uplm_hdlr ev hdr = failwith "local message event"

  and upnm_hdlr ev = match getType ev with
  | EView ->
      s.got_view <- true ;
      
      (* Deliver unordered messages and then the view.
       *)
      for i = 0 to pred ls.nmembers do
	queue_clean (fun (seqno,abv,iov) ->
	  if seqno >= s.ordered.(i) then
	    up (castOriginIov name i iov) abv
	) s.up_unord.(i)
      done ;
      upnm ev
  | _ -> upnm ev

  and dn_hdlr ev abv = match getType ev with
  | ECast ->
      if getNoTotal ev then (
      	(* Send without ordering properties.
	 *)
      	dn ev abv NoHdr
      ) else if s.blocking then (
      	eprintf "SEQUENCER:warning dropping ECast after EBlockOk\n" ;
	free name ev
      ) else if ls.rank = s.seq_rank then (
	(* If I'm the sequencer, send out immediately as an ordered message.
	 * Otherwise, send pt2pt to sequencer and stash a copy of it in
	 * case the sequencer fails and we need to resend it.
	 *)
	if ls.nmembers > 1 then
	  dn ev abv (Ordered(ls.rank)) ;
	up (castOriginIov name ls.rank (getIov ev)) abv ;
	s.ordered.(ls.rank) <- succ s.ordered.(ls.rank)
      ) else (
	Iq.add s.casts (Msg(abv,(getIov ev))) ;
	dn (sendRanksIov name [s.seq_rank] (getIov ev)) abv ToSeq ;
	free name ev
      )

    (* Handle local delivery for sends.
     *)
  | ESend ->
      if List.mem ls.rank (getRanks ev) then (
      	up (create name ESend[Origin ls.rank; Iov (getIov ev)]) abv ;
	if List.length (getRanks ev) = 1 then
  	  free name ev
	else
	  dn (set name ev[Ranks (except ls.rank (getRanks ev))]) abv NoHdr
      ) else (
	dn ev abv NoHdr
      )

  | _ -> dn ev abv NoHdr

  and dnnm_hdlr ev = match getType ev with
  | EBlockOk ->
      now_blocking () ;
      dnnm ev
  | EAck ->
      free name ev
  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

(**************************************************************)
