(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PR_SUSPECT.ML *)
(* Author: Mark Hayden, 11/96 *)
(**************************************************************)
open Trans
open Layer
open Event
open Util
open View
open Arrayop
(**************************************************************)
let name = Trace.source_file "PR_SUSPECT"
(**************************************************************)

type header = NoHdr 
  | Gossip of bool list * int list

type state = {
  sweep		: Time.t ;
  max_idle	: int ;
  fanout        : int ;
  mutable gossip : int array ;
  failed        : bool array ;
  idle		: int array ;
  mutable gossiped : bool ;
  mutable next_gossip : Time.t
}

(**************************************************************)

let dump (ls,vs) s =
  eprintf "PR_SUSPECT:dump:%s\n" ls.name ;
  eprintf "  nmembers=%d, rank=%d\n" ls.nmembers ls.rank ;
  eprintf "  failed   =%s\n" (string_of_bool_array s.failed) ;
  eprintf "  idle     =%s\n" (string_of_int_array s.idle)

(**************************************************************)

let init () (ls,vs) = 
  let failed = array_create name ls.nmembers false in {
  max_idle    = Param.int vs.params "pr_suspect_max_idle" ;
  sweep       = Param.time vs.params "pr_suspect_sweep" ;
  fanout      = Param.int vs.params "pr_suspect_fanout" ;

  failed      = failed ;
  idle 	      = array_create name ls.nmembers 0 ;
  gossiped    = true ;
  gossip      = gossip failed ls.rank ;
  next_gossip = Time.invalid
}

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in

  let do_gossip () =
    let dests = choose s.gossip s.fanout in
    let failed = Array.to_list s.failed in
    let idle = Array.to_list s.idle in
    dnlm (create name ESend[
      Ranks dests ;
      Unreliable
    ]) (Gossip(failed,idle))
  in

  let up_hdlr ev abv hdr = match getType ev,hdr with
  | _, NoHdr -> up ev abv
  | _, _     -> failwith "bad up event"

  and uplm_hdlr ev hdr = match getType ev,hdr with
    (* Gossip Message.
     *)
  | (ECast|ESend), Gossip(failed,idle) ->
      let origin = (getOrigin ev) in
      if (not (List.nth failed ls.rank)) (* BUG: could auto-fail him *)
      && (not s.failed.(origin))
      then (
	let idle = Array.of_list idle in
	minify s.idle idle ;
	if s.idle.(ls.rank) <> 0 then
	  failwith "sanity" ;
	if not s.gossiped then (
	  s.gossiped <- true ;
	  do_gossip ()
	)
      ) ;
      free name ev

  | _ -> failwith "unknown local message"


  and upnm_hdlr ev = match getType ev with

    (* EFail: mark the failed members and check if any
     * messages are now suspect.
     *)
  | EFail ->
      let failed = getFailures ev in
      List.iter (fun rank ->
      	s.failed.(rank) <- true
      ) failed ;
      s.gossip <- gossip s.failed ls.rank ;
      upnm ev

    (* EInit: request a first timer alarm to bootstrap things.
     *)
  | EInit ->
      dnnm (timerAlarm name Time.zero) ;
      upnm ev

    (* ETimer: every so often:
     *   1. check for suspicions
     *   2. ping the other members
     *)
  | ETimer ->
      let time = getTime ev in
      if time >= s.next_gossip then (
	s.next_gossip <- Time.add time s.sweep ;
	dnnm (timerAlarm name s.next_gossip) ; (* request next gossip *)

	(* Gossip, if haven't done so already.
	 *)
	if not s.gossiped then (
	  s.gossiped <- true ;
	  do_gossip ()
	) ;
	s.gossiped <- false ;

	(* Increment idle counters and check threshhold.
	 *)
	let suspicions = ref [] in
	for i = 0 to pred ls.nmembers do
	  if i <> ls.rank 
	  && not s.failed.(i) 
	  then (
	    s.idle.(i) <- succ s.idle.(i) ;
	    if s.idle.(i) > s.max_idle then
	      suspicions := i :: !suspicions
	  )
	done ;

	(* Announce suspicions.
	 *)
	if !suspicions <> [] then
	  dnnm (suspectReason name !suspicions "PR_SUSPECT:timeout") ;

	(* If rank 0, start the gossip cascade.
	 *)
	if ls.rank = 0 then (
	  s.gossiped <- true ;
	  do_gossip ()
        )
      ) ;
      upnm ev

  | EDump -> dump (ls,vs) s ; upnm ev
  | _ -> upnm ev

  and dn_hdlr ev abv = match getType ev with
  | _ -> dn ev abv NoHdr

  and dnnm_hdlr = dnnm

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ =
  Param.default "pr_suspect_sweep" (Param.Time (Time.of_float 0.5)) ;
  Param.default "pr_suspect_max_idle" (Param.Int 15) ;
  Param.default "pr_suspect_fanout" (Param.Int 3) ;
  Layer.install name (Layer.init l)

(**************************************************************)
