(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* TOP_APPL.ML *)
(* Author: Mark Hayden, 8/95 *)
(**************************************************************)
open Layer
open Util
open View
open Event
open Appl_intf
(**************************************************************)
let name = Trace.source_file "TOP_APPL"
(**************************************************************)

type header = NoHdr
  | Merge
  | View

type trans =
  | Merging
  | Normal
  | Blocked

type delay =
  | DCast of rank * Iovecl.t
  | DSend of rank * Iovecl.t

let string_of_trans = function
  | Merging -> "Merging"
  | Normal  -> "Normal"
  | Blocked -> "Blocked"

type state = {
  intf			: Appl_intf.t ;
  states                : Iovecl.t option array ;
  mutable state 	: trans ;
  mutable leaving	: bool ;
  mutable next_sweep	: Time.t ;
  mutable up_block      : bool ;
  delay                 : delay Queue.t
} 

(**************************************************************)

let dump (ls,vs) s =
  eprintf "TOP_APPL:dump:%s: state=%s\n" ls.name (string_of_trans s.state) ;
  eprintf "  size(delay)=%d\n" (Queue.length s.delay)

(**************************************************************)

let init intf (ls,vs) = {
  intf		= intf ;
  states        = array_create name ls.nmembers None ;
  state         = Merging ;
  leaving 	= false ;
  next_sweep	= Time.zero ;	
  up_block      = false ;
  delay         = Queue.create ()
}

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let log = Trace.log name ls.name in
  let ack = make_acker name dnnm in

  let marsh,unmarsh = Iovecl.make_marsh name in
  
  (* Block this member, if not blocked already.
   *)
  let rec block () =
    begin
      match s.state with
      | Normal ->
	  let actions = s.intf.block () in
	  handle_actions actions ;

	  (* If Blocked(false) is one of the actions, do
	   * not actually block.
	   *)
	  let rec loop = function
	    | [] -> s.state <- Blocked
	    | Block false :: _ -> ()
	    | hd::tl -> loop tl
	  in loop actions
      | _ -> s.state <- Blocked
    end ;

    (* If we're blocked and an EBlock was intercepted
     * then generate another EBlock event.
     *)
    if s.state = Blocked && s.up_block then (
      s.up_block <- false ;
      upnm (create name EBlock[])
    )

  and handle_actions actions =
    if s.state <> Normal then 
      failwith "handle_actions:not in Normal state" ;

    List.iter (function
      | Cast(msg) ->
	  let ev = castIov name msg in
	  let ev = set name ev [ApplMsg] in
      	  dnlm ev NoHdr
      | Send(dests,msg) ->
	  List.iter (fun dest ->
	    if dest = ls.rank then
	      log (fun () -> sprintf "send to myself dests=%s" (string_of_int_list dests)) ;
	    if (* dest = ls.rank || *) dest < 0 || dest > ls.nmembers then (
	      eprintf "TOP_APPL:dest=%d,rank=%d,nmembers=%d\n" dest ls.rank ls.nmembers ;
	      failwith "bad destination" ;
	    )
	  ) dests ;
      	  dnlm (sendRanksIov name dests msg) NoHdr
      | Leave ->
	  s.leaving <- true ;
	  s.state <- Blocked ;
	  dnnm (create name ELeave[])
      | XferDone           -> dnnm (create name EXferDone[])
      | Protocol(proto_id) -> dnnm (create name EProtocol[ProtoId proto_id])
      | Migrate(addrs)     -> dnnm (create name EMigrate[Address addrs])
      | Rekey              -> dnnm (create name ERekey[])
      | Dump               -> dnnm (create name EDump[])
      |	Prompt             -> dnnm (create name EPrompt[])
      |	Suspect ranks ->
	  if List.mem ls.rank ranks then
	    failwith "I'm listed as a suspect" ;
	  dnnm (create name ESuspect[Suspects ranks])
      | Timeout t ->
      	  failwith "Timeout not supported"
      |	Block b -> 
	  (* Blocked(false) is handled in the block() case.
	   *)
	  if b then (
	    s.state <- Blocked ;
	    s.up_block <- false ;
	    upnm (create name EBlock[])
	  )
    ) actions ;
  in

  (* Check if coordinator is ready to send out view.
   *)
  let check_ready new_states =
    if ls.rank <> 0 then failwith "check_ready:non-coord" ;

    if s.state = Merging then (
      List.iter (fun (rank,state) ->
      	s.states.(rank) <- Some state
      ) new_states ;

      if not (array_mem None s.states) then (
      	s.state <- Normal ;
	
	(* Put all of the state info into a list.
	 *)
      	let states = Array.map some_of s.states in
	let states = Array.to_list states in
      	let view_msg = s.intf.block_install_view (ls,vs) states in
	log (fun () -> sprintf "got all states:view_size=%d" (Iovecl.len name view_msg)) ;
      	let actions = s.intf.unblock_view (ls,vs) view_msg in

      	(* Send message first, then handle actions.
	 *)
      	dnlm (castIov name view_msg) View;
      	handle_actions actions ;

	(* Clear out the states info.
	 *)
	for i = 0 to pred ls.nmembers do
	  s.states.(i) <- None
	done ;
      )
    )
  in      

  let up_hdlr ev abv hdr = match getType ev,hdr with
  | _,NoHdr -> up ev abv
  | _ -> failwith "bad up event"

  and uplm_hdlr ev hdr = match getType ev,hdr with
  | ECast, NoHdr ->
      let iov = getIov ev in
      let origin = getOrigin ev in
      begin match s.state with
      | Merging ->
	  Queue.add (DCast(origin,iov)) s.delay 
      | Normal ->
	  handle_actions (s.intf.recv_cast origin iov)
      | Blocked ->
      	  s.intf.block_recv_cast origin iov
      end ;
      ack ev ; free name ev

  | ESend, NoHdr ->
      let iov = getIov ev in
      let origin = getOrigin ev in
      begin match s.state with
      | Merging ->
	  Queue.add (DSend(origin,iov)) s.delay 
      | Normal ->
	  handle_actions (s.intf.recv_send origin iov)
      | Blocked ->
	  s.intf.block_recv_send origin iov
      end ;
      ack ev ; free name ev

  | ESend, Merge ->
      if getOrigin ev = 0 then failwith "Merge from coordinator" ;
      if ls.rank <> 0 then failwith "non-coord got Merge" ;

      let iov = getIov ev in
      let states = unmarsh iov in
      let states =
      	List.map (fun (rank,state) ->
	  let rbuf = Iovec.heap name state in
	  let state = Iovec.alloc name rbuf 0 (String.length state) in
	  let state = [| state |] in
	  (rank,state)
	) states
      in
      log (fun () -> sprintf "Merge:info from %s" 
        (string_of_int_list (List.map fst states))) ;
      check_ready states ;

      ack ev ; free name ev

  | ECast, View ->
      let origin = getOrigin ev in
      if origin <> 0 then failwith "got View from non-coord" ;
      if origin <> ls.rank then (
	if s.state = Normal then 
	  failwith "view when state is Normal" ;

	if s.state = Merging then (
	  s.state <- Normal ;
	  let view_msg = getIov ev in
	  handle_actions (s.intf.unblock_view (ls,vs) view_msg) ;
	  queue_clean (function
	    | DCast(origin,iov) ->
	      	handle_actions (s.intf.recv_cast origin iov)
	    | DSend(origin,iov) ->
	      	handle_actions (s.intf.recv_send origin iov)
	  ) s.delay
	)
      ) ;
      ack ev ; free name ev
      
  | _,_ -> failwith "got bad uplm event"

  and upnm_hdlr ev = match getType ev with
  | EInit ->
      upnm ev ;
      
      (* Get the merge message from this member.
       *)
      let states = s.intf.block_view (ls,vs) in

      (* If coord then stash the states, and check if ready.
       *)
      if ls.rank = 0 then (
      	check_ready states ;
      ) else if states <> [] then (
	(* If not coord and got some states, then send them
	 * to coord.  
	 *)
	let states = 
	  List.map (fun (rank,iov) -> 
	    let iov = Iovecl.flatten name iov in
	    let iov = Iovec.to_string name iov in
	    (rank,iov)
	  ) states
	in
	let iovl = marsh states in	(* Hack! *)
	dnlm (sendRanksIov name [0] iovl) Merge ;
      ) ;

      (* Request first alarm.
       *)
      dnnm (timerAlarm name Time.zero) ;

  | EAsync ->
      (* If not blocked, wake up the application.
       *)
      if s.state = Normal then (
	handle_actions (s.intf.heartbeat (getTime ev))
      ) ;
      upnm ev

  | EBlock -> 
      if s.up_block then
	failwith "2nd EBlock" ;
      s.up_block <- true ;
      block ()

  | ETimer ->
      let time = getTime ev in
      if time >= s.next_sweep then (
      	(* Schedule a new heartbeat.
      	 *)
	s.next_sweep <- Time.add time s.intf.heartbeat_rate ;
	dnnm (timerAlarm name s.next_sweep) ;

      	(* If not blocked, wake up the application.
	 *)
	if s.state = Normal then (
	  handle_actions (s.intf.heartbeat time)
	)
      ) ;

      upnm ev

  | EExit ->
      if s.leaving then (
	block () ;
	s.intf.exit ()
      ) ;

      (* GC the delay queue.
       *)
      queue_clean (function 
	| DCast(_,iov) -> Iovecl.free name iov
	| DSend(_,iov) -> Iovecl.free name iov
      ) s.delay ;
      
      upnm ev

    (* Transis.
     *)
  | EAlive -> free name ev
 
    (* Transis - Here, we can deliver the transitional configuration. 
     *)
  | ETransView ->  
      let t = getTransitional ev in 
      log (fun () -> sprintf "Trans view=%s\n" (string_of_bool_array t));
      upnm ev

  | EDump -> dump (ls,vs) s ; upnm ev
  | _ -> upnm ev

  and dn_hdlr ev abv = match getType ev with
    (* Hack!!! Add NoTotal flag to all messages from above.
     *)
  | ECast | ESend ->
      let ev = set name ev [NoTotal] in
      dn ev abv NoHdr
  | _ -> dn ev abv NoHdr

  and dnnm_hdlr = dnnm

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr_state init hdlrs None (LocalNoHdr NoHdr) args vs

let _ = Layer.install name (Layer.initTop_appl l)

(**************************************************************)
