(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PRIMARY.ML *)
(* Author: Mark Hayden, Zhen Xiao, 5/97 *)
(**************************************************************)
open Layer
open View
open Event
open Util
(**************************************************************)
let name = Trace.source_file "PRIMARY"
(**************************************************************)

type header = InView

type state = {
  in_view : bool array ;
  majority : Endpt.id array -> bool
}

let dump (ls,vs) s =
  eprintf "XFER:dump, in_view=%s\n" (string_of_bool_array s.in_view)

let init () (ls,vs) = {
  in_view = array_create name ls.nmembers false ;
  majority = (fun a -> Array.length a > 3)
}

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let ack = make_acker name dnnm in
  let log = Trace.log name ls.name in

  let up_hdlr ev abv () = up ev abv

  and uplm_hdlr ev hdr = match getType ev,hdr with

    (* ECast|ESend(InView): other member is telling me
     * that he is in the view.
     *)
  | (ECast|ESend), InView ->
      let origin = getOrigin ev in
      s.in_view.(origin) <- true ;
      ack ev ; free name ev

  | _ -> failwith "bad uplm event"

  and upnm_hdlr ev = match getType ev with

    (* Send a message to the coordinator saying that I'm
     * here.  
     *)
  | EInit ->
      if ls.am_coord then
	s.in_view.(ls.rank) <- true
      else
	dnlm (sendRank name vs.coord) InView ;
      upnm ev ;

      (* If we have a majority, but we are not marked as
       * primary, then prompt a view change.  
       *)
      if ls.am_coord
      && s.majority vs.view
      && not vs.primary 
      then
	dnnm (create name EPrompt[])

    (* At all members, check that the ltime is one more
     * than the current logical time.
     *)
  | EView ->
      let next_vs = getViewState ev in
      let next_ltime = fst next_vs.view_id in
      let this_ltime = fst vs.view_id in

      let primary = 
	next_vs.primary &&
	next_ltime = succ this_ltime
      in

      let next_vs = View.set next_vs [Vs_primary primary] in
      upnm (set name ev [ViewState next_vs])

  | EDump -> dump (ls,vs) s ; upnm ev
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv ()

  and dnnm_hdlr ev = match getType ev with

    (* EView: decide whether next view is 
     * primary.
     *)
  | EView ->
      let next_vs = getViewState ev in
      let this_coord = vs.view.(vs.coord) in
      let next_coord = next_vs.view.(next_vs.coord) in

      let in_view = array_combine s.in_view vs.view in
      let in_view = 
	Array.map (fun (in_view,endpt) ->
	  if in_view then Some(endpt) else None
	) in_view
      in
      let in_view = array_filter_nones in_view in
      let in_view_majority = s.majority in_view in

      let next_majority = s.majority next_vs.view in

      let primary =
	in_view_majority &&
	next_majority &&
	next_coord = this_coord
      in

      log (fun () -> sprintf "in_view_maj=%b, next_maj=%b, same_coord=%b"
	in_view_majority
	next_majority
	(next_coord = this_coord)
      ) ;

      let next_vs = View.set next_vs [Vs_primary primary] in
      dnnm (set name ev [ViewState next_vs])

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

