(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* CLTSVR.ML *)
(* Author: Mark Hayden, 6/95 *)
(* Based on code by Robbert vanRenesse *)
(* Notes:

 * 1. Monitors who are the clients and who are the servers.
      This info is passed up in the View.state record.

 * 2. Supports subcasts to just the servers or just the
      clients.
*)
(**************************************************************)
open Layer
open View
open Event
open Util
(**************************************************************)
let name = Trace.source_file "CLTSVR"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type header = NoHdr 
| ClientOnly 
| ServerOnly
| Merge of (Endpt.id * bool) array

type state = {
  client : bool ;
  mutable mergers : (Endpt.id * bool) array
}
  
let init () (ls,vs) = {
  client = vs.clients.(ls.rank) ;
  mergers = [||]
}

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let ack = make_acker name dnnm in

  let up_hdlr ev abv hdr = match getType ev, hdr with
    (* Save the merger info until the view.
     *)
  | EMergeRequest, Merge(mergers) ->
      s.mergers <- Array.append s.mergers mergers ;
      up ev abv

    (* Filter out messages not to us.
     *)
  | ECast, ClientOnly when s.client -> 
      up ev abv
  | ECast, ServerOnly when not s.client -> 
      up ev abv
  | ECast, NoHdr -> 
      up ev abv
  | ECast, _ ->
      ack ev ;
      free name ev
  | _, NoHdr -> up ev abv
  | _ -> failwith "bad up event"

  and uplm_hdlr ev hdr = failwith "unknown local message"
  and upnm_hdlr ev = match getType ev with
  | EView ->
      (* First, do a sanity check.
       *)
      let vs = getViewState ev in
      let rank = array_index ls.endpt vs.view in
      let clients = vs.clients in
      let client = clients.(rank) in
      if client <> s.client then
        failwith "sanity:client info inconsistent" ;

      upnm ev
  | _ -> upnm ev
  
  and dn_hdlr ev abv = match getType ev with

    (* Check bitfields to see if this is a client
     * or server subcast.
     *)
  | ECast ->
      let clt_only = getClientOnly ev
      and svr_only = getServerOnly ev in
      let hdr = match clt_only, svr_only with
      | true,  true  -> failwith "invalid options"
      | true,  false -> ClientOnly
      | false, true  -> ServerOnly
      | _,_	     -> NoHdr
      in
      dn ev abv hdr

    (* EMergeRequest: also send along my client info as assoc list.
     *)
  | EMergeRequest ->
      let cl = vs.clients in		(* get current clients *)
      let cl = array_combine vs.view cl in (* mark w/endpts *)
      dn ev abv (Merge cl)

  | _ -> dn ev abv NoHdr

  and dnnm_hdlr ev = match getType ev with
    (* Figure out the new clients vector for new view.
     *)
  | EView ->
      let cl = vs.clients in		(* get current clients *)
      let cl = array_combine vs.view cl in (* mark w/endpts *)
      let cl = Array.append cl s.mergers in (* add mergers *)
      let cl = Array.to_list cl in

      let new_vs = getViewState ev in	(* find new view *)
      let cl = Array.map (fun e ->	(* get client status *)
      	List.assoc e cl) new_vs.view	(* BUG: perf *)
      in

      (* Then add the info to the event.
       *)
      let vs = View.set vs [Vs_clients cl] in
      let ev = set name ev [ViewState vs] in
      dnnm ev
  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)
    
(**************************************************************)
