(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* HEAL.ML *)
(* Author: Mark Hayden, 6/95 *)
(* Based on code by Robbert vanRenesse & Dalia Malki *)
(* Originally named merge.ml *)
(**************************************************************)
(* Notes:

 * Only merge with higher numbered views to prevent cycles.

 * Only communicate when I think I will be able to merge
 * quickly.

 * Tries to prevent merge attempts that may fail.  Be
 * conservative.

 *)
(**************************************************************)
open Layer
open Util
open View
open Event
(**************************************************************)
let name = Trace.source_file "HEAL"
let failwith = make_failwith name
(**************************************************************)

type header = NoHdr

type state = {
  mutable wait_view  : bool ;
  mutable stable     : bool ;
  mutable merge_vid  : View.id ;
  mutable merge_con  : Endpt.full ;
  mutable bcasted    : bool
}

(**************************************************************)

let dump (ls,vs) s =
  eprintf "HEAL:dump:%s\n" ls.name ;
  eprintf "  stable=%b wait_view=%b\n" 
    s.stable s.wait_view ;
  eprintf "  view_id=%s con_view_id=%s\n"
    (View.string_of_id vs.view_id)
    (View.string_of_id s.merge_vid)

(**************************************************************)

let init () (ls,vs) =
  let wait_stable = Param.bool vs.params "heal_wait_stable" in {
  wait_view  = false ;
  stable     = (not wait_stable) || (ls.nmembers = 1) ;	(* singletons start stable *)
  merge_vid  = vs.view_id ;
  merge_con  = (vs.view.(0) , vs.address.(0)) ;(*BUG?*)
  bcasted    = false
}

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let log = Trace.log name ls.name in
  let my_con = (ls.endpt,ls.addr) in
  let gossip_send () =
    let msg = HealGos(vs.proto_id,vs.view_id,my_con,vs.view) in
    msg
  in

  let gossip_recv (bc_proto_id,bc_view_id,bc_con,bc_view) =
    let bc_coord = fst bc_con in

(* Removed: because zapped cases that were legal.
    (* If I hear of some of my members then zap them.
     *)
    if ls.am_coord
    && (not s.wait_view)
    && snd bc_view_id <> snd vs.view_id
    && bc_view_id > vs.view_id 
    && intersect bc_view (except ls.endpt vs.view) <> []
    then (
      let suspects = intersect bc_view vs.view in
      let suspects = except ls.endpt vs.view in
      let suspects = List.map (fun e -> index e vs.view) suspects in
      dnnm (suspectReason name suspects name) ;
    ) ;
*)

    (* Check if I'm interested in the other partition at all.
     *)
    let view = Array.to_list vs.view in
    let bc_view = Array.to_list bc_view in
    if bc_proto_id = vs.proto_id	(* Protocols are the same *)
    && ls.am_coord			(* I'm elected *)
    && s.stable				(* my view is stable *)
    && (not (array_mem bc_coord vs.view)) (* He is not in my view *)
    && (Lset.intersect view bc_view) = [] (* In fact, no members are shared *)
    then (
      log (fun () -> sprintf "recd interesting bcast from %s (curr contact=%s)"
	(Endpt.string_of_id bc_coord) 
	(Endpt.string_of_id (fst s.merge_con))) ;

      (* First check to see if I want to merge with him.
       *)
      if bc_view_id > s.merge_vid	(* his view_id is older *)
      then (
	log (fun () -> sprintf "suggesting merge with %s" (View.string_of_id bc_view_id)) ;
	s.merge_vid <- bc_view_id ;
	s.merge_con <- bc_con ;

	(* If I'm not installing a view currently, start a
	 * view change.
	 *)
	if not s.wait_view then (
	  s.wait_view <- true ;
	  dnnm (create name EPrompt [])
	)
      ) ;

      (* Otherwise, he might think my partition is interesting.
       *)
      if vs.view_id = s.merge_vid	(* I'm not planning to merge myself *)
      && not s.wait_view		(* I'm not waiting for a view *)
      && not s.bcasted			(* don't bcast all the time *)
      && vs.view_id > bc_view_id	(* check that I am older than this one *)
      then (
	s.bcasted <- true ;
	log (fun () -> "bcasting(recv)") ;
	dnnm (create name EGossipExt[(gossip_send ())])
      )
    )
  in

  let up_hdlr ev abv hdr = match getType ev with
  | _ -> up ev abv

  and uplm_hdlr ev hdr = failwith "got local message"

  and upnm_hdlr ev = match getType ev with
  | EGossipExt ->
      getExtendOpt ev (function
	| HealGos(a,b,c,d) -> gossip_recv (a,b,c,d) ; true
	| _ -> false) ;
      upnm ev

  | EBlock ->
      s.wait_view <- true ;
      upnm ev

  | EStable ->
      (* BUG: assumes another protocol will send a cast
       * at some point. *)
      if not s.stable then (
	let stable = getStability ev in
	for i = 0 to pred ls.nmembers do
          if stable.(i) > 0 then
	    s.stable <- true
	done
      ) ;
      upnm ev

  | EBlockOk ->
      if s.merge_con = my_con then
	upnm ev
      else (
	log (fun () -> sprintf "EBlock(suggest->%s)" (Endpt.string_of_id (fst s.merge_con))) ;
	upnm (set name ev [
	  Contact(s.merge_con,Some s.merge_vid)
	])
      )

  | EDump -> ( dump (ls,vs) s ; upnm ev )
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv NoHdr

  and dnnm_hdlr ev = match getType ev with
  | EGossipExt -> (
      s.bcasted <- false ;
      let ev =
	if ls.am_coord			(* I'm elected *)
	&& s.stable			(* my view is stable *)
	&& not s.wait_view		(* I'm not waiting for view *)
	&& vs.view_id = s.merge_vid	(* I'm not planning to merge *)
	then
	  set name ev [(gossip_send ())]
	else ev
      in
      dnnm ev
    )
  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = 
  Param.default "heal_wait_stable" (Param.Bool true) ;
  Layer.install name (Layer.init l)

(**************************************************************)
