(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* EXCHANGE.ML : key exchange protocol *)
(* Authors: Mark Hayden, Ohad Rodeh, 8/96 *)
(* With suggestions by Ron Minsky *)
(**************************************************************)
(* Improvments
 * -----------
 * (1) Include signing of messages.
*)

open Layer
open View
open Event
open Util
(**************************************************************)
let name = Trace.source_file "EXCHANGE"
let failwith = make_failwith name
(**************************************************************)

type exch_msg = 
  | Id of Endpt.id * Addr.set
  | Ticket of Endpt.id * Auth.ticket

type state = {
  myhkey                : Digest.t ;
  mutable myname        : Addr.set ;
  mutable know          : Endpt.id list ;
  mutable key_sug 	: Security.key ;
  mutable blocked	: bool
}

let dump (ls,vs) s = ()

let init () (ls,vs) = {
  myhkey  = Digest.string (Security.str_of_key vs.key);
  myname  = View.addr (ls,vs) ;
  know    = [];
  key_sug = vs.key;
  blocked = false
}

(**************************************************************)

let marshal, unmarshal = Util.make_marsh "EXCHANGE" true

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm}  =
  let log = Trace.log name ls.name in
  log (fun () -> sprintf "my-key=%s" (Security.string_of_key vs.key)) ;

  (* Need to add signature of the message via Digest+encryption via
   * s.key_sug. *)
  let gossip_send msg =
    let msg = marshal msg in
    let msg = ExchangeGos(msg) in
    msg
    
  (* Need to add verification of the message via Digest+encryption via
   * s.key_sug. *)
  and gossip_get a = 	
    let src = unmarshal a 0 (String.length a) in 
    src
  in
    
  let gossip_recv msg = 
    let msg = gossip_get msg in
    if not s.blocked then (
      match msg with 
      | Id(sendpt, saddr) -> (
	  if sendpt > ls.endpt then (
	    log (fun () -> sprintf "recvd id") ;
	    match Auth.ticket s.myname saddr (Security.str_of_key s.key_sug) with
	    | Some ticket ->
		log (fun () -> sprintf "sending ticket") ;
		let msg = gossip_send(Ticket(ls.endpt,ticket)) in 
		dnnm (create name EGossipExt[msg])
	    | None ->
		log (fun () -> sprintf "failed to authenticate id")
	  ) else if sendpt < ls.endpt then (
	    log (fun () -> sprintf "discarding id") ;
	  )
	)
      | Ticket(sendpt,ticket) -> (
	  if sendpt <> ls.endpt then (
	    log (fun () -> sprintf "recvd ticket") ;
	    let clear = Auth.check s.myname ticket in
	    match clear with
	    | Some key -> 
		let sec_key = Security.Common(key) in 
		if sec_key <> s.key_sug then (
		  log (fun () -> "accepting key") ;
		  s.key_sug <- sec_key ;
		  dnnm (create name EPrompt [])
		)
	    | None ->
		log (fun () -> sprintf "discarding ticket")
	  )
    	)
    ) else ( log (fun () -> sprintf "blocked") )
  in

  let up_hdlr ev abv () = up ev abv
  and uplm_hdlr ev hdr = failwith "bad uplm event"
  and upnm_hdlr ev = match getType ev with
  | EView ->
      s.blocked <- true ;
      upnm ev
  | EBlock ->
      s.blocked <- true ;
      upnm ev
  | EGossipExt -> 
      if ls.am_coord then (
    	 getExtendOpt ev (function
	   | ExchangeGos(a) -> gossip_recv a ; true
	   | _ -> false
	 )
      ) ;
      upnm ev 

  | EDump -> ( dump (ls,vs) s ; upnm ev )
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv ()

  and dnnm_hdlr ev = match getType ev with
  | EView ->
      let vs = getViewState ev in
      let vs = View.set vs [Vs_key s.key_sug] in
      dnnm (set name ev[ViewState vs])
  | EGossipExt ->
      let ev =
	if ls.am_coord && not s.blocked then (
	  let msg = gossip_send(Id(ls.endpt,s.myname)) in
	  set name ev [msg]
	) else ev
      in
      dnnm ev

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

(**************************************************************)
