(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* MFLOW.ML : MFLOW window-based flow control. *)
(* Authors: Zhen Xiao, Mark Hayden, 2/97 *)
(* Note that the window-cost of a message here is 1 more
 * than the number of bytes in the iov field of the message. *)
(**************************************************************)
open Layer
open Event
open Util
open View
(**************************************************************)
let name = Trace.source_file "MFLOW"
(**************************************************************)

module Queue = Queuee

type header = NoHdr
  | Ack of int
  | Data

(**************************************************************)

type 'abv state = 
  { window : int ;
    ack_thresh : int ;
    overhead : int ;
    send_buf : (Event.dn * 'abv) Queue.t ;
    credit : Mcredit.t

    (* output file handler *)

(*
    ; file_handler : out_channel
*)
  } 

(**************************************************************)

let msg_len s ev = Iovecl.len name (getIov ev) + s.overhead

let string_of_queue_len q = string_of_int (Queue.length q)
let string_of_queue_bytes s q = 
  let len = ref 0 in
  Queue.iter (fun (ev,_) -> len := !len + msg_len s ev) q ;
  string_of_int !len

let dump (ls,vs) s =
  eprintf "MFLOW:dump:%s\n" ls.name ;
  eprintf "  send_buf=%s\n" (string_of_queue_len s.send_buf) (*;
  eprintf "%s\n" (Mcredit.to_string_list s.credit)*)

(**************************************************************)

let init_credit sdr dst nmembers ack_thresh window =
  if nmembers = 1 then
    (0, window)			(* Avoid div-by-0 *)
  else (
    let sdr = if sdr > dst then pred sdr else sdr in
    let amt = - ((sdr * ack_thresh) / (pred nmembers)) in
    (amt,window-amt)
  )

let init () (ls,vs) = 
  let window = Param.int vs.params "mflow_window" in
  let ack_thresh = Param.int vs.params "mflow_ack_thresh" in
  let send_credit = 
    array_createf ls.nmembers (fun dst -> 
      snd (init_credit dst ls.rank ls.nmembers ack_thresh window)
    ) 
  in
  let recv_credit = 
    array_createf ls.nmembers (fun dst -> 
      fst (init_credit ls.rank dst ls.nmembers ack_thresh window)
    ) 
  in
(*
  let fname = 
    eprintf "mflow: Staggered\n" ;
    Param.string vs.params "fname" 
  in
*)
  { window = window ;
    ack_thresh = ack_thresh ;
    send_buf = Queue.create () ;
    overhead = Param.int vs.params "mflow_overhead" ;
    credit = Mcredit.create ls.rank ls.nmembers ack_thresh send_credit recv_credit

(*
    ; file_handler = open_out (fname)
*)
  }

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let ack = make_acker name dnnm in
  let log = Trace.log name ls.name in
  let logl = Trace.logl name ls.name in
  let logb = Trace.log "BUFFER" (name^":"^ls.name) in

(*
  eprintf "%s\n" (Mcredit.to_string_list s.credit) ;
*)

(* ================= performance test ================== *)
(*
  let pending = ref [] in
  let add_time origin =
    pending := (origin,(Hsys.gettimeofday ())) :: !pending in
  let print_time () =
    List.iter (fun (origin,time) -> 
      Printf.fprintf s.file_handler "%d\t%12.10f\n" origin time
    ) (List.rev !pending) 
  in
*)
(* ===================================================== *)

  let up_hdlr ev abv hdr = match getType ev, hdr with
    (* Increment amount of credit to pass back to sender.
     * If the amount of credit is beyond the threshhold,
     * send an acknowledgement.  Finally, deliver the
     * message.  
     *)
  | ECast, Data ->
      let origin = getOrigin ev in
      let len = msg_len s ev in

      (* Don't do any credit stuff with casts from myself.
       *)
      if origin <>| ls.rank then (
	let current_credit = Mcredit.got_msg s.credit origin len in
	if current_credit >| s.ack_thresh then (
	  let nacks = current_credit / s.ack_thresh in
	  let remainder = current_credit - (nacks * s.ack_thresh) in
	  dnlm (sendRank name origin) (Ack nacks) ;
	  Mcredit.set_credit s.credit origin remainder
	)
      ) ;
      up ev abv

  | _, NoHdr -> up ev abv
  | _        -> failwith "bad header"

  and uplm_hdlr ev hdr = match getType ev,hdr with
    (* Some credit were sent back, send more data if its waiting.
     *)
  | ESend, Ack(nacks) ->
      let origin = getOrigin ev in 
      for i = 1 to nacks do
      	Mcredit.got_credit s.credit origin ;
      done ;

(* ================== performance test ======================= *)
(*
      add_time origin;
*)
(* =========================================================== *)

      let burst_len = ref 0 in

      while (not (Queue.empty s.send_buf))
	 && (Mcredit.check s.credit) 
      do
	(* Queue.take will not fail here.
	 *)
        let ev,abv = Queue.take s.send_buf in
	let len = msg_len s ev in
	burst_len := !burst_len + len ;
	Mcredit.take s.credit len ;
(*	logl (fun () -> Mcredit.to_string_list s.credit) ; *)
	dn ev abv Data
      done;  (* while *)
      log (fun () -> sprintf "burst_len=%d" !burst_len) ;
  
      ack ev ; free name ev 
  | _ -> failwith "unknown local message"

  and upnm_hdlr ev = match getType ev with
    (* EFail: Mark failed members, and pass on up.
     *)
  | EFail ->      
      let failed = getFailures ev in
      List.iter (fun rank ->
	Mcredit.fail s.credit rank ;
      ) failed ;
      upnm ev

(* ================ performance test ======================== *)
(*
  | EExit -> 
      print_time ();
      close_out s.file_handler ;

      upnm ev
*)
(* ========================================================== *)

  | EAccount ->
      logb (fun () -> sprintf "blocked(msgs):%s" 
	(string_of_queue_len s.send_buf)) ;
      logb (fun () -> sprintf "blocked(byte):%s" 
	(string_of_queue_bytes s s.send_buf)) ;
      logl (fun () -> Mcredit.to_string_list s.credit) ;
      Mcredit.clear s.credit ;
      upnm ev

  | EDump -> ( dump (ls,vs) s ; upnm ev )
  | _ -> upnm ev

  and dn_hdlr ev abv = match getType ev with

    (* Send a message to each destination.  If we don't have
     * any credit, then buffer it.  
     *)
  | ECast ->      
      if getUnreliable ev then (
        dn ev abv NoHdr
      ) else if Mcredit.check s.credit then (
	Mcredit.take s.credit (msg_len s ev) ;
	dn ev abv Data
      ) else (
	Queue.add (ev,abv) s.send_buf
      )

  | _ -> dn ev abv NoHdr

  and dnnm_hdlr = dnnm

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr_state init hdlrs None (FullNoHdr Data) args vs

let _ = 
  Param.default "mflow_window" (Param.Int 50000) ;
  Param.default "mflow_ack_thresh" (Param.Int 25000) ;
  Param.default "mflow_overhead" (Param.Int 300) ;

(* ================ performance test ====================== *)
(*
  Param.default "fname" (Param.String "NULL") ;
*)
(* ======================================================== *)

  Layer.install name (Layer.init l)

(**************************************************************)
