(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  TransisE, (Version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(* TCAUSAL.ML : Checking causal ordering properties *)
(* Author: Ohad Rodeh, 12/96 *)
(* Based on code by: Mark Hayden *)
(**************************************************************)
(* 
General description
-------------------
  This layer checks the causal order generated by the tcausal
layer. It performs two tests - 
(1) consistency of the ack_vectors attached to messages, by 
   Tcausal.
(2) Independent check, via addition of a header containing
sender_id, sequence number, to each message ECast through
this layer. Upon receite, consistency check upon this header. 
*)
(**************************************************************)
open Layer
open Trans
open Util
open Event
open View
(**************************************************************)
let name = Trace.source_file "CHK_CAUSAL"
(**************************************************************)


type header = NoHdr
| Cast of int * Endpt.id * rank * seqno array

type member = {
  mendpt : Endpt.id ;
  mutable failed : bool ;
  mutable cast_up : int
}

type state = {
    mutable uniqueid : int;
    ack_chk          : int array;     (* Check Tcausal ack vectors *)
    members          : member array   (* Locally generated ack vct's *)
  } 

(**************************************************************)

type ('a,'b) t = (header,'a,'b, state) Layer.t

(**************************************************************)

let init () (ls,vs) = 
  let mmbs = (Array.of_list vs.view) 
  and n = ls.nmembers in {
  uniqueid = 0;
  ack_chk = array_create name n (-1);
  members = Array.map (fun endpt -> {
      mendpt = endpt ;
      failed = false ;
      cast_up = 0 
    }) mmbs
} 

(**************************************************************)

let print_members members =
  for_array (fun rank m ->
    eprintf "  mbr(%d):" rank ;
    if m.failed then
      eprintf "failed\n"
    else
      eprintf "%s,cup=%d\n" (Endpt.string_of_id m.mendpt) m.cast_up 
  ) members

let dump s vs  =
  eprintf "CHK_CAUSAL:dump:%s\n" ls.name ;
  eprintf "  rank=%d\n" ls.rank ;
  print_members s.members

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump s vs ; failwith (name^":"^m) in
  let stop_exec m = dnnm (create name EDump[]); eprintf "%s\n" (name^":"^m) in (* BUG? *)
  let assert = make_assert failwith in
  let log = Trace.log name ls.name in

  let check_acks ack_vct = 
    let v = ref [] in 
    for i = 0 to ls.nmembers-1 do
      if s.members.(i).cast_up <= ack_vct.(i) then v := (i,ack_vct.(i)) :: !v;
    done;
    !v
  in

  let up_hdlr ev abv hdr = match getType ev,hdr with
  | ECast, Cast(unique,chk_from,chk_origin,ack_vct) -> (
      let org = (getOrigin ev) in 
      let m = s.members.(org) in
      let {cast_up=cast_up; mendpt=endpt} = m in

      assert (getCausal ev) "SANITY";
      if chk_from <> m.mendpt  then (
      	eprintf "CHK_CAUSAL:incorrect origin m.mendpt.";
     	eprintf "sent=(%s,%d)" (Endpt.string_of_id chk_from) chk_origin;
	eprintf "<> (%s,%d)=my_info\n" (Endpt.string_of_id m.mendpt) org;
        stop_exec "sanity check failed" 
        ) ;

      let l = check_acks ack_vct in 
      if List.length l <> 0 then (
	eprintf "<CHK_CAUSAL:deliverying (r=%d,s=%d) without predecessor" 
	org s.members.(org).cast_up;
	List.iter (fun (r,s) -> printf "(r=%d,s=%d)" r s) l;
	eprintf ">\n";
        stop_exec "sanity check failed" 
	);

      let some_acks = getAckVct ev in
      let acks = Array.map (fun i -> int_of_some i) some_acks in 
      if s.ack_chk.(org) <>  pred acks.(org) then (
	eprintf "<CHK_CAUSAL:Causal:bad ack vct (r=%d,s=%d should be=%d)\n" 
	org acks.(org) (succ s.ack_chk.(org));
        stop_exec "sanity check failed" 
 	);
      s.ack_chk.(org) <- acks.(org);
      if not (array_for_all2 (fun a ac -> a <= ac) acks s.ack_chk) then (
	eprintf "<CHK_CAUSAL:Causal:bad ack vector (r=%d,s=%d)\n" 
	org acks.(org);
	eprintf "Message acks = %s\n" (string_of_int_array acks);
	eprintf "Check vector = %s\n" (string_of_int_array s.ack_chk);
        stop_exec "sanity check failed";
  	);
      s.members.(org).cast_up <- succ s.members.(org).cast_up;
      up ev abv
      )

  | _, NoHdr when (ev.flags land xxx_causal) <> 0 -> failwith "bad msg"
  | _, NoHdr -> up ev abv
  | _, _     -> failwith "bad up event"
  and uplm_hdlr _ _ = failwith "unknown local message"

  and upnm_hdlr ev = match getType ev with

  | EAlive ->
      let org = (getOrigin ev) in 
      let some_acks = getAckVct ev in
      let acks = Array.map (fun i -> int_of_some i) some_acks in 
      if s.ack_chk.(org) <>  pred acks.(org) then (
	eprintf "<CHK_CAUSAL:ImAlive:bad ack vct (r=%d,s=%d should be=%d)\n"
 	org acks.(org) (succ s.ack_chk.(org));
        stop_exec "sanity check failed" 
 	);
      s.ack_chk.(org) <- acks.(org);
      if not (array_for_all2 (fun a ac -> a <= ac) acks s.ack_chk) then (
	eprintf "<CHK_CAUSAL:ImAlive:bad ack vector (r=%d,s=%d)\n" 
	org acks.(org);
	eprintf "Message acks = %s\n" (string_of_int_array acks);
	eprintf "Check vector = %s\n" (string_of_int_array s.ack_chk);
        stop_exec "sanity check failed";
  	);
      upnm ev

  | EFail ->
      let suspicions = getFailures ev in
      List.iter (fun i ->
      	s.members.(i).failed <- true
      ) suspicions ;
      upnm ev

  | EDump -> (dump s vs ; upnm ev)
  | _ -> upnm ev

  and dn_hdlr ev abv = match getType ev with
  | ECast when not (getUnreliable ev) -> 
      let ack_vct = Array.map (fun m -> m.cast_up-1) s.members in 
      dn ev abv (Cast(s.uniqueid,ls.endpt,ls.rank,ack_vct)) ;
      s.uniqueid <- succ s.uniqueid

  | _ -> dn ev abv NoHdr

  and dnnm_hdlr = dnnm

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

(**************************************************************)
