(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* HSYSSUPP.ML *)
(* Author: Mark Hayden, 6/96 *)
(**************************************************************)
let name = Trace.source_file "HSYSSUPP"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
open Util
(**************************************************************)

let block mbuf handler info time =
  let check = List.map fst info in
  fun () ->
    let avail,_,_ = Hsys.select check [] [] time in
    List.map (fun sock ->
      (* Hack!! We're using tests for physical equality here.
       *)
      match List.assq sock info with
      | Hsys.Handler0(f) -> f ()
      | Hsys.Handler1 ->
	  Mbuf.alloc_dyn 
	    name
	    mbuf
	    (fun buf ofs len -> Hsys.recv sock buf ofs len)
	    (fun iov ->
	      let dint = Iovec.read_int name iov in
	      handler dint iov ;
	      Iovec.free name iov
	    )
	    (fun _ -> failwith "error")
	    false ; ()
    ) avail ;

    avail <> []

let block mbuf handler info time =
  if time = 0.0 then (
    let info = Hsys.manager info in
    let error () = failwith "error" in
    Mbuf.fast name mbuf info Hsys.poll handler error
  ) else (
    let handler dint iov =
      Iovec.break name iov (handler dint)
    in
    block mbuf handler info time
  )

(**************************************************************)

let max_msg_len = Hsys.max_msg_len ()

(**************************************************************)

let tcp sock =
  (* Space for partial recvs.
   *)
  let partial = ref "" in
  let recv () =
    let recd = ref [] in
    let tmp = String.create max_msg_len in
    let len = Hsys.recv sock tmp 0 max_msg_len in
    if len > 0 then (
(*
      eprintf "IP:TCP:got %d bytes\n" len ;
*)
      let tmp = String.sub tmp 0 len in
      let buf = 
	if !partial <> "" then !partial ^ tmp else tmp
      in
      let rec loop buf =
	if String.length buf >= 4 then
	  let len = Hsys.pop_int buf 0 in
	  if len + 4 <= String.length buf then
	    let msg = String.sub buf 4 len in
	    recd := msg :: !recd ;
	    let buf = String.sub buf (4+len) (String.length buf - 4 - len) in
	    loop buf
	  else buf
	else buf
      in
      partial := loop buf ;
      Some(!recd)
(*;
      eprintf "IP:TCP:left %d bytes\n" (String.length !partial)
*)
    ) else (
      None
    )
  in

  (* Temporary scribbling area.
   *)
  let scribble = String.create 4 in

  let send sock buf ofs len =
    let rec loop i =
      if i = len then
	len
      else (
      	let ret = Hsys.send sock buf (ofs + i) (len - i) in
	if ret = 0 then 
	  ret
	else 
	  loop (i + ret)
      )
    in loop 0
  in

  let send buf ofs len =
    if ofs > 4 then (
      String.blit buf (ofs-4) scribble 0 4 ;
      Hsys.push_int buf (ofs-4) len ;
      try 
        let ret = send sock buf (ofs-4) (len + 4) in
	String.blit scribble 0 buf (ofs-4) 4 ;
	ret
      with exc -> (
	String.blit scribble 0 buf (ofs-4) 4 ;
	raise exc
      )
    ) else (
      let tmp = String.create (len + 4) in
      Hsys.push_int tmp 0 len ;
      String.blit buf ofs tmp 4 len ;
      send sock tmp 0 (len + 4)
    )
      
  in (send, recv)

(**************************************************************)
(* This is a hack to send Iovecs quickly.
 *)
(*
external magic : t -> Hsys.iovec array = "%identity"

let sendvopt info iovl =
  let iovl = magic iovl in
  Hsys.sendvopt info iovl
*)
(**************************************************************)
(*
let flatlen = Hsys.max_msg_len ()
let flatbuf = String.create flatlen
let sendvopt info iovl =
  let len = Iovecl.flatten_buf "sendvopt" iovl flatbuf 0 flatlen in
  Hsys.sendopt info flatbuf 0 len
*)
(**************************************************************)
(* BUG: perf: this is kind of a slow way to do things.
 *)
let sendvopt info iovl =
  let iovl =
    Array.map (fun iov ->
      Iovec.read name iov (fun buf ofs len -> (buf,ofs,len))
    ) iovl
  in
  Hsys.sendvopt info iovl

(**************************************************************)
