(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* GLUE.ML *)
(* Author: Mark Hayden, 10/96 *)
(**************************************************************)
open Util
open Event 
open Layer
open View
(**************************************************************)
let name = "GLUE"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

module type S =
  sig
    type ('a,'b) t
    val convert : ('a,'b,'c) Layer.basic -> ('a,'b) t
    val compose : ('a,'b) t -> ('b,'c) t -> ('a,'c) t
    type ('top,'bot) init = 
      ('top,'bot) t -> 
      'top -> 'bot ->
      Sched.t ->
      View.full ->
      (Event.up -> unit) -> 
      (Event.dn -> unit)
    (*
    val init : ('top,'bot) init
    *)
    val init : ('top,('a,'b,'c)Layer.msg) init
  end

(**************************************************************)

module ImperativeOld : S = struct
  let name = "Imperative"
  let failwith s = failwith (Util.failmsg name s)

  type ('a,'b) t = 
    View.full -> 
    Sched.t -> 
    (('a,'b) handlers_lout -> ('a,'b) handlers_lin)

  type ('top,'bot) init = 
    ('top,'bot) t -> 
    'top -> 'bot ->
    Sched.t ->
    View.full ->
    (Event.up -> unit) -> 
    (Event.dn -> unit)

  let convert l vs sched =
    let _,h = l vs in
    let h out =
      let {up_lin=up;dn_lin=dn} = h out in
      let up ev msg = Sched.enqueue_2arg sched up ev msg 
      and dn ev msg = Sched.enqueue_2arg sched dn ev msg in
      {up_lin=up;dn_lin=dn}
    in h

  let compose top bot vs sched =
    let l {up_lout=top_up;dn_lout=bot_dn} =
      let mid_dn_r = ref (fun e h -> failwith "compose:sanity") in
      let mid_dn e h = !mid_dn_r e h in

      let {up_lin=mid_up;dn_lin=top_dn} =
	top vs sched {up_lout=top_up;dn_lout=mid_dn} in
      let {up_lin=bot_up;dn_lin=mid_dn} =
	bot vs sched {up_lout=mid_up;dn_lout=bot_dn} in

      mid_dn_r := mid_dn ;
      {up_lin=bot_up;dn_lin=top_dn}
    in l

  let init l top_msg bot_nomsg sched vs up_out =
    let dn_r = ref (fun _ _ -> failwith "premature event") in
    let dn ev msg = !dn_r ev msg in
    let up ev msg = up_out ev in

    let {up_lin=up;dn_lin=dn} = l vs sched {up_lout=up;dn_lout=dn} in
    Sched.enqueue_2arg sched up (create "init_protocol(a)" EInit[]) bot_nomsg ;
    dn_r := Config_trans.f bot_nomsg vs up ;
    let dn ev = dn ev top_msg in
    dn
end

(**************************************************************)

module Imperative : S = struct
  let name = "ImperativeOpt"
  let failwith s = failwith (Util.failmsg name s)

  type ('a,'b) t = 
    View.full -> 
    Sched.t -> 
    (('a,'b) handlers_lout -> ('a,'b) handlers_lin)

    type ('top,'bot) init = 
      ('top,'bot) t -> 
      'top -> 'bot ->
      Sched.t ->
      View.full ->
      (Event.up -> unit) -> 
      (Event.dn -> unit)

  let convert l vs sched =
    let _,h = l vs in
    let h out =
      let count = ref 0 in
      let {up_lin=up;dn_lin=dn} = h out in
	
      let wrap handler = 
      	let enqueued ev msg =
	  handler ev msg ;
	  decr count
	in
	fun ev msg ->
	  if !count = 0 then (
	    incr count ;
	    handler ev msg ;
	    decr count
	  ) else (
	    incr count ;		(* matched in enqueued *)
	    Sched.enqueue_2arg sched enqueued ev msg
	  )
      in
      let up = wrap up
      and dn = wrap dn in
      {up_lin=up;dn_lin=dn}
    in h

  let compose top bot vs sched =
    let l {up_lout=top_up;dn_lout=bot_dn} =
      let mid_dn_r = ref (fun e h -> failwith "compose:sanity") in
      let mid_dn e h = !mid_dn_r e h in

      let {up_lin=mid_up;dn_lin=top_dn} =
	top vs sched {up_lout=top_up;dn_lout=mid_dn} in
      let {up_lin=bot_up;dn_lin=mid_dn} =
	bot vs sched {up_lout=mid_up;dn_lout=bot_dn} in

      mid_dn_r := mid_dn ;
      {up_lin=bot_up;dn_lin=top_dn}
    in l

  let init l top_msg bot_nomsg sched vs up_out =
    let dn_r = ref (fun _ _ -> failwith "premature event") in
    let dn ev msg = !dn_r ev msg in
    let up ev msg = up_out ev in

    let {up_lin=up;dn_lin=dn} = l vs sched {up_lout=up;dn_lout=dn} in
    Sched.enqueue_2arg sched up (create "init_protocol(a)" EInit[]) bot_nomsg ;
    dn_r := Config_trans.f bot_nomsg vs up ;
    let dn ev = dn ev top_msg in
    dn
end

(**************************************************************)

module Functional : S = struct
  let name = "Func"
  let failwith s = failwith (Util.failmsg name s)

  type ('a,'b) z = ('a,'b) Event.dirm -> ('b,'a) Event.dirm list 

  type ('b,'a) t = View.full -> ('a,'b) z

    type ('top,'bot) init = 
      ('top,'bot) t -> 
      'top -> 'bot ->
      Sched.t ->
      View.full ->
      (Event.up -> unit) -> 
      (Event.dn -> unit)

  let list_of_queue q =
    let l = Util.list_of_queue q in
    Util.queue_clean (fun _ -> ()) q ;
    l

  (* For conversion, we need to use imperative queues.
   *)
  let convert l vs = 
    let q = Queue.create () in
    let up ev msg = Queue.add (UpM(ev,msg)) q in
    let dn ev msg = Queue.add (DnM(ev,msg)) q in
    let (_,h) = l vs in
    let {up_lin=up;dn_lin=dn} =
      h {up_lout=up;dn_lout=dn}
    in
    let hdlr ev = match ev with
    | UpM(ev,msg) ->
	up ev msg ;
	list_of_queue q
    | DnM(ev,msg) ->
	dn ev msg ;
	list_of_queue q
    in hdlr

  (* A totally functional layer composition function.
   *)
  let compose top bot vs =
    (* Initialize the two layers.
     *)
    let top = top vs
    and bot = bot vs
    in

    (* Three mutually recursive functions.
     * split_top : splits up and down events into the emit and midl queues
     * split_bot : splits up and down events into the emit and midl queues
     * middle : applies next event in midl queue to top or bottom layer
     *)
    let rec split_top emit midl = function
      | [] -> middle emit midl
      | UpM(ev,msg) :: tl -> 
	  let emit = Fqueue.add (UpM(ev,msg)) emit in
	  split_top emit midl tl
      | DnM(ev,msg) :: tl ->
	  let midl = Fqueue.add (DnM(ev,msg)) midl in
	  split_top emit midl tl
    and split_bot emit midl = function
      | [] -> middle emit midl
      | UpM(ev,msg) :: tl -> 
	  let midl = Fqueue.add (UpM(ev,msg)) midl in
	  split_bot emit midl tl
      | DnM(ev,msg) :: tl ->
	  let emit = Fqueue.add (DnM(ev,msg)) emit in
	  split_bot emit midl tl
    and middle emit midl =
      if Fqueue.is_empty midl then (
	Fqueue.to_list emit
      ) else (
	let hd,midl = Fqueue.take midl in
	match hd with
	| UpM(ev,msg) -> split_top emit midl (top (UpM(ev,msg)))
	| DnM(ev,msg) -> split_bot emit midl (bot (DnM(ev,msg)))
      )
    in

    (* Outer handler takes a single event and then passes to
     * appropriate layer and then splits the emitted events.
     *)
    let hdlr = function
      | DnM(ev,msg) -> 
	  let emitted = top (DnM(ev,msg)) in
	  split_top Fqueue.empty Fqueue.empty emitted
      | UpM(ev,msg) -> 
	  let emitted = bot (UpM(ev,msg)) in
	  split_bot Fqueue.empty Fqueue.empty emitted
    in
    hdlr

  let init l top_msg bot_nomsg sched vs up_out =
    let dn_r = ref (fun _ _ -> failwith "premature event") in
    let dn ev msg = !dn_r ev msg in
    let up ev msg = up_out ev in

    let hdlr = l vs in

    let up ev msg =
      let emit = hdlr (UpM(ev,msg)) in
      List.iter (function
	| UpM(ev,msg) -> up ev msg
	| DnM(ev,msg) -> dn ev msg
      ) emit
    and dn ev =
      let emit = hdlr (DnM(ev,top_msg)) in
      List.iter (function
	| UpM(ev,msg) -> up ev msg
	| DnM(ev,msg) -> dn ev msg
      ) emit
    in

    (* BUG: init may not be first event. *)
    Sched.enqueue_2arg sched up (create "init_protocol(a)" EInit[]) bot_nomsg ;
    dn_r := Config_trans.f bot_nomsg vs up ;
    dn
end

(**************************************************************)
(**************************************************************)
(**************************************************************)

let lock_count = ref 0
let thread_create = ref 0
let context_count = ref 0

let stats () =
  eprintf "GLUE:stats\n" ;
  eprintf "  thread_create=%d" !thread_create ;
  eprintf "  lock_count=%d\n" !lock_count ;
  eprintf "  context_count=%d\n" !context_count

(*
module Thread = struct
  type t = Thread.t
  let create f a = 
    incr thread_create ;
    incr context_count ;
    Thread.create f a
  let exit = Thread.exit
  let critical_section = Thread.critical_section
  let self = Thread.self
  let sleep () = 
    incr context_count ;
    Thread.sleep ()
  let wakeup = Thread.wakeup
end

module Mutex = struct
  type t = { mutable locked: bool; mutable waiting: Thread.t list }

  let create () = { locked = false; waiting = [] }

  let rec lock m =
    if m.locked then begin                (* test and set atomic *)
      Thread.critical_section := true;
      m.waiting <- Thread.self() :: m.waiting;
      Thread.sleep();
      lock m
    end else begin
      incr lock_count ;
      m.locked <- true                    (* test and set atomic *)
    end

  let try_lock m =                        (* test and set atomic *)
    if m.locked then false else begin 
      incr lock_count ;
      m.locked <- true; true 
    end

  let unlock m =
    (* Don't play with Thread.critical_section here because of Condition.wait *)
    let w = m.waiting in                  (* atomic *)
    m.waiting <- [];                      (* atomic *)
    m.locked <- false;                    (* atomic *)
    List.iter Thread.wakeup w
end

module Condition = struct
  type t = { mutable waiting: Thread.t list }

  let create () = { waiting = [] }

  let wait cond mut =
    Thread.critical_section := true;
    Mutex.unlock mut;
    cond.waiting <- Thread.self() :: cond.waiting;
    Thread.sleep();
    Mutex.lock mut

  let signal cond =
    match cond.waiting with               (* atomic *)
      [] -> ()
    | th :: rem -> cond.waiting <- rem (* atomic *); Thread.wakeup th

  let broadcast cond =
    let w = cond.waiting in                  (* atomic *)
    cond.waiting <- [];                      (* atomic *)
    List.iter Thread.wakeup w
end
*)
(**************************************************************)
(**************************************************************)
(**************************************************************)

module Threader : S = struct
  let name = "Threaded"
  let failwith s = failwith (Util.failmsg name s)

  type ('a,'b) t = 
    View.full -> 
    (('a,'b) handlers_lout -> ('a,'b) handlers_lin)

    type ('top,'bot) init = 
      ('top,'bot) t -> 
      'top -> 'bot ->
      Sched.t ->
      View.full ->
      (Event.up -> unit) -> 
      (Event.dn -> unit)

  let convert l vs = failwith "convert:unimplemented"
  let compose l1 l2 = failwith "compose:unimplemented"
  let init _ _ _ _ _ _ = failwith "init:unimplemented"

(*
  let convert l vs =
    let _,h = l vs in
    let h out =
      let {empty_lin=empty;up_lin=up;dn_lin=dn} = h out in
      let seqno_r = ref 0 in
      let seqno_w = ref 0 in
      let seqno_rl = Mutex.create () in
      let seqno_rc = Condition.create () in
      let seqno_wl = Mutex.create () in

      let insert f =
	Mutex.lock seqno_wl ;
	let seqno = !seqno_w in
	incr seqno_w ;
	Mutex.unlock seqno_wl ;

	let opt = 
	  if Mutex.try_lock seqno_rl then
	    if !seqno_r = seqno then (
	      f () ;
	      incr seqno_r ;
	      Condition.broadcast seqno_rc ;(* Should not be necessary *)
	      Mutex.unlock seqno_rl ;
              true
	    ) else (
	      Mutex.unlock seqno_rl ;
	      false
	    )
	  else false
	in	      
	if not opt then (
	  let spawn () =
	    Mutex.lock seqno_rl ;
	    while !seqno_r <> seqno do
	      Condition.wait seqno_rc seqno_rl
	    done ;
	    f () ;
	    incr seqno_r ;
	    Condition.broadcast seqno_rc ;
	    Mutex.unlock seqno_rl ;
	    Thread.exit ()
	  in
	  Thread.create spawn () ; 
	  ()
	)
      in

      let up ev msg = 
        insert (fun () -> up ev msg)
      and dn ev msg = 
        insert (fun () -> dn ev msg)
      in

      {empty_lin=empty;up_lin=up;dn_lin=dn}
    in h

  let compose top bot vs =
    let l {empty_lout=top_empty;up_lout=top_up;dn_lout=bot_dn} =
      let mid_dn_r = ref (fun e h -> failwith "compose:sanity") in
      let mid_dn e h = !mid_dn_r e h in

      let {empty_lin=mid_empty;up_lin=mid_up;dn_lin=top_dn} =
	top vs {empty_lout=top_empty;up_lout=top_up;dn_lout=mid_dn} in
      let {empty_lin=bot_empty;up_lin=bot_up;dn_lin=mid_dn} =
	bot vs {empty_lout=mid_empty;up_lout=mid_up;dn_lout=bot_dn} in

      mid_dn_r := mid_dn ;
      {empty_lin=bot_empty;up_lin=bot_up;dn_lin=top_dn}
    in l

  let init l top_msg bot_nomsg sched vs up_out =
    let dn_r = ref (fun _ _ -> failwith "premature event") in
    let dn ev msg = !dn_r ev msg in
    let up ev msg = up_out ev in
    let {up_lin=up;dn_lin=dn} = l vs {empty_lout=top_msg;up_lout=up;dn_lout=dn} in
    dn_r := Config_trans.f bot_nomsg vs up ;
    up (create "init_protocol(a)" EInit[]) bot_nomsg ;
    let dn ev = dn ev top_msg in
    dn
*)
end

(**************************************************************)
(**************************************************************)

type glue = 
  | Imperative 
  | Functional
  | Threaded

type ('a,'b) t =
  | Imp of ('a,'b) Imperative.t
  | Fun of ('a,'b) Functional.t
  | Thr of ('a,'b) Threader.t

type ('top,'bot) init = 
    ('top,'bot) t -> 
      'top -> 'bot ->
      	Sched.t ->
	  View.full ->
      	    (Event.up -> unit) -> 
	      (Event.dn -> unit)
	      
let of_string s = 
  match String.uppercase s with
  | "IMPERATIVE" -> Imperative
  | "FUNCTIONAL" -> Functional
  | "THREADED" -> Threaded
  | _ -> failwith "glue_of_string:unknown glue"

let convert g l =
  match g with
  | Imperative -> Imp (Imperative.convert l)
  | Functional -> Fun (Functional.convert l)
  | Threaded   -> Thr (Threader  .convert l)

let compose l1 l2 = match l1,l2 with
| Imp(l1),Imp(l2) -> Imp(Imperative.compose l1 l2)
| Fun(l1),Fun(l2) -> Fun(Functional.compose l1 l2)
| Thr(l1),Thr(l2) -> Thr(Threader  .compose l1 l2)
| _,_ -> failwith "mismatched layers"

let init = function 
| Imp(l) -> Imperative.init l
| Fun(l) -> Functional.init l
| Thr(l) -> Threader  .init l

(**************************************************************)
(**************************************************************)
