(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* CONFIG_TRANS.ML *)
(* Author: Mark Hayden, 10/96 *)
(**************************************************************)
open Util
open Event 
open Layer
open View
(**************************************************************)
let name = "CONFIG_TRANS"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
let name_send = "CONFIG_TRANS(send)"
let name_cast = "CONFIG_TRANS(cast)"
let name_recv = "CONFIG_TRANS(recv)"
let name_mrge = "CONFIG_TRANS(mrge)"
let name_goss = "CONFIG_TRANS(goss)"
(**************************************************************)

let marsh_gossip, unmarsh_gossip = 
  Util.make_marsh "CONFIG_TRANS:gossip" true

let pack_gossip ev =
  let exchange = getExtender
    (function ExchangeGos(a) -> (Some (Some (a))) | _ -> None)
    None ev
  in

  let switch = getExtender
    (function SwitchGos(a,b,c) -> (Some (Some (a,b,c))) | _ -> None)
    None ev
  in

  let heal = getExtender
    (function HealGos(a,b,c,d) -> (Some (Some (a,b,c,d))) | _ -> None)
    None ev
  in

  match exchange,switch,heal with
  | None,None,None -> None
  | _ ->
      let marsh = Marsh.init () in
      Marsh.write_option marsh (Marsh.write_string marsh) exchange ;
      Marsh.write_string marsh (marsh_gossip (switch,heal)) ;
      Some [| Iovec.of_string name (Marsh.marsh marsh) |]
  
let unpack_gossip secure iovl =
  let iov = Iovecl.flatten name_goss iovl in
  let msg = Iovec.to_string name_goss iov in
  Iovec.free name_goss iov ;
  let msg = Marsh.unmarsh msg in

  let exchange = 
    Marsh.read_option msg (fun () -> Marsh.read_string msg)
  in

  (* Only read the other fields if things are secure.
   *)
  let switch,heal = 
    if secure then (
      let msg = Marsh.read_string msg in
      unmarsh_gossip msg 0 (String.length msg)
    ) else (None,None)
  in

  let exchange = match exchange with
  | Some(a) -> [ExchangeGos(a)] 
  | None -> []
  in

  let switch = match switch with 
  | Some(a,b,c) when secure -> [SwitchGos(a,b,c)] 
  | _ -> []
  in

  let heal = match heal with
  | Some(a,b,c,d) when secure -> [HealGos(a,b,c,d)] 
  | _ -> []
  in
  
  let fields = heal @ switch @ exchange in

  if fields = [] then
    None 
  else (
    let ev = create name EGossipExt fields in
    Some ev
  )

(**************************************************************)

let f bot_nomsg (ls,vs) up_hdlr =
  let pack,unpack = Util.make_magic () in

  let alarm = Alarm.get () in
  let schedule = Alarm.alarm alarm
    (fun time -> up_hdlr (timerTime name time) bot_nomsg)
  in

  let async_disable = 
    Async.add (vs.group,ls.endpt) (fun () ->
      up_hdlr (create name EAsync[
	(Time (Alarm.gettime alarm))
      ]) bot_nomsg
    )
  in

  let (gos_gossip,gos_disable) =
    let enable router xmit receive =
      let receive _ _ =
	receive (fun secure iov -> 
	  match unpack_gossip secure iov with
	  | None -> ()			(* No fields *)
	  | Some ev ->
	      up_hdlr ev bot_nomsg
        )
      in
      
      let trans =
      	Transport.f
	vs.version vs.group ls.endpt
      	Stack_id.Gossip
      	Proto.raw vs.key
      	vs.view 
	vs.address 
	vs.view_id
      	true
      	router
      	receive
      in

      let gossip = xmit (Transport.gossip trans) in
      let disable () = Transport.disable trans in
      (gossip,disable)
    in

    if vs.key = Security.NoKey then (
      let receive f =
	fun _ _ iov ->
	  f true iov
      in

      let xmit f = fun iov ->
	f (-1) None iov 
      in

      enable Unsigned.f xmit receive
    ) else (
      let receive f =
	fun secure _ msg iov ->
	  f secure iov
      in
      
      let xmit f =
	fun iov ->
	  f true (-1) None iov 
      in
      
      enable Signed.f xmit receive
    )
  in

  let (pri_cast,pri_send,pri_merge,pri_disable) =
    let enable router xmit receive =
      let receive kind rank =
	let ut = match kind with
	| Conn.Send  -> ESend
	| Conn.Cast  -> ECast
	| Conn.Other -> EMergeRequest
	in
	let debug = addinfo (Event.string_of_type ut) name in
	receive (fun msg iov -> 
	  Iovecl.ref debug iov ;
	  up_hdlr (bodyCore name ut rank iov) msg
	)
      in
      
      let trans =
      	Transport.f
	vs.version vs.group ls.endpt
      	Stack_id.Primary
      	vs.proto_id vs.key
      	vs.view vs.address vs.view_id
      	false
      	router
      	receive
      in
      
      let cast = xmit (Transport.cast trans) in
      let sends = array_create name ls.nmembers None in
      let send rank =
	match sends.(rank) with
	| Some send -> send
	| None ->
	    let send = xmit (Transport.send trans rank) in
	    sends.(rank) <- Some(send) ;
	    send
      in
      let merge vid con = xmit (Transport.merge trans vid con) in
      let disable () = Transport.disable trans in
      (cast,send,merge,disable)
    in

    if vs.key = Security.NoKey then (
(*
      let receive f kind rank msg iov =
	let msg = match msg with
	| None -> failwith "got message without protocol headers"
	| Some d -> unpack d
	in
	f kind rank msg iov
      in

      let xmit f msg iov = f (Some (pack(msg))) iov in
      
      enable Scale.f xmit receive
*)
      let receive f =
	fun seqno msg iov ->
	  let msg = match msg with
	  | None -> Local_seqno seqno
	  | Some d -> unpack d
	  in
	  f msg iov
      in

      let xmit f = fun msg iov ->
	match msg with
	| Local_seqno seqno -> 
(*
	    printf "opt\n" ;
*)
	    f seqno None iov
	| _ ->
(*
	    printf "nonopt\n" ;
*)
	    f (-1) (Some (pack(msg))) iov 
      in

      enable Unsigned.f xmit receive
    ) else (
      let receive f =
	fun secure _ msg iov ->
	  if secure then (
	    let msg = match msg with
	    | None -> failwith "got message without protocol headers"
	    | Some d -> unpack d
	    in
	    f msg iov
	  )
      in
      
      let xmit f =
	fun msg iov ->
	  f true (-1) (Some (pack(msg))) iov 
      in
      
      enable Signed.f xmit receive
    )
  in
  
  let dn ev msg = match getType ev with
  | ECast ->
      pri_cast msg (getIov ev) ;
      free name_cast ev
  | ESend ->
      let iov = getIov ev in
      begin
	match getRanks ev with
      	| [] -> ()
      	| [rank] ->
	    pri_send rank msg iov
      	| ranks ->
	    List.iter (fun rank ->
	      pri_send rank msg iov
	    ) ranks
      end ;
      free name_send ev
  | EMergeRequest|EMergeGranted|EMergeDenied ->
      let contact,view_id = getContact ev in
      pri_merge view_id contact msg (getIov ev) ;
      free name_mrge ev
  | EExit -> 
      pri_disable () ;
      gos_disable () ;
      async_disable ()
  | ETimer -> 
      Alarm.schedule schedule (getAlarm ev)
  | EGossipExt -> (
      match pack_gossip ev with
      |	None -> ()
      |	Some msg ->
	  gos_gossip msg
    )
  | _ -> failwith "sanity:bad dn"
  in

  dn

(**************************************************************)
