/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/* 
 * Contents: Interface to pthreads.
 *
 * Author:  Alexey Vaysburd, December 1996
 */

#include <assert.h>
#include <pthread.h>
#include "hot_sys.h"
#include "hot_error.h"
#include "hot_thread.h"

#ifdef HOT_USE_THREADS

static pthread_t td;

typedef void*(*hot_thread_routine)(void*);

/******************************** Threads ***********************************/

/* Create a new thread
 */
hot_err_t hot_thread_Create(void (*routine)(void*), 
			    void *arg, 
			    hot_thread_attr_t *attr) {
  pthread_attr_t pthr_attr;	
#ifndef OSF1_THREADS
  if (pthread_attr_init(&pthr_attr))
    return hot_err_Create(0,"hot_thread_Create: pthread_attr_init");
  if (attr != NULL) {
    if (pthread_attr_setstacksize(&pthr_attr, attr->stacksize))
      return hot_err_Create(0,"hot_thread_Create: pthread_attr_setstacksize");
  }
  if (pthread_create(&td,NULL,(hot_thread_routine) routine, arg))
    return hot_err_Create(0,"hot_thread_Create: pthread_create");
#else OSF1_THREADS
  if (pthread_attr_create(&pthr_attr))
    return hot_err_Create(0,"hot_thread_Create: pthread_attr_create");
  if (pthread_create(&td,pthr_attr,(hot_thread_routine) routine, arg))
    return hot_err_Create(0,"hot_thread_Create: pthread_create");
#endif OSF1_THREADS
  return HOT_OK ;
}

/* Yield the processor to another thread.
 */
hot_err_t hot_thread_Yield() {
#ifndef OSF1_THREADS
  pthread_yield(NULL);
#else
  pthread_yield();
#endif
  return HOT_OK ;
}

/* Sleep for usecs.
 */
hot_err_t hot_thread_Usleep(int usecs) {
  struct timespec ts ;
  int ret ;
  int secs ;
  if (usecs <= 0)
    return HOT_OK ;

  secs = usecs / 1000000/*1E6*/ ;
  usecs %= 1000000/*1E6*/ ;
  
  ts.tv_sec = secs ;
  ts.tv_nsec = usecs * 1000 ;

  assert(ts.tv_nsec >= 0) ;
  assert(ts.tv_nsec < 1000000000/*1E9*/) ;
  assert(ts.tv_sec  >= 0) ;

  ret = nanosleep(&ts, NULL) ;
  if (ret < 0)
    return hot_err_Create(0,"hot_thread_Usleep: nanosleep") ;
  return HOT_OK ;
}

/******************************** Semaphores *******************************/

struct hot_sema {
  pthread_mutex_t mutex;
  pthread_cond_t cond;
  int value ;
  int nwake ;
};

/* Create a semaphore.
 */
static
hot_err_t hot_sema_Create_help(int initial_value, /*OUT*/ hot_sema_t sema) {

  /* Initialize mutex.
   */
  {
    pthread_mutexattr_t attr;
#ifndef OSF1_THREADS
    if (pthread_mutex_init(&sema->mutex, NULL))
      return hot_err_Create(0,"hot_lck_Create: pthread_mutex_init") ;
#else OSF1_THREADS
    if (pthread_mutexattr_create(&attr))
      return hot_err_Create(0,"hot_lck_Create: pthread_mutexattr_init");
    if (pthread_mutex_init(&sema->mutex, attr))
      return hot_err_Create(0,"hot_lck_Create: pthread_mutex_init") ;
#endif OSF1_THREADS
  }
  
  /* Initialize condition variable.
   */
  {
    pthread_condattr_t attr;
#ifndef OSF1_THREADS
    if (pthread_cond_init(&sema->cond, NULL))
      return hot_err_Create(0,"hot_sema_Create: pthread_cond_init");
#else OSF1_THREADS
    if (pthread_condattr_create(&attr))
      return hot_err_Create(0,"hot_sema_Create: pthread_condattr_init");
    if (pthread_cond_init(&sema->cond, attr))
      return hot_err_Create(0,"hot_sema_Create: pthread_cond_init");
#endif OSF1_THREADS
  }
  
  sema->value = initial_value;
  sema->nwake = 0 ;
  return HOT_OK;
}


hot_err_t hot_sema_Create(int initial_value, /*OUT*/ hot_sema_t *semap) {
  hot_sema_t sema ;
  hot_err_t err ;
  assert(semap) ;
  *semap = NULL ;

  sema = (hot_sema_t) malloc(sizeof(struct hot_sema)) ;
  if (!sema) 
    return hot_err_Create(0,"hot_sema_Create: malloc returned NULL");

  err = hot_sema_Create_help(initial_value, sema) ;
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err)) ;

  *semap = sema ;
  return HOT_OK ;
}


hot_err_t hot_sema_Destroy(hot_sema_t sema) {
  assert(sema) ;

  if (pthread_mutex_destroy(&sema->mutex) ||
      pthread_cond_destroy(&sema->cond))
    return hot_err_Create(0,"hot_sema_Destroy");

  /* Zero the memory.
   */
  memset(sema,0,sizeof(*sema)) ;
  free(sema);
  return HOT_OK;
}

hot_err_t hot_sema_Inc(hot_sema_t sema) {
  int value ;
  assert(sema) ;

  if (pthread_mutex_lock(&sema->mutex))
    return hot_err_Create(0,"hot_sema_Inc: pthread_mutex_lock") ;

  /* Save the semaphore value prior to incrementing it.
   */
  value = sema->value ;

  /* If the current value is less than 0 then increment
   * the count of the # threads to awaken.
   */
  if (sema->value < 0)
    sema->nwake++ ;

  sema->value ++ ;

  if (pthread_mutex_unlock(&sema->mutex))
    return hot_err_Create(0,"hot_sema_Inc: pthread_mutex_unlock") ;

  /* If the old value was less than zero then some thread is blocked,
   * so we signal it.  RvR suggested using broadcast instaed of 
   * signal.
   */
  if (value < 0) {
    if (pthread_cond_broadcast(&sema->cond))
      return hot_err_Create(0,"hot_sema_Inc: pthread_cond_signal");
  }

  return HOT_OK;	
}

hot_err_t hot_sema_Dec(hot_sema_t sema) {
  assert(sema) ;

  if (pthread_mutex_lock(&sema->mutex))
    return hot_err_Create(0,"hot_sema_Dec: pthread_mutex_lock") ;

  sema->value -- ;

  /* If the semaphore's value is now negative then
   * block on the condition variable until the
   * value for nwake is greater than zero.
   */
  if (sema->value < 0) {
    while(sema->nwake == 0) {
      /* It is a bug if we have to block on the condition
       * variable when value<0 and nwake==0.
       */
      assert(sema->value < 0) ;

      if (pthread_cond_wait(&sema->cond,&sema->mutex)) {
	/* Release the mutex before returning.
	 */
	if (pthread_mutex_unlock(&sema->mutex))
	  return hot_err_Create(0,"hot_sema_Dec: pthread_mutex_unlock(error)") ;
	return hot_err_Create(0,"hot_sema_Dec: pthread_cond_wait") ;
      }
    }

    /* Decrement the count for nwake.
     */
    assert(sema->nwake > 0) ;
    sema->nwake -- ;
  }
  
  if (pthread_mutex_unlock(&sema->mutex))
    return hot_err_Create(0,"hot_sema_Dec: pthread_mutex_unlock") ;
  
  return HOT_OK;
}

/******************************** Locks ***********************************/

struct hot_lock {
  struct hot_sema sema ;
} ;

/* Create a lock.
 */
hot_err_t hot_lck_Create(hot_lock_t *lockp) {
  hot_lock_t lock ;
  hot_err_t err ;
  assert(lockp) ;
  *lockp = NULL ;

  lock = (hot_lock_t) malloc(sizeof(*lock)) ;
  if (!lock) 
    return hot_err_Create(0,"hot_lock_Create: malloc returned NULL");

  err = hot_sema_Create_help(1, &lock->sema) ;
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err)) ;

  *lockp = lock ;
  return HOT_OK ;
}

/* Destroy a lock
 */
hot_err_t hot_lck_Destroy(hot_lock_t l) {
  return hot_sema_Destroy(&l->sema) ;
}

/* Lock a lock
 */
hot_err_t hot_lck_Lock(hot_lock_t l) {
  return hot_sema_Dec(&l->sema) ;
}

/* Unlock a lock
 */
hot_err_t hot_lck_Unlock(hot_lock_t l) {
  return hot_sema_Inc(&l->sema) ;
}

/******************************** Locks ***********************************/

#else /* HOT_USE_THREADS */

struct hot_sema {
  int magic ;
} global_sema ;

struct hot_lock {
  int magic ;
} global_lock ;


hot_err_t hot_thread_Create(void (*routine)(void*), 
			    void *arg, 
			    hot_thread_attr_t *attr) { 
  (*routine)(arg);
  return HOT_OK;
}
hot_err_t hot_thread_Yield(void) { return HOT_OK; }
hot_err_t hot_thread_Usleep(int usecs) { return HOT_OK; }
hot_err_t hot_lck_Create(hot_lock_t *l) { 
  *l = &global_lock ;
  return HOT_OK; }
hot_err_t hot_lck_Destroy(hot_lock_t l) { return HOT_OK; }
hot_err_t hot_lck_Lock(hot_lock_t l) { return HOT_OK; }
hot_err_t hot_lck_Unlock(hot_lock_t l) { return HOT_OK; }
hot_err_t hot_sema_Create(int initial_value, /*OUT*/ hot_sema_t *semap) {
  *semap = &global_sema ;
  return HOT_OK;
}
hot_err_t hot_sema_Destroy(hot_sema_t sema) { return HOT_OK; }
hot_err_t hot_sema_Inc(hot_sema_t sema) { return HOT_OK; }
hot_err_t hot_sema_Dec(hot_sema_t sema) { return HOT_OK; }

#endif /* HOT_USE_THREADS */
