/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/* 
 * Test of Ensemble with the HOT C interface.
 *
 * Author:  Alexey Vaysburd, Dec. 96.
 *
 * NB: Access to the global struct is not protected with a mutex
 * (that could cause a deadlock). 
 */

#ifdef _WIN32
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include "hot_sys.h"
#include "hot_error.h"
#include "hot_msg.h"
#include "hot_ens.h"
#include "hot_thread.h"

#define HOT_TEST_MAGIC 234324

struct {
  int ncasts ;
  int nsends ;
  int nviews ;
  int nexits ;
  int nblocks ;
  int nheartbeats ;
  int njoins ;
} stats ;

typedef struct {
  int magic ;
  hot_gctx_t gctx;
  hot_lock_t mutex;
  unsigned seqno;
  int running;
  int thresh;
  int leaving;
  unsigned int next_sweep;
  unsigned int sweep;
  int exited ;
  hot_view_state_t vs;
  hot_ens_JoinOps_t jops;
} state ;

void scheck(state *s,hot_gctx_t g) {
  assert(s) ;
  assert(s->magic == HOT_TEST_MAGIC) ;
  assert(!s->exited) ;
  assert(s->gctx==g) ;
}

void trace(char *s) {
  /*
  printf("HOT_TEST:%s\n",s) ;
  */
}


void join(
        int thresh,
	char **argv,
	hot_ens_thread_mode thread_mode,
	int dncall_mode
) ;

void check_stats(void) {
  static int nevents = 0 ;
  if (nevents % 100 == 0) {
    printf ("HOT_TEST:stats:c=%d s=%d v=%d e=%d b=%d h=%d j=%d (total=%d)\n", 
	    stats.ncasts,
	    stats.nsends,
	    stats.nviews,
	    stats.nexits,
	    stats.nblocks,
	    stats.nheartbeats,
	    stats.njoins,
	    (stats.ncasts+stats.nsends+stats.nviews+stats.nexits+stats.nblocks+stats.nheartbeats+stats.njoins)
	   ) ;
  }
  nevents++ ;
}

/********************** Actions *******************************************/

typedef enum {
  LEAVE,
  CAST,
  SEND,
  REQUEST_NEW_VIEW,
  NO_OP
} action ;

action rand_action(int nmembers, int thresh) {
  double p = (rand() % 1000) / 1000.0 ;
  if (nmembers >= thresh && p < 0.03)
    return LEAVE ;
  else if (p < 0.5)
    return CAST ;
  else if (p <= 1.0)
    return SEND ;
  else 
    return NO_OP ;
}

/* Send a random number of multicast messages->
 */
void action_cast(state *s) {
  int i;
  char buf[128];
  int nmsgs;
  hot_err_t err;
  hot_msg_t msg;
  hot_uint32 u;
  trace("action:  cast");
  
  nmsgs = rand() % 5 ;
  
  for (i = 0; i < nmsgs; i++, s->seqno++) {
    msg = hot_msg_Create();
    
    sprintf(buf, "mcast<%d>", s->seqno);
    err = hot_msg_Write(msg, buf, strlen(buf));
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    
    u = s->seqno;
    err = hot_msg_WriteUint32(msg, &u);
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    
    err = hot_ens_Cast(s->gctx, msg, NULL);
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    
    err = hot_msg_Release(&msg);
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
}

/* Leave the group, wait for some time, and rejoin.
 */
void action_leave_and_rejoin(state *s) {
  hot_err_t err;
  
  trace("action: leave and rejoin");
  printf("HOT_TEST:leaving (nmembers=%d, rank=%d)\n",s->vs.nmembers,s->vs.rank);

  /* Send some cast messages first.
   */
  action_cast(s) ;

  /* Leave the group.
   */
  s->leaving = 1;
  err = hot_ens_Leave(s->gctx);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));

#ifdef NOT  
  trace("LEFT -- will rejoin now...");
  
  if (s->jops.dncall_mode == HOT_ENS_DNCALL_SYNC && s->leaving)
    hot_sys_Panic("hot_ens_Leave() returned before Exit callback");
  
  /* Join the group.
   */
  err = hot_ens_Join(&s->jops, &s->gctx);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  else
    trace("successfully rejoined the group");
#endif
  join(s->thresh,s->jops.argv,s->jops.thread_mode,s->jops.dncall_mode) ;

  if (s->jops.dncall_mode == HOT_ENS_DNCALL_SYNC) {
    if (s->vs.members != NULL) {
      free(s->vs.members);
    }
    memset(s,0,sizeof(*s)) ;
    s->exited = 1 ;
    free(s) ;
  }
}

/* Send a random number of p2p messages to a random view member.
 */
void action_send(state *s) {
  int i;
  char buf[128];
  hot_endpt_t dest;
  int rank;
  int nmsgs;
  hot_err_t err;
  hot_msg_t msg;
  
  trace("action:  send");
  
  rank = rand() % s->vs.nmembers;
  nmsgs = (rank == s->vs.rank) ? 0 : (rand() % 5);
  
  dest = s->vs.members[rank];
  
  /* If destination is me (b/c of the race condition when updating the
   * s struct during the view change), don't send anything.
   */
  if (strcmp(dest.name, 
	     s->vs.members[s->vs.rank].name) == 0)
    return;
  
  /*
    if (nmsgs) {
    printf("sender: %s\n", 
    s->vs.members[s->vs.my_rank].name);
    printf("dest: %s\n", dest.name);
    }
  */
    
  for (i = 0; i < nmsgs; i++) {
    msg = hot_msg_Create();
	
    sprintf(buf, "p2p<%d>", s->seqno++);
    err = hot_msg_Write(msg, buf, strlen(buf));
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    
    err = hot_ens_Send(s->gctx, &dest, msg, NULL);
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    
    err = hot_msg_Release(&msg);
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
}

/* Request a view change.
 */
void action_request_new_view(state *s) {
  hot_err_t err;
  
  trace("action:  request new view");
  
  /* View-change request will be ignored if our rank is not 0.
   */
  if (s->vs.rank == 0) {
    err = hot_ens_RequestNewView(s->gctx);
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
}

/********************** Callbacks *****************************************/

void receive_cast(
        hot_gctx_t gctx,
	void *env,
	hot_endpt_t *origin, 
	hot_msg_t msg
) {
  state *s = (state*) env ;
  char contents[128];
  hot_err_t err;
  unsigned pos;
  hot_uint32 u;

  scheck(s,gctx) ;

  err = hot_msg_ReadUint32(msg, &u);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  err = hot_msg_GetPos(msg, &pos);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  err = hot_msg_Read(msg, contents, pos);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  contents[pos] = 0;
  
  stats.ncasts++ ;
  check_stats() ;
  /*printf("mcast [%d]: '%s' from %s\n", (int)u, contents, origin->name);*/
}

void receive_send(
        hot_gctx_t gctx,
	void *env,
	hot_endpt_t *origin,
	hot_msg_t msg
) {
  state *s = (state*) env ;
  char contents[128];
  hot_err_t err;
  unsigned pos;
 
  scheck(s,gctx) ;

  err = hot_msg_GetPos(msg, &pos);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  err = hot_msg_Read(msg, contents, pos);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  contents[pos] = 0;
  stats.nsends++ ;
  check_stats() ;
  /*printf("pt2pt: '%s' from %s\n", contents, origin->name);*/
}

/* We have accepted a new view.  The new view state is specified.
 */
void accepted_view(
        hot_gctx_t gctx,
	void *env,
	hot_view_state_t *view_state
) {
  state *s = (state*) env ;
  int size ;
  scheck(s,gctx) ;

  /* Release the old view.
   */
  if (s->vs.members != NULL) {
    free(s->vs.members) ;
  }
  s->vs = *view_state; 
  size = sizeof(s->vs.members[0])*s->vs.nmembers ;
  s->vs.members = (hot_endpt_t*) malloc(size) ;
  memcpy(s->vs.members,view_state->members,size) ;

  if (s->vs.rank == 0)
    printf("HOT_TEST:view (nmembers=%d, rank=%d)\n",s->vs.nmembers,s->vs.rank);

/*
  printf("\tview_id = (%d,%s)\n", view_state->view_id.ltime,
	 view_state->view_id.coord.name);
  printf("\tversion = \"%s\"\n", view_state->version);
  printf("\tgroup_name = \"%s\"\n", view_state->group_name);
  printf("\tprotocol = \"%s\"\n", view_state->protocol);
  printf("\tmy_rank = %d\n", 	view_state->my_rank);
  printf("\tgroup daemon is %s\n", view_state->groupd ? "in use" : "not in use");
  printf("\tparameters = \"%s\"\n", view_state->params);
  printf("\txfer_view = %d\n", view_state->xfer_view);
  printf("\tprimary = %d\n", view_state->primary);
  
  for (i = 0; i < view_state->nmembers; i++)
    printf("%s\n", view_state->members[i].name);
*/
  s->running = 1;

  stats.nviews++ ;
  check_stats() ;
}

/* A periodic heartbeat event has occurred.  The 'time' argument 
 * is an approximation of the current time.
 */
void heartbeat(
        hot_gctx_t gctx,
	void *env,
	unsigned time
) {
  state *s = (state*) env ;
  scheck(s,gctx) ;

  if (time >= s->next_sweep) {
    stats.nheartbeats++ ;
    check_stats() ;

    s->next_sweep = time + s->sweep ;
    trace("heartbeat") ;
    switch (rand_action(s->vs.nmembers,s->thresh)) { 
    case LEAVE:
      hot_thread_Create((void*)action_leave_and_rejoin,s,NULL) ;
      /* action_leave_and_rejoin(s); */
      break;
    case CAST:
      action_cast(s);
      break;
    case SEND:
      action_send(s);
      break;
    case REQUEST_NEW_VIEW:
      action_request_new_view(s);
      break;
    case NO_OP:
      break;
    default:
      assert(0) ;
      break;
    }
  }    
}

void exit_cb(
        hot_gctx_t gctx,
        void *env
) {
  state *s = (state*) env ;
  scheck(s,gctx) ;
  trace("EXIT");

  stats.nexits++ ;
  check_stats() ;
  
  if (s->jops.dncall_mode == HOT_ENS_DNCALL_ASYNC) {
    if (s->vs.members != NULL) {
      free(s->vs.members);
    }
    memset(s,0,sizeof(*s)) ;
    s->exited = 1 ;
    free(s) ;
  }
}

void block(
        hot_gctx_t gctx,
	void *env
) {
  state *s = (state*) env ;
  scheck(s,gctx) ;
  trace("BLOCK");

  stats.nblocks++ ;
  check_stats() ;
}

void join(
        int thresh,
	char **argv,
	hot_ens_thread_mode thread_mode,
	int dncall_mode
) {
  hot_err_t err ;
  state *s ;
  s = (state *) malloc(sizeof(*s)) ;
  memset(s,0,sizeof(*s)) ;
  
  s->thresh = thresh ;
  s->magic = HOT_TEST_MAGIC ;
  s->jops.heartbeat_rate = 3.0 ;
  s->sweep = 3000 ;
  s->next_sweep = 0 ;
  strcpy(s->jops.transports, "UDP");
  strcpy(s->jops.group_name, "HOT_test");
  
  s->jops.argv = argv;
  
  strcpy(s->jops.protocol, "") ; 
  strcpy(s->jops.properties, "Gmp:Sync:Heal:Frag:Suspect");
  s->jops.use_properties = 1;
  strcpy(s->jops.params, "suspect_max_idle=3:int;suspect_sweep=1.000:time");
  s->jops.groupd = 0;
  
  s->jops.conf.receive_cast = receive_cast;
  s->jops.conf.receive_send = receive_send;
  s->jops.conf.accepted_view = accepted_view;
  s->jops.conf.heartbeat = heartbeat;
  s->jops.conf.exit = exit_cb;
  s->jops.conf.block = block;
  s->jops.debug = 0 ;
  
  s->jops.env = s;
  s->jops.dncall_mode = dncall_mode ;
  s->jops.thread_mode = thread_mode ;
  
  /* Join the group.
   */
  err = hot_ens_Join(&s->jops, &s->gctx);
  if (err != HOT_OK) {
    hot_sys_Panic(hot_err_ErrString(err));
  }

  stats.njoins++ ;
  check_stats() ;
}

int main(int argc, char *argv[]) {
  hot_sema_t sema ;
  hot_err_t err ;
  int thresh = 5 ;
  int nmembers = 7 ;
/*
  int thread_mode = HOT_ENS_THREAD_SINGLE ;
  int dncall_mode = HOT_ENS_DNCALL_ASYNC ;
*/
  int thread_mode = HOT_ENS_THREAD_MULTI ;
  int dncall_mode = HOT_ENS_DNCALL_SYNC ;
  int i ;
  
  trace("CDEMO: starting");
  
  /* Initialize state data.
   */
  srand(time(NULL));

  for (i=0;i<nmembers;i++)
    join(thresh,argv,thread_mode,dncall_mode) ;

  if (thread_mode == HOT_ENS_THREAD_MULTI) {
    hot_sema_Create(0,&sema) ;
    hot_sema_Dec(sema) ;
  } else {
    err = hot_ens_Start(argv) ;
    if (err != HOT_OK)
      hot_err_Panic(err,"hot_test:main");
  }
  return 0 ;
}
