(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(*****************************************************************************)
(* HOT_ML.ML: ML side of the Ensemble C application interface *)
(* Author:  Alexey Vaysburd, 11/96 *)
(* init_view_state and capp_main_loop are based on code by Mark Hayden *)
(* Bug Fixes, Cleanup: Mark Hayden, 4/97 *)
(* More Bug Fixes, Cleanup: Mark Hayden, 5/97 *)
(*****************************************************************************)
open Util
open Appl_intf
open View
open Gctx
(*****************************************************************************)
let name = Trace.source_file "HOT_ML"
let failwith s = failwith (name^":"^s)
(**************************************************************)

(* View state structure passed to the application. 
 *)
type ens_view_state = {
  c_version	: string ;
  c_group	: string ;
  c_view	: endpt array ; 
  c_rank	: int ;
  c_proto_name : string ;
  c_groupd	: bool ;
  c_view_id	: Trans.ltime * endpt ;
  c_params	: string ;
  c_xfer_view	: bool ; 
  c_primary	: bool
}

type mainloop_status = 
  | Active
  | About_to_block

(**************************************************************)
(* External C functions.
 *)

external thread_yield : unit -> unit
        = "hot_ens_thread_yield"

external usleep : int -> unit 
        = "hot_ens_thread_usleep"

external cback_recv_cast: id -> endpt -> msg -> unit 
	= "hot_ens_ReceiveCast_cbd"

external cback_recv_send: id -> endpt -> msg -> unit 
	= "hot_ens_ReceiveSend_cbd"

external cback_accepted_view: id -> ens_view_state -> unit 
	= "hot_ens_AcceptedView_cbd"

external cback_heartbeat: id -> Time.t -> unit
	= "hot_ens_Heartbeat_cbd"

external cback_block: id -> unit 
	= "hot_ens_Block_cbd"

external cback_exit: id -> unit
	= "hot_ens_Exit_cbd"

(**************************************************************)

(* Return true iff there are pending downcalls enqueued by
 * the C application *and* an alarm to process them has not
 * yet been scheduled.
 *
 * If going to return false and mainloop_status is
 * About_to_block, set a blocking flag (in C app.) to let
 * the app. know Ensemble might be blocked.  If
 * mainloop_status is Active, clear the blocking flag
 * regardless of the return value.
 *
 * NB: At the C app. side, need to write to the pipe to get
 * attention of Ens.  only when Ens-blocked flag is set
 * *and* a processing alarm has not been requested/scheduled
 * *and* the group is not blocked.  (NB: after C app. writes
 * to the pipe, it knows an alarm will be scheduled).  
 *)

external c_dncalls_pending: mainloop_status -> bool = "hot_ens_DncallsPending"

(* Open a pipe for C-OCAML communication;  return the read fd. 
 *)
external c_open_app_pipe: unit -> Hsys.socket = "hot_ens_OpenAppPipe"

(* Retrieve pending downcalls from C application.
 * NB: Don't forget to clear the "alarm-is-scheduled" flag in C app.
 *)
external c_get_dncalls: unit -> c_dncall array = "hot_ens_GetDncalls"

(*****************************************************************************)

let appl_of_msg m = 
  Iovec.of_string name m

let msg_of_appl m = 
  Iovec.to_string name m

let empty () = Iovec.empty name

let converter = Appl_intf.iov

(*****************************************************************************)
(*
let appl_of_msg = ident
let msg_of_appl = ident
let empty () = ()
let converter = Appl_intf.full
*)
(*****************************************************************************)

(* Convert a param string into a param list.
 * Format of the param string: "name=val:type;...." 
 * (see Param.t for supported types).  The type of param list is Param.tl.
 *) 
let param_l_of_str s = 
  if String.length s = 0 then []
  else
    let l = Util.string_split ";" s in
    let param_of_str tok = 
      let l = Util.string_split "=" tok in
      let param_name = List.hd l in
      let val_type_name_l = Util.string_split ":" (List.nth l 1) in
      let param_val l = 
     	let param_type = List.nth l 1
     	and param_str = List.hd l
     	in
     	match param_type with
     	| "string" -> Param.String(param_str)
     	| "int"    -> Param.Int(int_of_string(param_str))
     	| "float"  -> Param.Float(float_of_string(param_str))
     	| "bool"   -> Param.Bool(bool_of_string(param_str))
     	| "time"   -> Param.Time(Time.of_float(float_of_string(param_str)))
     	| _ -> failwith "param_val: bad type"
      in
      (param_name, param_val val_type_name_l)
    in
    List.map param_of_str l
   
(* Convert a param list into a param string.
 *)
let str_of_param_l p = 
  let p =
    List.map (fun (n,p) ->
      let v,t =
        match p with
        | Param.Time(t)   -> (string_of_float (Time.to_float t) , "time")
        | Param.Int(i)    -> (string_of_int i, "int")
        | Param.Bool(b)   -> (string_of_bool b, "bool")
        | Param.String(s) -> (s, "string")
        | Param.Float(f)  -> (string_of_float(f), "float")
      in
      sprintf "%s=%s:%s" n v t
    ) p
  in
  String.concat ";" p

(**************************************************************)

(* Convert a view state into a structure in the C application format.
 *)
let c_view_state (ls,vs) = {
  c_version = Version.string_of_id vs.version;
  c_group = Group.string_of_id vs.group;
  c_view = Array.map Endpt.string_of_id vs.view;
  c_rank = ls.rank;
  c_proto_name = Proto.string_of_id vs.proto_id;
  c_groupd = vs.groupd;
  c_view_id = (
    match vs.view_id with 
    (ltime,eid) -> (ltime,Endpt.string_of_id eid)
  );
  c_params = str_of_param_l vs.params;
  c_xfer_view = vs.xfer_view;
  c_primary = vs.primary
}

(*****************************************************************************)

(* Initialize a group member and return the initial singleton view 
 *)
let init_view_state jops = 
  let groupd = jops.jops_groupd in
  let setup_protocol () = 
    if jops.jops_use_properties then (
      let properties = 
	let props_l = Util.string_split ":" jops.jops_properties in
	let props_l = List.map Property.id_of_string props_l in
      	if groupd then
	  Property.strip_groupd props_l
	else 
	  props_l
      in
      Property.choose properties
    ) else (
      eprintf "HOT_ML:warning:using a raw protocol stack\n" ;
      Proto.id_of_string jops.jops_protocol
    )
  in      
  let group_name = jops.jops_group_name in
  let protocol = setup_protocol () in
  let transports = jops.jops_transports in
  let transp_list = string_split ":" transports in
  let modes = List.map Addr.id_of_string transp_list in
  let key = match Arge.get Arge.key with
  | None -> Security.NoKey
  | Some s -> Security.Common s
  in
  let params = jops.jops_params in
    (* Alarm.choose (Arge.get Arge.alarm) ; *) (* BUG: unnecessary? *)
  let (ls,vs) =
    Appl.full_info group_name groupd protocol modes key
  in
  
  let vs = View.set vs [Vs_params (param_l_of_str params)] in
  (ls,vs)
(* init_view_state *)

(* Join a goup; use specified group context record. *)
let join jops gctx dispatch_dncalls =
  (* Initialize the state.
   *)
  let (ls_init,vs_init) = init_view_state jops in
  
  let s = {
    id = gctx ;
    vs = vs_init ;
    ls = ls_init ;
    dncalls = Queuee.create () ;
    hb_requested = false ;
    sview = [||];
    exited = false
  } in
  
  (* Register a function to be invoked in order to request
   * a heartbeat.  
   *)
  let async = Appl.async (vs_init.group,ls_init.endpt) in
  
  Gctx.add s.id (fun dncall ->
    Queuee.add dncall s.dncalls ;
    if not s.hb_requested then (
      s.hb_requested <- true ;
      async ()
    )
  );

  (* Return pending actions for the given gctx.
   *
   * For send actions, need to convert endp_uid into the rank
   * in the current view.  If there is no member in the
   * current view with the given eid, drop the action.
   *

   * For suspect actions, if a suspected endp is not in the
   * current view, don't include it in the suspect action.  
   *)
  let pending_actions () =
    dispatch_dncalls () ;
    let action_list = ref [] in
    while not (Queuee.empty s.dncalls) do
      match Queuee.take s.dncalls with
      | C_Leave () -> 
	  (*eprintf "**** HOT_ML:  gctx = %d is LEAVING\n" gctx;*)
	  (* The leave downcall blocks the member. 
	   *)
	  action_list := !action_list @ [Leave]
      | C_Cast(msg) -> 
	  let msg = appl_of_msg msg in
	  action_list := !action_list @ [Cast msg]
      | C_Send(endpt, msg) -> 
	  let msg = appl_of_msg msg in
	  begin
	    try 
	      let rank = array_index endpt s.sview in
	      if rank = s.ls.rank then
		eprintf "HOT_ML:warning:Send to myself" ;
	      action_list := !action_list @ [Send([rank],msg)]
	    with Not_found -> 
	      eprintf "HOT_ML:warning:Send:unknown destination\n"
	  end		 
  (*
      | C_Suspect(endpts) -> 
	  let suspects = ref [] in
	  let add_suspect endpt =
	    let rank = rank_in_gctx endpt gctx in
	    if rank >= 0 then
	      suspects := !suspects @ [rank]
	  in
	  Array.iter add_suspect endpts;
	  action_list := !action_list @ [Suspect(!suspects)];
  *)
      | C_Protocol(protocol) ->
	  action_list := !action_list @ [Protocol(Proto.id_of_string protocol)]
      | C_Properties(properties) ->
	  let protocol =
	    let props = 
	      let props_l = Util.string_split ":" properties in
	      let props_l = List.map Property.id_of_string props_l in
	      if s.vs.groupd then
		Property.strip_groupd props_l
	      else 
		props_l
	    in
	    Property.choose props
	  in
	  action_list := !action_list @ [Protocol(protocol)]
      |	C_Prompt() ->
	  action_list := !action_list @ [Prompt]
      |	C_Void() ->
	  failwith "Void downcall"
      | _ -> failwith "bad dncall type"
    done ;
    !action_list
    (* pending_actions *)
  in

  let check () =
    if s.exited then
      failwith "got upcall after exit callback"
  in

  (* Return callback configuration for the given gctx and
   * heartbeat rate.
   *
   * When defining application configuration for Ensemble,
   * we have to define all callback functions.  Typically a
   * callback function will convert its arguments into
   * arguments for the corresp. C upcall; invoke the upcall;
   * retrieve (from C app.) all enqueued requests (if any),
   * convert them into actions, and add to the actions list;
   * return the action list for the gctx (if appropriate for
   * this callback type).  
   *)
  let conf (ls,vs) hrtbt_rate =
    let id_of_rank rank = 
      if rank < 0 || rank >= Array.length s.sview then (
	eprintf "Array size = %d, origin rank = %d\n" (Array.length s.sview) rank ;
	failwith "id_of_rank:bad rank"
      ) else (
        s.sview.(rank)
      )
    in

    (* Got a multicast message.  Invoke C app's callback
     * dispatcher; return a list of pending actions.  
     *)
    let app_recv_cast origin msg = 
      check () ;
      let msg = msg_of_appl msg in
      cback_recv_cast gctx (id_of_rank origin) msg;
      pending_actions ()
    in

    (* Got a point-to-point message.  Invoke C app's
     * callback dispatcher; return a list of pending
     * actions.  
     *)
    let app_recv_send origin msg = 
      check () ;
      let msg = msg_of_appl msg in
      cback_recv_send gctx (id_of_rank origin) msg;
      pending_actions ()
    in

    (* Got a heartbeat event.  Invoke C app's callback
     * dispatcher; return a list of pending actions.  
     *)
    let app_heartbeat time = 
      check () ;
      cback_heartbeat gctx time ;
      s.hb_requested <- false ;
      pending_actions ()
    in

    (* The group is about to block.  Tell the application it
     * should not request Ensemble's attention until the
     * next view is installed.  Return the list of pending
     * actions.  
     *)
    let app_block () =
      check () ;
      cback_block gctx ;
      pending_actions ()
    in

    (* Got a multicast in the blocked state.  Invoke C app's
     * callback dispatcher.  
     *)
    let app_block_recv_cast origin msg =
      check () ;
      let msg = msg_of_appl msg in
      cback_recv_cast gctx (id_of_rank origin) msg
    in

    (* Got a point-to-point message in the blocked state.  
     * Invoke C app's callback dispatcher.
     *)
    let app_block_recv_send origin msg = 
      check () ;
      let msg = msg_of_appl msg in
      cback_recv_send gctx (id_of_rank origin) msg
    in

    (* The member can send it state to the coordinator with
     * this callback before the coordinator proposes the new
     * view.  
     *)
    let app_block_view (ls,vs) = 
      check () ;
      if ls.rank = 0 then
      	List.map (fun rank -> 
	  (rank,(empty ()))
	) (Util.sequence ls.nmembers)
      else []
    in

    (* We are about to propose a new view.
     *)
    let app_block_install_view (ls,vs) _ = 
      check () ;
      empty ()
    in

    (* A new view has been accepted. Invoke C app's callback
     * dispatcher; return a list of pending actions.  
     *)
    let app_unblock_view (ls,vs) _ =
      check () ;
      let view = vs.view in
      let view = Array.map Endpt.string_of_id view in
      s.sview <- view ;
      s.ls <- ls ;
      s.vs <- vs ;
      let cvs = c_view_state (ls,vs) in
      cback_accepted_view gctx cvs ;
      pending_actions ()
    in

    (* We have left the group.  Notify the C application.
     * The gctx is invalidated.
     *)
    let app_exit () =
      check () ;
      s.exited <- true ;
      (*eprintf "*** HOT_ML:  cback_exit, gctx = %d\n" gctx;*)
      cback_exit gctx ;
      Gctx.destroy "app_exit" gctx
    in
    
    let interface = {
      recv_cast 		= app_recv_cast ;
      recv_send			= app_recv_send ;
      heartbeat 		= app_heartbeat ;
      heartbeat_rate 		= Time.of_float hrtbt_rate;
      block 			= app_block ;
      block_recv_cast		= app_block_recv_cast ;
      block_recv_send		= app_block_recv_send ;
      block_view                = app_block_view ;
      block_install_view	= app_block_install_view ;
      unblock_view		= app_unblock_view ;
      exit			= app_exit
    } in
    
    let interface = 
      if jops.jops_debug then
 	Appl_intf.debug interface
      else 
	interface
    in

    converter interface
  in

  let view_state = init_view_state jops in
  let interface = conf view_state jops.jops_hrtbt_rate in
  Appl.config interface view_state


  
(* Get pending dncalls from C application and dispatch among resp. groups. 
 *)
let rec dispatch_dncalls () = 
  let dn_arr = c_get_dncalls () in 
  Array.iter (function {gid = gctx; dncall = req} ->
    (* Dispatch a downcall request to the corresp. group context.
     *)
    match req with
    | C_Join(ops) -> join ops gctx dispatch_dncalls
    | _ ->
        (* Add req to the list of pending dncalls for the given gctx.
         *)
 	get gctx req 
  ) dn_arr

(****************************************************************************)

(* If there are any application downcalls pending, schedule an alarm
 * to process them.  Return true iff there are new pending dncalls. 
 *)
let dncalls_pending mainloop_status = 
  if c_dncalls_pending mainloop_status then (
    dispatch_dncalls () ;
    true
  ) else (
    false
  )

(****************************************************************************)

(* Loop forever processing Ensemble events.
 *)
let main_loop () =
  let sched_step = Sched.step Appl.root_sched in
  let count = 200 in
  let alarm = Alarm.get () in
  
  let event_loop () =
    while true do
      (* Check for messages.
       *)
      let got_msgs = Alarm.poll alarm Alarm.SocksPolls in
      
      (* Check for application downcalls.  If there are pending downcalls,
       * an alarm is scheduled (if it hasn't been done already).
       *)
      let got_dncalls = dncalls_pending Active in
      
      (* Schedule some events in the layers.
       *)
      let got_events = sched_step count in

      (* Yield to other threads.
       *)
(*
      thread_yield() ;
*)

      (* If nothing happened, prepare to block
       *)
      if not (Alarm.check alarm || got_msgs || got_events || got_dncalls) then (

  	(* Notify the c-stub of our block attempt
	 *)
  	if not (dncalls_pending About_to_block) then (
	  Alarm.block alarm ;
	)
      )
    done;
  in Printexc.catch event_loop ()

(****************************************************************************)

(* When C application starts Ensemble/OCAML, go into main loop right away.
 *) 
let _ =
  Arge.parse [
    (*
     * Extra arguments can go here.
     *)
  ] (Arge.badarg name) "libhot:Ensemble HOT tools library" ;

  let pipe_fd = ref (c_open_app_pipe ()) in

  let dispatch_dncalls_pipe () =
    let buf = String.create 1 in 
    let len = Hsys.recv !pipe_fd buf 0 1 in
    if len = 0 then 
      failwith "dispatch_dncalls_pipe: recv" ;
    dispatch_dncalls () ;
  in

  (* Install this in the Real alarm so that it calls usleep
   * rather than select().  A bunch of threads packages don't
   * implement select() properly.  Windows NT is not a problem
   * here, however.
   *)
  begin
    match Sys.os_type with
    | "Win32" -> ()
    | _ ->
	Real.install_blocker (fun timeout ->
	  let timeout = timeout *. 1000000.0 in
	  let timeout = truncate timeout in
	  let timeout = min timeout 100000 in
	  usleep timeout
	)
  end ;

  (* Install an alarm on the read end of the pipe for communication 
   * with the C application.  C app. write a byte to the pipe whenever
   * it wants to notify Ensemble of pending downcall requests.
   *)
  Alarm.choose (Alarm.id_of_string "REAL") Appl.root_sched ;
  let alarm = Alarm.get () in
  Printexc.print 
    (Alarm.add_sock alarm !pipe_fd) 
    (Hsys.Handler0 dispatch_dncalls_pipe) ;

  Appl.start_monitor () ;
  main_loop ()

(*****************************************************************************)
