/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/*
 * Contents:  Error interface for HOT (implementation)
 *
 * Author:  Alexey Vayeburd, November 1996
 *
 */

#define HOT_ERROR_NO_DEBUG_WRAPPERS
#include <assert.h>
#include "hot_sys.h"
#include "hot_error.h"

/* An error object:  specified are the error code and the reason.
 */ 
struct hot_err {
  char *reason;
  unsigned code;
};

/* Create an error object with specified code and reason
 */ 
hot_err_t 
hot_err_Create(unsigned err_code, char *err_string) {
  struct hot_err *err = (struct hot_err*) malloc(sizeof(struct hot_err));
  if (err == NULL)
    hot_sys_Panic("out of memory");
  
  if (err_string == NULL)
    err_string = "unspecified error"; 

  err->reason = (char*) malloc(strlen(err_string) + 1);
  if (err->reason == NULL)
    hot_sys_Panic("out of memory");

  strcpy(err->reason, err_string);
  err->code = err_code;

  return (hot_err_t) err;
}

/* Release an error object
 */
void hot_err_Release(hot_err_t err) {
  struct hot_err *errp = (struct hot_err*) err;
  free(errp->reason);
  free(errp);
}

/* Return the reason of an error
 */
char* hot_err_ErrString(hot_err_t err) {
  struct hot_err *errp = (struct hot_err*) err;
  return errp->reason;
}

/* Return the code of an error
 */
unsigned hot_err_ErrCode(hot_err_t err) {
  struct hot_err *errp = (struct hot_err*) err;
  return errp->code;
}

void hot_err_Panic(hot_err_t err, char *msg2) {
  char *msg1 = hot_err_ErrString(err) ;
  char *comb ;
  
  comb = malloc(sizeof(char) * (strlen(msg1) + strlen(msg2) + 2)) ;
  if (comb == NULL)
    hot_sys_Panic("hot_err_Panic:malloc returned NULL") ;
  
  sprintf(comb,"%s:%s",msg1,msg2) ;
  assert(strlen(comb) == strlen(msg1) + strlen(msg2) + 1) ;
  hot_sys_Panic(comb) ;
}

/********************** memory-leakage debugging *************************/

static unsigned long n_mallocs, n_frees;

struct hot_debug_mem_stats {
    char file[128];
    unsigned line;
    unsigned ntimes;
};

#define HOT_MAX_MALLOC_PLACES 1000

static int n_malloc_places;
static int n_free_places;
struct hot_debug_mem_stats hot_mallocs[HOT_MAX_MALLOC_PLACES];
struct hot_debug_mem_stats hot_frees[HOT_MAX_MALLOC_PLACES];

static int mcount;

static void hot_debug_mem_AddMalloc(char *file, unsigned line) {
    int i;
    for (i = 0; i < n_malloc_places; i++) {
	if ((strcmp(hot_mallocs[i].file, file) == 0) && 
	    (hot_mallocs[i].line == line)) {
	    hot_mallocs[i].ntimes++;
	    break;
	}
    }
    if (i == n_malloc_places) {
	if (n_malloc_places >= HOT_MAX_MALLOC_PLACES) {
	    hot_sys_Panic("hot_debug_mem_AddMalloc: too many malloc places");
	}
	strcpy(hot_mallocs[i].file, file);
	hot_mallocs[i].line = line;
	hot_mallocs[i].ntimes = 1;
	n_malloc_places++;
    }
}

static void hot_debug_mem_AddFree(char *file, unsigned line) {
    int i;
    for (i = 0; i < n_free_places; i++) {
	if ((strcmp(hot_frees[i].file, file) == 0) && 
	    (hot_frees[i].line == line)) {
	    hot_frees[i].ntimes++;
	    break;
	}
    }
    if (i == n_free_places) {
	if (n_free_places >= HOT_MAX_MALLOC_PLACES) {
	    hot_sys_Panic("hot_debug_mem_AddFree: too many free places");
	}
	strcpy(hot_frees[i].file, file);
	hot_frees[i].line = line;
	hot_frees[i].ntimes = 1;
	n_free_places++;
    }
}

static void hot_debug_mem_PrintStats() {
    int i;

    printf("\n************** Memory allocation statistics ***************\n");

    for (i = 0; i < n_malloc_places; i++) {
	printf("MALLOC: file: %16s, line: %4d, nmallocs: %4d\n",
	       hot_mallocs[i].file, hot_mallocs[i].line, hot_mallocs[i].ntimes);
    }

    for (i = 0; i < n_free_places; i++) {
	printf("FREE:   file: %16s, line: %4d, nfrees:   %4d\n",
	       hot_frees[i].file, hot_frees[i].line, hot_frees[i].ntimes);
    }

    printf("************************************************************\n");
}

char *hot_debug_mem_Malloc(unsigned size, char *file, unsigned line) {
    ++n_mallocs;
    hot_debug_mem_AddMalloc(file, line);
    if (++mcount == 100) {
	hot_debug_mem_PrintStats();
	mcount = 0;
    }
    return malloc(size);
}

void hot_debug_mem_Free(void *ptr, char *file, unsigned line) {
    ++n_frees;
    hot_debug_mem_AddFree(file, line);
    free(ptr);
}
