/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/*
 * Contents: C side of the HOT C interface to Ensemble.
 *
 * Author:  Alexey Vaysburd, Dec. 1996.
 */ 

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef WIN32
void Perror(char *s);
#include <winsock2.h>
#else
#define Perror(s) perror(s)
#include <sys/socket.h>
#include <unistd.h>
#endif

#include "hot_sys.h"
#include "hot_error.h"
#include "hot_thread.h"
#include "hot_msg.h"
#include "hot_mem.h"
#include "hot_ens.h"

/****************************************************************************/

value hot_ml_alloc(char *debug,mlsize_t s, tag_t t) {
  /*
  fprintf(stderr,"hot_ml_alloc:%s:%d:%d\n", debug, (int)s, (int)t) ;
  */
  assert(s >= 0) ;
  if (s == 0) {
    return Atom(0) ;
  } else if (s < Max_young_wosize) {
    return alloc(s,t) ;
  } else {
    return alloc_shr(s,t) ;
  }
}  

value hot_ml_alloc_tuple(char *debug, mlsize_t s) {
  return hot_ml_alloc(debug,s,0) ;
}

value hot_ml_copy_string(char *s) {
  int len;
  value res;
  
  len = strlen(s);
  res = alloc_string(len);
  bcopy(s, String_val(res), len);
  return res;
}

/****************************************************************************/

/******************* Threads-interface configurations ************************/

static hot_err_t noop_thread_Create(
        void (*routine)(void*), 
	void *arg,
	hot_thread_attr_t *attr
) {
  (*routine)(arg);
  return HOT_OK;
}
static hot_err_t noop_thread_Yield(void) { return HOT_OK; }
static hot_err_t noop_thread_Usleep(int usecs) { return HOT_OK; }
static hot_err_t noop_lck_Create(hot_lock_t *l) { return HOT_OK; }
static hot_err_t noop_lck_Destroy(hot_lock_t l) { return HOT_OK; }
static hot_err_t noop_lck_Lock(hot_lock_t l) { return HOT_OK; }
static hot_err_t noop_lck_Unlock(hot_lock_t l) { return HOT_OK; }
static hot_err_t noop_sema_Create(int initial_value, hot_sema_t *semap) { return HOT_OK; } 
static hot_err_t noop_sema_Destroy(hot_sema_t sema) { return HOT_OK; }
static hot_err_t noop_sema_Inc(hot_sema_t sema) { return HOT_OK; }
static hot_err_t noop_sema_Dec(hot_sema_t sema) { return HOT_OK; }

/* No-op threads configuration -- for single-threaded execution mode.
 */ 
static hot_thread_conf_t hot_noop_thread_conf = {
  noop_thread_Create,
  noop_thread_Yield,
  noop_thread_Usleep,
  noop_lck_Create,
  noop_lck_Destroy,
  noop_lck_Lock,
  noop_lck_Unlock,
  noop_sema_Create,
  noop_sema_Destroy,
  noop_sema_Inc,
  noop_sema_Dec
} ;
  
/* Standard threads configuration -- for multi-threaded execution.
 */
static hot_thread_conf_t hot_std_thread_conf = {
  hot_thread_Create,
  hot_thread_Yield,
  hot_thread_Usleep,
  hot_lck_Create,
  hot_lck_Destroy,
  hot_lck_Lock,
  hot_lck_Unlock,
  hot_sema_Create,
  hot_sema_Destroy,
  hot_sema_Inc,
  hot_sema_Dec
} ;

/****************************************************************************/

/* This describes a group context.
 */
struct hot_gctx {
  unsigned int id ;		/* unique integer identifier */
  hot_ens_cbacks_t conf;	/* application callbacks */
  void *env;			/* application state */
  int group_blocked;		/* is group blocked? */
  int joining ;			/* are we joining? */
  int leaving ;			/* are we leaving? */
  
  /* Various semaphores.
   */
  hot_sema_t join_sema ;
  hot_sema_t leave_sema ;

  hot_view_state_t vs ;		/* view state */
  hot_ens_dncall_mode dncall_mode; /* sync/async downcalls (joins/leaves) */
} ;

/****************************************************************************/

/* Global state.
 */
static struct {
  hot_lock_t mutex;		/* global mutex lock  */
  int in_critical;		/* Set when mutex is locked */
  int initialized;		/* Set when hot_c is initialized. */
  int started;			/* Set when Ensemble/OCAML is started */
  FILE *out ;
  FILE *in ;
} global ;

/****************************************************************************/

/* Startup options.
 */
typedef enum {
  HOT_ENS_INIT_DONT_START,
  HOT_ENS_INIT_SAME_THREAD,
  HOT_ENS_INIT_SEPARATE_THREAD
} hot_ens_init_ops ;

/****************************************************************************/

static inline void begin_critical(void) {
  hot_err_t err ;
  if ((err = (*global.thr->lck_lock)(global.mutex)) != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err)) ;
  assert(!global.in_critical) ;
  global.in_critical = 1 ;
}

static inline void end_critical(void) {
  hot_err_t err ;
  assert(global.in_critical) ;
  global.in_critical = 0 ;
  if ((err = (*global.thr->lck_unlock)(global.mutex)) != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
}

/****************************************************************************/

char *hot_ens_argv_null[2] = {"(?)", NULL};

/* Start OCAML.  The function does not return.
 */
static void ens_do_init(void *param) {
  static int called_already = 0 ;
  if (called_already)
    hot_sys_Panic("caml_startup:called twice") ;
  called_already = 1 ;

  caml_startup((char**) param ? param : hot_ens_argv_null);
}

/* Initialize the interface. Start Ensemble/OCAML if necessary.
 */ 
static hot_err_t hot_ens_Init(char **argv, hot_ens_init_ops init_ops) {
    int i;
    hot_err_t err;
    hot_thread_attr_t attr;
    int size ;

#ifdef _WIN32_PRECOMPILED
    if (init_ops != HOT_ENS_INIT_SEPARATE_THREAD) {
    	fprintf(stderr, "This pre-compiled WIN32 version of HOT only supports the SEPARATE_THREAD options\n");
    	return hot_err_Create(0, "hot_ens_Init:  unsupported init option");;
    }
    global.initialized = 1;
    global.thread_mode = HOT_ENS_THREAD_MULTI;
    global.thr = &hot_std_thread_conf;
#else
    if (global.initialized) {
      switch (init_ops) {
      case HOT_ENS_INIT_SAME_THREAD:
	/* If hot_c has already been initialized, start Ensemble now.
	 */
	begin_critical();
	if (global.started) {
	  end_critical();
	  return HOT_OK;
	}
	global.started = 1;
	end_critical();
	return (*global.thr->thread_create)(ens_do_init, (void*) argv, NULL);
	
      case HOT_ENS_INIT_DONT_START:   	   /* FALL THROUGH */
      case HOT_ENS_INIT_SEPARATE_THREAD:
	return HOT_OK;
	
      default:
	hot_sys_Panic("hot_ens_Init:  unknown init option");
	break;
      }
    }
    global.initialized = 1;

    /* Choose a threads configuration.  For single-threaded mode, use no-op.
     */
    switch (init_ops) {
    case HOT_ENS_INIT_SEPARATE_THREAD:
      global.thread_mode = HOT_ENS_THREAD_MULTI;
      global.thr = &hot_std_thread_conf;
      break;
    case HOT_ENS_INIT_DONT_START:
      global.thread_mode = HOT_ENS_THREAD_SINGLE;
      global.thr = &hot_noop_thread_conf;
      break;
    default:
      return hot_err_Create(0, "hot_ens_Init:  unsupported init option");
      break;
    }
#endif

    /* Initialize C side of the interface (preallocate/initialize gctx entries,
     * dncall structures, mutex.
     */
    if ((err = (*global.thr->lck_create)(&global.mutex)) != HOT_OK)
	hot_sys_Panic(hot_err_ErrString(err));
    
    begin_critical();

    global.ensemble_blocked = 0;
    global.alarm_scheduled = 0;

    global.dn_free = NULL ;
    global.dn_tail = NULL ;

    global.g_alive = NULL ;
    
    /* Open a socketpair for communicating with Ensemble.
     */    
    if (socketpair(PF_UNIX,SOCK_STREAM,0,global.fd) < 0) {
	hot_sys_Panic("hot_ens_Init: socket() failed");
    }
    
    end_critical();

    switch (init_ops) {
    case HOT_ENS_INIT_SAME_THREAD:
      /* FALL THROUGH.
       * NB:  No-op thread_create is implemented as a direct invocation.
       */ 
    case HOT_ENS_INIT_SEPARATE_THREAD:
      /* Start Ensemble/OCAML.  Don't allow starting Ensemble more than once.
       */    
      begin_critical();
      if (global.started) {
	end_critical();
	return hot_err_Create(0, "hot_ens_Init:  Ensemble already started");
      }
      global.started = 1;
      end_critical();

      attr.stacksize = 1024 * 1024; /* stack size for Ensemble thread = 1M */
   
      if ((err = (*global.thr->thread_create)(ens_do_init, (void*) argv, &attr)) != HOT_OK)
	return err;
      break;

    case HOT_ENS_INIT_DONT_START:
      break;

    default:
      hot_sys_Panic("hot_ens_Init:  bad init option");
      break;
    }

    return HOT_OK;
}

/****************************************************************************/

static void release_endpt_array(
        channel_t *ch,
        hot_endpt_t *endpts,
	unsigned nendpts
) {
  int size = nendpts * sizeof(*endpts) ;
  assert(endpts) ;
  assert(nendpts > 0) ;

  hot_free(ch,size,endpts) ;
}

/****************************************************************************/

static 
hot_gctx_t alloc_gctx(void) {
  static unsigned int id = 0 ;
  int index;
  hot_gctx_t gctx ;
  hot_err_t err;
  
  gctx = hot_malloc(&ch_gctx,sizeof(*gctx)) ;

  gctx->id = id++ ;
  global.g_alive = gctx ;
  gctx->vs.members = NULL ;
  gctx->vs.nmembers = -1 ;
  
  return gctx ;
}

/* Release a gctx descriptor.
 * The global data record must be locked at this point.
 */
static 
void release_gctx(hot_gctx_t g) {
  hot_gctx_t *tmpp, tmp ;

  assert(g) ;

  /* Release the view information.
   */
  if ((g->vs.members == NULL) ||
      (g->vs.nmembers == 0))
    hot_sys_Panic("HOT:release_gctx: internal inconsistency");
  release_endpt_array(&ch_endpt_array, g->vs.members,g->vs.nmembers) ;
  g->vs.members = NULL ;
  g->vs.nmembers = -1 ;

  assert(!g->join_sema) ;
  assert(!g->leave_sema) ;
  assert(!g->vs.members) ;
  assert(g->vs.nmembers == -1) ;

  hot_free(&ch_gctx,sizeof(*g),g) ;
}

/* Return a HOT message created from the given msg value.
 */
static inline hot_msg_t hot_ens_Msg_of_val(value msg_v) {
  hot_msg_t msg;
  char *data;
  unsigned size;

  data = String_val(msg_v);
  size = string_length(msg_v);

#ifdef HOT_MSG_PADDING
  {
    char npad = data[0];
    msg = hot_msg_CreateDontCopy(data + npad, size - npad, NULL, NULL);
  }
#else
  msg = hot_msg_CreateDontCopy(data, size, NULL, NULL);
#endif

  return msg;
}

/* Return a msg value corresponding to the given HOT message.
 */
static inline value Val_msg(hot_msg_t msg) {
  value msg_v;
  void *data;
  unsigned size;
  hot_err_t err;
    
  err = hot_msg_GetPos(msg, &size);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  err = hot_msg_Look(msg, size, &data);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
#ifdef HOT_MSG_PADDING
  {
    char pad[HOT_MSG_PADDING];
    char npad = HOT_MSG_PADDING - (size % HOT_MSG_PADDING);
    
    pad[0] = npad;
    msg_v = alloc_string(size + npad);
    memcpy(String_val(msg_v), pad, npad);
    memcpy(String_val(msg_v) + npad, data, size);
  }
#else
  msg_v = alloc_string(size);
  memcpy(String_val(msg_v), data, size);
#endif

  return msg_v;
}


/* Exported interface:  
 */

/* Start Ensemble in the current thread.  
 * If successful, this call will block forever.
 */
hot_err_t hot_ens_Start(char **argv) {
  return hot_ens_Init(argv, HOT_ENS_INIT_SAME_THREAD);
}

/* Initialize/reset an options struct.
 */
void hot_ens_InitJoinOps(hot_ens_JoinOps_t *ops) {
  if (ops == NULL)
    hot_sys_Panic("hot_ens_InitJoinOps: bad argument (NULL)");
  
  memset(ops, 0, sizeof(hot_ens_JoinOps_t));
  ops->heartbeat_rate = HOT_ENS_DEFAULT_HEARTBEAT_RATE;
  strcpy(ops->transports, HOT_ENS_DEFAULT_TRANSPORT);
  strcpy(ops->group_name, HOT_ENS_DEFAULT_GROUP_NAME);
  strcpy(ops->protocol, HOT_ENS_DEFAULT_PROTOCOL);
  strcpy(ops->properties, HOT_ENS_DEFAULT_PROPERTIES);
  ops->use_properties = 1;
  strcpy(ops->params, "");
  ops->groupd = 0;
  ops->dncall_mode = HOT_ENS_DNCALL_SYNC;
  ops->thread_mode = HOT_ENS_THREAD_MULTI;
  ops->argv = NULL ;
  ops->env = NULL ;
}

/* Join a group.  In case of success, the call returns after the first view
 * is delivered.  The group context is returned in *gctxp.
 */
hot_err_t hot_ens_Join(
        hot_ens_JoinOps_t *jops,
	hot_gctx_t *gctxp	/*OUT*/ 
) {
  hot_err_t err = HOT_OK;
  hot_gctx_t g;
  dncall_t *dn;

  /* Initialize global state if not done so already.
   */
  if (!global.initialized) {
    err = hot_ens_Init(jops->argv, 
		       (jops->thread_mode == HOT_ENS_THREAD_SINGLE) ?
		       HOT_ENS_INIT_DONT_START : 
		       HOT_ENS_INIT_SEPARATE_THREAD);
    if (err != HOT_OK)
      return err;
  }
  
  begin_critical();    
  
  /* Allocate a new group context 
   * Initialize the group record.
   */
  g = alloc_gctx();
  g->joining = 1 ;
  g->leaving = 0 ;
  g->conf = jops->conf;
  g->env = jops->env;
  g->dncall_mode = jops->dncall_mode;
  g->vs.members = NULL;
  g->vs.nmembers = 0;
  g->join_sema = NULL ;
  g->leave_sema = NULL ;
  
  /* We actually set the "return" value here so that
   * it is available when the accepted view call arrives.
   * This is somewhat of a hack.
   */
  *gctxp = g;
  
  /* Block on the semaphore until the first view is delivered.
   */
  if (g->dncall_mode == HOT_ENS_DNCALL_SYNC) {
    err = global.thr->sema_create(0, &g->join_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err)) ;
      
    end_critical();
    err = global.thr->sema_dec(g->join_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    begin_critical();

    err = global.thr->sema_destroy(g->join_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    g->join_sema = NULL ;
  }
  
  end_critical();
  return HOT_OK;
}

/* Leave a group.  This should be the last call made to a given gctx.
 * No more events will be delivered to this gctx after the call
 * returns.  
 */
hot_err_t hot_ens_Leave(hot_gctx_t g) {
  hot_err_t err = HOT_OK;

  begin_critical();

  g->leaving = 1 ;
  
  fprintf(g->out,"leave\n") ;
  fflush(g->out) ;
  
  if (g->dncall_mode == HOT_ENS_DNCALL_SYNC) {
    /* Temporarily release the global lock while we block on the
     * semaphore until the exit event is delivered.  
     */
    assert(!g->leave_sema) ;
    err = global.thr->sema_create(0, &g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err)) ;

    end_critical();
    err = (*global.thr->sema_dec)(g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    begin_critical();

    err = global.thr->sema_destroy(g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err)) ;
    g->leave_sema = NULL ;

    release_gctx(g);
  }
  
  end_critical();

  return HOT_OK;
}

/* Send a multicast message to the group.
 */
hot_err_t hot_ens_Cast(
         hot_gctx_t g,
	 hot_msg_t msg, 
	 hot_ens_MsgSendView *send_view	/*OUT*/
) {
  dncall_t *dn;
  hot_err_t err;

  begin_critical() ;
  if (send_view != NULL) {
    *send_view = (g->group_blocked) ?
      HOT_ENS_MSG_SEND_NEXT_VIEW : HOT_ENS_MSG_SEND_CURRENT_VIEW ;
  }

  fprintf(g->out,"cast %s\n",) ;
  fflush(g->out) ;
  
  end_critical() ;
  return HOT_OK ;
}

/* Send a point-to-point message to the specified group member.
 */
hot_err_t hot_ens_Send(
        hot_gctx_t g, 
	hot_endpt_t *dest,
	hot_msg_t msg,
	hot_ens_MsgSendView *send_view /*OUT*/
) {
  dncall_t *dn;
  hot_err_t err;

  begin_critical() ;

  if (send_view != NULL) {
    *send_view = (g->group_blocked) ?
      HOT_ENS_MSG_SEND_NEXT_VIEW : HOT_ENS_MSG_SEND_CURRENT_VIEW ;
  }
  
  /* Enqueue the send request.
   */
  fprintf(g->out,"send %s\n",) ;
  fflush(g->out) ;
  
  end_critical() ;
  return HOT_OK ;
}

/* Report group members as failure-suspected.
 * 
 * NB:  In the initial implementation, this downcall will not be supported.
 *      (if invoked, an exeption will be raised by OCAML).
 */
hot_err_t hot_ens_Suspect(
        hot_gctx_t gctx,
	hot_endpt_t *suspects, 
	int nsuspects
) {
  int i ;

  begin_critical();    

  for (i=0;i<nsuspects;i++) {
    fprintf(g->out,"suspect %s\n",) ;
  }
  fflush(g->out) ;
  
  end_critical();
  return HOT_OK;
}

/* Request a protocol change.
 */
hot_err_t hot_ens_ChangeProtocol(
        hot_gctx_t gctx,
	char *protocol
) {
  dncall_t *dn;

  begin_critical();

  fprintf(g->out,"protocol %s\n", protocol) ;
  fflush(g->out) ;

  end_critical();
  return HOT_OK;
}

/* Request a protocol change specifying properties.
 */
hot_err_t hot_ens_ChangeProperties(
        hot_gctx_t gctx,
	char *properties
) {
  dncall_t *dn;
  assert(properties) ;

  begin_critical();

  fprintf(g->out,"properties %s\n", properties) ;
  fflush(g->out) ;

  end_critical();
  return HOT_OK;
}

/* Request a new view to be installed.
 */
hot_err_t hot_ens_RequestNewView(
        hot_gctx_t gctx
) {
  dncall_t *dn;
  begin_critical();
    
  fprintf(g->out,"prompt\n") ;
  fflush(g->out) ;
    
  end_critical();
  return HOT_OK;
}

/****************************************************************************/




void data_ready(gctx *g) {
  fgets(in,skjd) ;

  if (streq(type,"cast")) {
    origin = nexttok() ;
    msg = rest() ;
    g->conf.hot_ens_Receive
  }
    
  



/* Got a multicast message. 
 */
value hot_ens_ReceiveCast_cbd(value gctx_v, value origin_v, value msg_v) {
  hot_endpt_t origin;
  hot_msg_t msg;
  hot_err_t err;
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_ReceiveCast_cback receive_cast;
  
  begin_critical();
  env = gctx->env;
  receive_cast = gctx->conf.receive_cast;
  end_critical();
  
  Endpt_val(origin_v, &origin);
  msg = hot_ens_Msg_of_val(msg_v);
  
  if (receive_cast != NULL) {
    (*receive_cast)(gctx, env, &origin, msg);
  }
  
  err = hot_msg_Release(&msg);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  return Val_unit;
}

/* Got a point-to-point message.
 */
value hot_ens_ReceiveSend_cbd(
        value gctx_v,
	value origin_v, 
	value msg_v
) {
  hot_endpt_t origin;
  hot_msg_t msg;
  hot_err_t err;
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_ReceiveSend_cback receive_send;
  
  begin_critical();
  env = gctx->env;
  receive_send = gctx->conf.receive_send;
  end_critical();
  
  Endpt_val(origin_v, &origin);
  msg = hot_ens_Msg_of_val(msg_v);
  
  if (receive_send != NULL) {
    (*receive_send)(gctx, env, &origin, msg);
  }
  
  err = hot_msg_Release(&msg);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  return Val_unit;
}

/* Accepted a new view.
 */
value hot_ens_AcceptedView_cbd(	/* ML */
        value gctx_v,
	value view_state_v
) {
  hot_view_state_t view_state;
  hot_err_t err;
  hot_gctx_t g = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_AcceptedView_cback accepted_view;
  
  begin_critical();
  env = g->env;
  accepted_view = g->conf.accepted_view;
  
  /* Release the old view if necessary.
   */
  if (g->joining) {
    if ((g->vs.members != NULL) ||
	(g->vs.nmembers != 0))
      hot_sys_Panic("hot_ens_AcceptedView_cbd: internal inconsistency");
  } else {
    if ((g->vs.members == NULL) ||
	(g->vs.nmembers == 0))
      hot_sys_Panic("hot_ens_AcceptedView_cbd: internal inconsistency");
    release_endpt_array(&ch_endpt_array,g->vs.members,g->vs.nmembers) ;
  }
  
  /* The group is unblocked now.
   */
  g->group_blocked = 0;
  
  /* Setup the new view.
   */
  ViewState_of_val(view_state_v, &g->vs); 
  ViewState_of_val(view_state_v, &view_state); 

  end_critical();

  if (accepted_view != NULL) {
    (*accepted_view)(g, env, &view_state);
  }
  
  /* If this is the first view for the given gctx, unblock the join dncall.
   */
  begin_critical();

  if (view_state.members != NULL) {
    release_endpt_array(&ch_endpt_array,view_state.members,view_state.nmembers) ;
  }

  if (g->joining &&
      g->dncall_mode == HOT_ENS_DNCALL_SYNC) {
    assert(g->join_sema) ;
    if ((err = (*global.thr->sema_inc)(g->join_sema)) != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
  g->joining = 0 ;
  
  end_critical();
  
  return Val_unit;
}

/* Got a heartbeat event.
 */
value hot_ens_Heartbeat_cbd(
        value gctx_v,
        value rate_v
) {
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_Heartbeat_cback heartbeat;
  
  begin_critical();
  env = gctx->env;
  heartbeat = gctx->conf.heartbeat;
  end_critical();
  
  if (heartbeat != NULL) {
    (*heartbeat)(gctx, env, Rate_val(rate_v));
  }
  
  return Val_unit;
}

/* The group is about to block.
 */
value hot_ens_Block_cbd(value gctx_v) {
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_Block_cback block;
  
  /* Set the "group_blocked" flag in the corresp. gctx.
   * The flag is cleared when the next view is delivered.
   */
  begin_critical();
  env = gctx->env;
  block = gctx->conf.block;
  gctx->group_blocked = 1;
  end_critical();
  
  if (block != NULL) {
    (*block)(gctx, env);
  }

  return Val_unit;
}

/* The member has left the group.
 */
value hot_ens_Exit_cbd(value gctx_v) {
  hot_gctx_t g = hot_ens_Gctx_of_val(gctx_v);
  hot_ens_Exit_cback exit_cb;
  hot_err_t err;
  void *env;

  /* Phase 1: get info for callback.
   */
  begin_critical();  
  
  if (!g->leaving)
    hot_sys_Panic("hot_ens_Exit_cbd: mbr state is not leaving");

  env = g->env;
  exit_cb = g->conf.exit;
  
  end_critical();

  if (exit_cb != NULL) {
    (*exit_cb)(g, env);
  }  
  
  /* Phase 2: cleanup.
   */
  begin_critical();  

  /* If leave downcall is asynchronous, release gctx now.
   */
  if (g->dncall_mode == HOT_ENS_DNCALL_ASYNC) {
    release_gctx(g);
  } else {
    /* If leave is synchronous, unblock the leave downcall for the
     * specified gctx.  The gctx is recycled in the leave dncall.  
     */
    assert(g->leave_sema) ;
    err = (*global.thr->sema_inc)(g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
  
  end_critical();  
  
  return Val_unit;
}


#ifdef WIN32

void
Perror(char *s) {
  char error[256];
  int errno;
  
  errno = GetLastError();
  (void) FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM ,
		       NULL,	// message format
		       errno,
		       0,		// language
		       error,
		       256,		// buffer size, ignored for ALLOC_BUF
		       NULL	// args for msg format
		       );
  printf("%s: (%d) %s\n", s, errno, error);
  //LocalFree(error);
}

socketpair(
        int af,
        int type,
        int protocol, 
	int fd[2]
) {
  struct sockaddr_in sin;
  int result, sinlen;
  WSADATA wsaData;
  SOCKET	s0, s1;
  
  (void) WSAStartup(MAKEWORD(1, 0), &wsaData);
  
  s0 = socket(AF_INET, SOCK_DGRAM, 0);
  if (s0 < 0) {
    Perror("Socket0");
    return -1;
  }
  
  s1 = socket(AF_INET, SOCK_DGRAM, 0);
  if (s1 < 0) {
    Perror("socket1");
    closesocket(s0);
    return -1;
  }
  
  sin.sin_family      = AF_INET;
  sin.sin_addr.s_addr = ntohl(INADDR_LOOPBACK);
  sin.sin_port        = ntohs(0);
  
  result = bind(s0, (struct sockaddr *)&sin, sizeof(sin));
  if (result < 0) {
    Perror("Bind");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  sinlen = sizeof(sin);
  result = getsockname(s0, (struct sockaddr *)&sin, &sinlen);
  if (result < 0) {
    Perror("getsockname");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  result = connect(s1, (struct sockaddr *)&sin, sinlen);
  if (result < 0) {
    Perror("connect");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }   
  
  if ((fd[0] = _open_osfhandle(s0,0)) == -1) {
    Perror("open_osfhandle s0");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  if ((fd[1] = _open_osfhandle(s1,0)) == -1) {
    Perror("open_osfhandle s1");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  return 0;
}

#endif
