/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/*
 * Contents: C side of the HOT C interface to Ensemble.
 *
 * Author:  Alexey Vaysburd, Dec. 1996.
 */ 

#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef WIN32
void Perror(char *s);
#include <winsock2.h>
#else
#define Perror(s) perror(s)
#include <sys/socket.h>
#include <unistd.h>
#endif
#include "mlvalues.h"
#include "memory.h"
#include "callback.h"
#include "alloc.h"
extern mlsize_t string_length P((value));

#include "hot_sys.h"
#include "hot_error.h"
#include "hot_thread.h"
#include "hot_msg.h"
#include "hot_mem.h"
#include "hot_ens.h"

/****************************************************************************/

value hot_ml_alloc(char *debug,mlsize_t s, tag_t t) {
  /*
  fprintf(stderr,"hot_ml_alloc:%s:%d:%d\n", debug, (int)s, (int)t) ;
  */
  assert(s >= 0) ;
  if (s == 0) {
    return Atom(0) ;
  } else if (s < Max_young_wosize) {
    return alloc(s,t) ;
  } else {
    return alloc_shr(s,t) ;
  }
}  

value hot_ml_alloc_tuple(char *debug, mlsize_t s) {
  return hot_ml_alloc(debug,s,0) ;
}

value hot_ml_copy_string(char *s) {
  int len;
  value res;
  
  len = strlen(s);
  res = alloc_string(len);
  bcopy(s, String_val(res), len);
  return res;
}

/****************************************************************************/

/******************* Threads-interface configurations ************************/

static hot_err_t noop_thread_Create(
        void (*routine)(void*), 
	void *arg,
	hot_thread_attr_t *attr
) {
  (*routine)(arg);
  return HOT_OK;
}
static hot_err_t noop_thread_Yield(void) { return HOT_OK; }
static hot_err_t noop_thread_Usleep(int usecs) { return HOT_OK; }
static hot_err_t noop_lck_Create(hot_lock_t *l) { return HOT_OK; }
static hot_err_t noop_lck_Destroy(hot_lock_t l) { return HOT_OK; }
static hot_err_t noop_lck_Lock(hot_lock_t l) { return HOT_OK; }
static hot_err_t noop_lck_Unlock(hot_lock_t l) { return HOT_OK; }
static hot_err_t noop_sema_Create(int initial_value, hot_sema_t *semap) { return HOT_OK; } 
static hot_err_t noop_sema_Destroy(hot_sema_t sema) { return HOT_OK; }
static hot_err_t noop_sema_Inc(hot_sema_t sema) { return HOT_OK; }
static hot_err_t noop_sema_Dec(hot_sema_t sema) { return HOT_OK; }

/* No-op threads configuration -- for single-threaded execution mode.
 */ 
static hot_thread_conf_t hot_noop_thread_conf = {
  noop_thread_Create,
  noop_thread_Yield,
  noop_thread_Usleep,
  noop_lck_Create,
  noop_lck_Destroy,
  noop_lck_Lock,
  noop_lck_Unlock,
  noop_sema_Create,
  noop_sema_Destroy,
  noop_sema_Inc,
  noop_sema_Dec
} ;
  
/* Standard threads configuration -- for multi-threaded execution.
 */
static hot_thread_conf_t hot_std_thread_conf = {
  hot_thread_Create,
  hot_thread_Yield,
  hot_thread_Usleep,
  hot_lck_Create,
  hot_lck_Destroy,
  hot_lck_Lock,
  hot_lck_Unlock,
  hot_sema_Create,
  hot_sema_Destroy,
  hot_sema_Inc,
  hot_sema_Dec
} ;

/****************************************************************************/

/* Downcall types. The ordering must match ML dncall_t type definition.
 */
typedef enum {
  DNCALL_JOIN,
  DNCALL_CAST,
  DNCALL_SEND,
  DNCALL_SUSPECT,
  DNCALL_PROTOCOL,
  DNCALL_PROPERTIES,
  DNCALL_LEAVE,
  DNCALL_PROMPT,
  DNCALL_VOID
} dncall_type ;

/****************************************************************************/

/* This describes a downcall request issued by the application.
 * Not all fields are used by all downcall types. 
 */
typedef struct dncall {
  struct dncall *next ;
  hot_gctx_t gctx ;
  dncall_type type ;
  hot_ens_JoinOps_t *jops ;	
  hot_msg_t msg ;		/* NB:  Store here an alias of the dncall msg */
  hot_endpt_t dest ;
  hot_endpt_t *suspects ;
  unsigned nsuspects ;
  char *protocol ;
} dncall_t;

/****************************************************************************/

/* This describes a group context.
 */
struct hot_gctx {
  unsigned int id ;		/* unique integer identifier */
  hot_gctx_t next;		/* linked list of alive members */
  hot_ens_cbacks_t conf;	/* application callbacks */
  void *env;			/* application state */
  int group_blocked;		/* is group blocked? */
  int joining ;			/* are we joining? */
  int leaving ;			/* are we leaving? */
  
  /* Various semaphores.
   */
  hot_sema_t join_sema ;
  hot_sema_t leave_sema ;

  hot_view_state_t vs ;		/* view state */
  hot_ens_dncall_mode dncall_mode; /* sync/async downcalls (joins/leaves) */
} ;

/****************************************************************************/

/* Global state.
 */
static struct {
  hot_lock_t mutex;		/* global mutex lock  */
  int in_critical;		/* Set when mutex is locked */
  int initialized;		/* Set when hot_c is initialized. */
  int started;			/* Set when Ensemble/OCAML is started */
  int fd[2];			/* Pipe for communicating with OCAML */
  
  int ensemble_blocked;		/* Set when Ensemble may be blocked */
  int alarm_scheduled;		/* Set when an alarm to process downcalls
				 * has been scheduled. */
  
  dncall_t *dn_free ;		/* free list of dncalls */
  dncall_t *dn_tail ;		/* last pending dncall */
  
  hot_gctx_t g_alive ;		/* allocated group contexts */
  
  hot_thread_conf_t *thr;	/* threads-interface configuration */
  hot_ens_thread_mode thread_mode; /* single-threaded/multi-threaded */
} global ;

/****************************************************************************/

/* Startup options.
 */
typedef enum {
  HOT_ENS_INIT_DONT_START,
  HOT_ENS_INIT_SAME_THREAD,
  HOT_ENS_INIT_SEPARATE_THREAD
} hot_ens_init_ops ;

/****************************************************************************/

static inline void begin_critical(void) {
  hot_err_t err ;
  if ((err = (*global.thr->lck_lock)(global.mutex)) != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err)) ;
  assert(!global.in_critical) ;
  global.in_critical = 1 ;
}

static inline void end_critical(void) {
  hot_err_t err ;
  assert(global.in_critical) ;
  global.in_critical = 0 ;
  if ((err = (*global.thr->lck_unlock)(global.mutex)) != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
}

/****************************************************************************/

char *hot_ens_argv_null[2] = {"(?)", NULL};

/* Start OCAML.  The function does not return.
 */
static void ens_do_init(void *param) {
  static int called_already = 0 ;
  if (called_already)
    hot_sys_Panic("caml_startup:called twice") ;
  called_already = 1 ;

  caml_startup((char**) param ? param : hot_ens_argv_null);
}

/* Initialize the interface. Start Ensemble/OCAML if necessary.
 */ 
static hot_err_t hot_ens_Init(char **argv, hot_ens_init_ops init_ops) {
    int i;
    hot_err_t err;
    hot_thread_attr_t attr;
    int size ;

#ifdef _WIN32_PRECOMPILED
    if (init_ops != HOT_ENS_INIT_SEPARATE_THREAD) {
    	fprintf(stderr, "This pre-compiled WIN32 version of HOT only supports the SEPARATE_THREAD options\n");
    	return hot_err_Create(0, "hot_ens_Init:  unsupported init option");;
    }
    global.initialized = 1;
    global.thread_mode = HOT_ENS_THREAD_MULTI;
    global.thr = &hot_std_thread_conf;
#else
    if (global.initialized) {
      switch (init_ops) {
      case HOT_ENS_INIT_SAME_THREAD:
	/* If hot_c has already been initialized, start Ensemble now.
	 */
	begin_critical();
	if (global.started) {
	  end_critical();
	  return HOT_OK;
	}
	global.started = 1;
	end_critical();
	return (*global.thr->thread_create)(ens_do_init, (void*) argv, NULL);
	
      case HOT_ENS_INIT_DONT_START:   	   /* FALL THROUGH */
      case HOT_ENS_INIT_SEPARATE_THREAD:
	return HOT_OK;
	
      default:
	hot_sys_Panic("hot_ens_Init:  unknown init option");
	break;
      }
    }
    global.initialized = 1;

    /* Choose a threads configuration.  For single-threaded mode, use no-op.
     */
    switch (init_ops) {
    case HOT_ENS_INIT_SEPARATE_THREAD:
      global.thread_mode = HOT_ENS_THREAD_MULTI;
      global.thr = &hot_std_thread_conf;
      break;
    case HOT_ENS_INIT_DONT_START:
      global.thread_mode = HOT_ENS_THREAD_SINGLE;
      global.thr = &hot_noop_thread_conf;
      break;
    default:
      return hot_err_Create(0, "hot_ens_Init:  unsupported init option");
      break;
    }
#endif

    /* Initialize C side of the interface (preallocate/initialize gctx entries,
     * dncall structures, mutex.
     */
    if ((err = (*global.thr->lck_create)(&global.mutex)) != HOT_OK)
	hot_sys_Panic(hot_err_ErrString(err));
    
    begin_critical();

    global.ensemble_blocked = 0;
    global.alarm_scheduled = 0;

    global.dn_free = NULL ;
    global.dn_tail = NULL ;

    global.g_alive = NULL ;
    
    /* Open a socketpair for communicating with Ensemble.
     */    
    if (socketpair(PF_UNIX,SOCK_STREAM,0,global.fd) < 0) {
	hot_sys_Panic("hot_ens_Init: socket() failed");
    }
    
    end_critical();

    switch (init_ops) {
    case HOT_ENS_INIT_SAME_THREAD:
      /* FALL THROUGH.
       * NB:  No-op thread_create is implemented as a direct invocation.
       */ 
    case HOT_ENS_INIT_SEPARATE_THREAD:
      /* Start Ensemble/OCAML.  Don't allow starting Ensemble more than once.
       */    
      begin_critical();
      if (global.started) {
	end_critical();
	return hot_err_Create(0, "hot_ens_Init:  Ensemble already started");
      }
      global.started = 1;
      end_critical();

      attr.stacksize = 1024 * 1024; /* stack size for Ensemble thread = 1M */
   
      if ((err = (*global.thr->thread_create)(ens_do_init, (void*) argv, &attr)) != HOT_OK)
	return err;
      break;

    case HOT_ENS_INIT_DONT_START:
      break;

    default:
      hot_sys_Panic("hot_ens_Init:  bad init option");
      break;
    }

    return HOT_OK;
}

value hot_ens_thread_yield(void) { /* ML */
  hot_thread_Yield() ;
  return Val_unit ;
}

value hot_ens_thread_usleep(value usecs_v) { /* ML */
  int usecs ;
  usecs = Int_val(usecs_v) ;
  hot_thread_Usleep(usecs) ;
  return Val_unit ;
}

/****************************************************************************/

static void release_endpt_array(
        channel_t *ch,
        hot_endpt_t *endpts,
	unsigned nendpts
) {
  int size = nendpts * sizeof(*endpts) ;
  assert(endpts) ;
  assert(nendpts > 0) ;

  hot_free(ch,size,endpts) ;
}

/****************************************************************************/

/******************** gctx + dn-request allocation ***************************/

/* Return the index of a newly allocated gctx.
 * The global data record must be locked at this point.
 */
static 
hot_gctx_t alloc_gctx(void) {
  static unsigned int id = 0 ;
  int index;
  hot_gctx_t gctx ;
  hot_err_t err;
  
  gctx = hot_malloc(&ch_gctx,sizeof(*gctx)) ;

  gctx->id = id++ ;
  gctx->next = global.g_alive ;
  global.g_alive = gctx ;
  gctx->vs.members = NULL ;
  gctx->vs.nmembers = -1 ;
  
  return gctx ;
}

/* Release a gctx descriptor.
 * The global data record must be locked at this point.
 */
static 
void release_gctx(hot_gctx_t g) {
  hot_gctx_t *tmpp, tmp ;

  assert(g) ;

  /* Release the view information.
   */
  if ((g->vs.members == NULL) ||
      (g->vs.nmembers == 0))
    hot_sys_Panic("HOT:release_gctx: internal inconsistency");
  release_endpt_array(&ch_endpt_array, g->vs.members,g->vs.nmembers) ;
  g->vs.members = NULL ;
  g->vs.nmembers = -1 ;

  assert(!g->join_sema) ;
  assert(!g->leave_sema) ;
  assert(!g->vs.members) ;
  assert(g->vs.nmembers == -1) ;

  /* Strip this member from list of live members.
   */
  for(tmpp=&global.g_alive;*tmpp;tmpp=&(*tmpp)->next)
    if (*tmpp == g) {
      *tmpp = (*tmpp)->next ;
      break ;
    }

  /* Check to make sure that it is no longer around.
   */
  for (tmp = global.g_alive;tmp; tmp = tmp->next)
    if (tmp == g)
      hot_sys_Panic("HOT:release_gctx:internal inconsistency") ;

  hot_free(&ch_gctx,sizeof(*g),g) ;
}

/* Return the index of a newly allocated downcall struct.
 * The global data record must be locked at this point.
 */
static 
dncall_t *alloc_dn(
        hot_gctx_t gctx,
	dncall_type type
) {
  dncall_t *d ;
  int ret ;
  
  /* Allocate more gctx structs if we are out of them.
   */
  if (global.dn_free) {
    d = global.dn_free ;
    global.dn_free = d->next ;
    assert(d->type == DNCALL_VOID) ;
  } else {
    d = (dncall_t*) hot_malloc(&ch_dncall,sizeof(*d)) ;
  }
  memset(d,0,sizeof(*d)) ;	/* BUG */

  /* Insert in pending queue.
   */
  d->next = global.dn_tail ;
  global.dn_tail = d ;

  /* Initialize some fields.
   */
  d->gctx = gctx ;
  d->type = type ;

  /* If Ensemble may be blocked and no dncall-processing alarm has
   * been scheduled and the group for the given gctx is not blocked,
   * request Ensemble's attention by writing to the C<->Ensemble
   * pipe.  
   */
  if (global.ensemble_blocked && 
      !global.alarm_scheduled && 
      !d->gctx->group_blocked) {
    char buf;
#ifdef WIN32
    ret = send(_get_osfhandle(global.fd[1]), &buf, 1, 0) ;
#else
    ret = write(global.fd[1], &buf, 1) ;
#endif
    if (ret != 1) {
      Perror("write");
      hot_sys_Panic("alloc_dn: write() failed");
    }
    global.alarm_scheduled = 1;
  }

  return d ;
}

/* Release a downcall struct.
 * The global data record must be locked at this point.
 */
static void release_dn(dncall_t *d) {
  assert(d) ;
  assert(!d->next) ;
  assert(!d->jops) ;
  assert(!d->suspects) ;
  assert(!d->protocol) ;

  memset(d,0,sizeof(*d)) ;
  d->next = global.dn_free ;
  d->type = DNCALL_VOID ;
  global.dn_free = d ;
}

/*###################### C <-> ML Conversion Routines ###############*/

/* Extract an endpoint structure from a value.
 */
static void Endpt_val(
        value endp_v,
	hot_endpt_t *endp	/*OUT*/ 
) {
  strcpy(endp->name, String_val(endp_v));
}
    
/* Return a value contaiing an endpoint.
 */
static value Val_endpt(
        hot_endpt_t *endp
) {
  return hot_ml_copy_string(endp->name);
}

/* Convert a heartbeat rate (in milliseconds) into the corresponding value.
 */
static value Val_of_rate(unsigned rate) {
  return copy_double(((double) rate) / 1000);
}

/* Convert a value into a heartbeat rate.  Note that in Ensemble heartbeat
 * rate is a float (# sec), whereas it is unsigned (# milliseconds) at the
 * application level, so we need to do this conversion, too.
 */
static unsigned Rate_val(value hbt_v) {
  unsigned rate = (unsigned) (1000 * Double_val(hbt_v));
  return rate;
}

/* Enum types specifying positions of fields in ML type definitions:
 */

/* ML type definition:
 *
 * type ens_join_options_t = {
 *   heartbeat_rate:	float;
 *   transports:       	string;
 *   protocol: 		string;
 *   group_name:       	string;
 *   properties:        string;
 *   use_properties:    bool;
 *   groupd:            bool;
 *   params:            string
 *   debug:             bool;
 * }
 */
enum join_ops_enum {
    JOPS_HEARTBEAT_RATE,
    JOPS_TRANSPORTS,
    JOPS_PROTOCOL,
    JOPS_GROUP_NAME,
    JOPS_PROPERTIES,
    JOPS_USE_PROPERTIES,
    JOPS_GROUPD,
    JOPS_PARAMS,
    JOPS_DEBUG,

    JOPS_NFIELDS
};

#define JOPS_NROOTS (JOPS_NFIELDS + 1)
#define JOPS_V JOPS_NFIELDS

/* Convert a join-options struct into a value.
 */
static value Val_of_joinOps(hot_ens_JoinOps_t *ops) {
    value jops_v;
    Push_roots(rt, JOPS_NROOTS);

    rt[JOPS_HEARTBEAT_RATE] = Val_of_rate(ops->heartbeat_rate);
    rt[JOPS_TRANSPORTS] = hot_ml_copy_string(ops->transports);
    rt[JOPS_PROTOCOL] = hot_ml_copy_string(ops->protocol);
    rt[JOPS_GROUP_NAME] = hot_ml_copy_string(ops->group_name);
    rt[JOPS_PROPERTIES] = hot_ml_copy_string(ops->properties);
    rt[JOPS_USE_PROPERTIES] = Val_bool(ops->use_properties) ;
    rt[JOPS_GROUPD] = Val_bool(ops->groupd) ;
    rt[JOPS_PARAMS] = hot_ml_copy_string(ops->params);
    rt[JOPS_DEBUG] = Val_bool(ops->debug) ;

    rt[JOPS_V] = alloc_tuple(JOPS_NFIELDS);

    Field(rt[JOPS_V], JOPS_HEARTBEAT_RATE) = rt[JOPS_HEARTBEAT_RATE];
    Field(rt[JOPS_V], JOPS_TRANSPORTS) = rt[JOPS_TRANSPORTS];
    Field(rt[JOPS_V], JOPS_PROTOCOL) = rt[JOPS_PROTOCOL];
    Field(rt[JOPS_V], JOPS_GROUP_NAME) = rt[JOPS_GROUP_NAME];
    Field(rt[JOPS_V], JOPS_PROPERTIES) = rt[JOPS_PROPERTIES];
    Field(rt[JOPS_V], JOPS_USE_PROPERTIES) = rt[JOPS_USE_PROPERTIES];
    Field(rt[JOPS_V], JOPS_GROUPD) = rt[JOPS_GROUPD];
    Field(rt[JOPS_V], JOPS_PARAMS) = rt[JOPS_PARAMS];
    Field(rt[JOPS_V], JOPS_DEBUG) = rt[JOPS_DEBUG] ;

    jops_v = rt[JOPS_V];
    Pop_roots();

    return jops_v;
}

/* Return a HOT message created from the given msg value.
 */
static inline hot_msg_t hot_ens_Msg_of_val(value msg_v) {
  hot_msg_t msg;
  char *data;
  unsigned size;

  data = String_val(msg_v);
  size = string_length(msg_v);

#ifdef HOT_MSG_PADDING
  {
    char npad = data[0];
    msg = hot_msg_CreateDontCopy(data + npad, size - npad, NULL, NULL);
  }
#else
  msg = hot_msg_CreateDontCopy(data, size, NULL, NULL);
#endif

  return msg;
}

/* Return a msg value corresponding to the given HOT message.
 */
static inline value Val_msg(hot_msg_t msg) {
  value msg_v;
  void *data;
  unsigned size;
  hot_err_t err;
    
  err = hot_msg_GetPos(msg, &size);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  err = hot_msg_Look(msg, size, &data);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
#ifdef HOT_MSG_PADDING
  {
    char pad[HOT_MSG_PADDING];
    char npad = HOT_MSG_PADDING - (size % HOT_MSG_PADDING);
    
    pad[0] = npad;
    msg_v = alloc_string(size + npad);
    memcpy(String_val(msg_v), pad, npad);
    memcpy(String_val(msg_v) + npad, data, size);
  }
#else
  msg_v = alloc_string(size);
  memcpy(String_val(msg_v), data, size);
#endif

  return msg_v;
}

/* Convert a group context into a value.
 */
static inline value Val_of_gctx(hot_gctx_t gctx) {
  return Val_int(gctx->id);
}

/* Convert a value into group context.
 */
static inline hot_gctx_t hot_ens_Gctx_of_val(value gctx_v) {
  int id = Int_val(gctx_v) ;
  hot_gctx_t g ;
  
  for (g=global.g_alive; g; g=g->next)
    if (g->id == id) break ;
  if (!g) {
    printf("HOT_C:id=%d\n", id) ;
    for (g=global.g_alive; g; g=g->next)
      printf(" alive.id=%d\n",g->id) ;
    
    hot_sys_Panic("Gctx_of_val:id not found") ;
  }
  return g ;
}

/* Convert a value into an array of endpoints.  Allocate the array and
 * return the pointer to it in *endp_p.
 */
static void Endpt_array_val(
        value endp_array_v,
	unsigned *n_endp,	/*OUT*/ 
	hot_endpt_t **endp_p	/*OUT*/ 
) {
  int i;
  hot_endpt_t *endp_arr;
  
  assert(n_endp) ;
  assert(endp_p) ;
  *n_endp = Wosize_val(endp_array_v);
  endp_arr = (hot_endpt_t*) hot_malloc(&ch_endpt_array,sizeof(hot_endpt_t) * (*n_endp));
  if (endp_arr == NULL)
    hot_sys_Panic("Endpt_array_val: hot_malloc failed");
  for (i = 0; i < *n_endp; i++)
    Endpt_val(Field(endp_array_v, i), &endp_arr[i]);
  *endp_p = endp_arr;
}

/* Convert an array of endpoints into a value.
 */
static value Val_endpt_array(
        unsigned nendp,
	hot_endpt_t *endp
) {
  int i;
  value ret ;
  Push_roots(r,1) ;
#define endp_v r[0]
  endp_v = alloc_tuple(nendp) ;
  
  /* Initialize the array because we are going to do
   * allocation.
   */
  for (i=0;i<nendp;i++)
    Field(endp_v,i) = Val_unit ;
  
  for (i=0;i<nendp;i++)
    Field(endp_v,i) = Val_endpt(&endp[i]) ;
  
  ret = endp_v ;
#undef endp_v
  Pop_roots() ;
  return ret ;
}

enum ens_view_state_enum {
    HOT_ENS_VIEW_STATE_VERSION,
    HOT_ENS_VIEW_STATE_GROUP,
    HOT_ENS_VIEW_STATE_VIEW,
    HOT_ENS_VIEW_STATE_RANK,
    HOT_ENS_VIEW_STATE_PROTOCOL,
    HOT_ENS_VIEW_STATE_GROUPD,
    HOT_ENS_VIEW_STATE_VIEW_ID,
    HOT_ENS_VIEW_STATE_PARAMS,
    HOT_ENS_VIEW_STATE_XFER_VIEW,
    HOT_ENS_VIEW_STATE_PRIMARY
};

/*
 * ML type definitions: 
 *
 * type ens_view_state = {
 *     version:      string;
 *     group:        string;
 *     view:         string array; 
 *     rank:         int;
 *     protocol:     string;
 *     groupd:       bool;
 *     view_id:      int * string;
 *     params:       string;
 *     xfer_view:    bool;
 *     primary:      bool
 * }
 *
 * C type definitions: 
 *
 * struct hot_view_state {
 *     char version[HOT_ENS_MAX_VERSION_LENGTH];
 *     char group_name[HOT_ENS_MAX_GROUP_NAME_LENGTH];
 *     hot_endpt_t *members;
 *     unsigned nmembers;
 *     unsigned my_rank;
 *     char protocol[HOT_ENS_MAX_PROTO_NAME_LENGTH];
 *     int groupd;
 *     hot_view_id view_id;
 *     char params[HOT_ENS_MAX_PARAMS_LENGTH];
 *     int xfer_view;
 *     int primary;
 * };
 */


static void string_of_val_copy(
	char *debug,
        value s_v,
	char *s,
	int max_len
) {
  int len ;
  assert(s) ;
  
  len = string_length(s_v) ;
  if (len+1 >= max_len) {
    char msg[1000] ;
    sprintf(msg,"ViewState_of_val: %s string too long", debug) ;
    hot_sys_Panic(msg) ;
  }
  
  assert(strlen(String_val(s_v)) == len) ;

  strcpy(s, String_val(s_v)) ;
}

/* Convert a value into a view state.
 */
static void ViewState_of_val(
        value vs_v, 
	hot_view_state_t *vs	/*OUT*/ 
) {
  memset(vs,0,sizeof(*vs)) ;
  vs->rank      = Int_val(Field(vs_v, HOT_ENS_VIEW_STATE_RANK));
  vs->groupd    = Int_val(Field(vs_v, HOT_ENS_VIEW_STATE_GROUPD));
  vs->xfer_view = Int_val(Field(vs_v, HOT_ENS_VIEW_STATE_XFER_VIEW));
  vs->primary   = Int_val(Field(vs_v, HOT_ENS_VIEW_STATE_PRIMARY));
  vs->view_id.ltime = Int_val(Field(Field(vs_v, HOT_ENS_VIEW_STATE_VIEW_ID), 0));
  
  string_of_val_copy("view_id.coord",
		     Field(Field(vs_v, HOT_ENS_VIEW_STATE_VIEW_ID), 1),
		     vs->view_id.coord.name,
		     HOT_ENDP_MAX_NAME_SIZE) ;

  string_of_val_copy("version",
		     Field(vs_v, HOT_ENS_VIEW_STATE_VERSION),
		     vs->version,
		     HOT_ENS_MAX_VERSION_LENGTH) ;

  string_of_val_copy("group_name",
		     Field(vs_v, HOT_ENS_VIEW_STATE_GROUP),
		     vs->group_name,
		     HOT_ENS_MAX_GROUP_NAME_LENGTH) ;
  
  string_of_val_copy("protocol",
		     Field(vs_v, HOT_ENS_VIEW_STATE_PROTOCOL),
		     vs->protocol,
		     HOT_ENS_MAX_PROTO_NAME_LENGTH) ;

  string_of_val_copy("params",
		     Field(vs_v, HOT_ENS_VIEW_STATE_PARAMS),
		     vs->params,
		     HOT_ENS_MAX_PARAMS_LENGTH) ;

  Endpt_array_val(Field(vs_v, HOT_ENS_VIEW_STATE_VIEW),
			    &vs->nmembers, &vs->members);
}

/* type dncall_t = 
 * | C_Join of ens_join_options_t (* tuple, Wosize = 1, tag = 0 *)
 * | C_Cast of msg_t              (* tuple, Wosize = 1, tag = 1 *)
 * | C_Send of endpt_t * msg_t    (* tuple, Wosize = 2, tag = 2 *)
 * | C_Suspect of endpt_t array   (* tuple, Wosize = 1, tag = 3 *)
 * | C_Protocol of string         (* tuple, Wosize = 1, tag = 4 *)
 * | C_Properties of string        * tuple, Wosize = 1, tag = 5 *)
 * | C_Leave of unit              (* tuple, Wosize = 1, tag = 6 *)
 * | C_Prompt of unit
 * | C_Void of unit
 *
 * type c_dncall_t = {
 *   gctx: gctx_t;
 *   req: dncall_t 
 * }
 */

/* Return a value containing the specified downcall request.
 * The value must match the corresp. ML type definition.
 */
static value Val_of_dn(dncall_t *dn) {
  value pair_v ;
  hot_err_t err;
  
  Push_roots(r, 3);
#define field0 r[0]
#define field1 r[1]
#define dn_v   r[2]

  switch (dn->type) {
  case DNCALL_JOIN:
    field0 = Val_of_joinOps(dn->jops);
    hot_free(&ch_joinops, sizeof(*dn->jops), dn->jops) ;
    dn->jops = NULL ;
    
    dn_v = hot_ml_alloc("Val_of_dn:join",1, DNCALL_JOIN);
    Field(dn_v, 0) = field0;
    break;
  case DNCALL_LEAVE:
    dn_v = hot_ml_alloc("Val_of_dn:leave",1, DNCALL_LEAVE);
    Field(dn_v, 0) = Val_unit ;
    break;
  case DNCALL_CAST:
    field0 = Val_msg(dn->msg);
    if ((err = hot_msg_Release(&dn->msg)) != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    dn_v = hot_ml_alloc("Val_of_dn:cast", 1, DNCALL_CAST);
    Field(dn_v, 0) = field0;
    break;
  case DNCALL_SEND:
    field0 = Val_endpt(&dn->dest);
    field1 = Val_msg(dn->msg);
    if ((err = hot_msg_Release(&dn->msg)) != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    dn_v = hot_ml_alloc("Val_of_dn:send", 2, DNCALL_SEND);
    Field(dn_v, 0) = field0;
    Field(dn_v, 1) = field1;
    break;
  case DNCALL_SUSPECT:
    field0 = Val_endpt_array(dn->nsuspects, dn->suspects);
    
    release_endpt_array(&ch_endpt_array,dn->suspects,dn->nsuspects) ;
    dn->suspects = NULL ;
    dn->nsuspects = -1 ;

    dn_v = hot_ml_alloc("Val_of_dn:suspect",1, DNCALL_SUSPECT);
    Field(dn_v, 0) = field0;
    break;
  case DNCALL_PROMPT:
    dn_v = hot_ml_alloc("Val_of_dn:prompt", 1, DNCALL_PROMPT);
    Field(dn_v, 0) = Val_unit ;
    break;
  case DNCALL_PROTOCOL:
    field0 = hot_ml_copy_string(dn->protocol);

    hot_free(&ch_protocol, HOT_ENS_MAX_PROTO_NAME_LENGTH, dn->protocol) ;
    dn->protocol = NULL ;

    dn_v = hot_ml_alloc("Val_of_dn:protocol", 1, DNCALL_PROTOCOL);
    Field(dn_v, 0) = field0;
    break;
  case DNCALL_PROPERTIES:
    field0 = hot_ml_copy_string(dn->protocol);

    hot_free(&ch_protocol, HOT_ENS_MAX_PROTO_NAME_LENGTH, dn->protocol) ;
    dn->protocol = NULL ;

    dn_v = hot_ml_alloc("Val_of_dn:properties", 1, DNCALL_PROPERTIES);
    Field(dn_v, 0) = field0;
    break;
  default:
    hot_sys_Panic("Val_of_dn: bad downcall type");
    break;
  }
  
  pair_v = hot_ml_alloc_tuple("Val_of_dn:pair", 2);
  Field(pair_v, 0) = Val_of_gctx(dn->gctx) ; /* no alloc */
  Field(pair_v, 1) = dn_v;
  
#undef field0
#undef field1
#undef dn_v
  Pop_roots();
  return pair_v;
}

/*######################## User Downcalls ###################################*/

/* We maintain an array of gctx structures mirroring that in ML.
 * When a downcall is made, messages/actions are enqueued, and an ML alarm
 * is scheduled.  To do that, we open a pipe at the initialization time.
 * ML side registers an alarm waiting on the read end of the pipe.
 * When a C downcall is made, enqueue the action
 * and write to the pipe. When alarm is invoked in the
 * ML thread, it will convert those actions into ML values and enqueue within
 * ML, and request a heartbeat for the group, if necessary.
 *
 * To avoid having to write to the pipe all the time:  In main_loop, before
 * Ensemble goes to sleep, set a special flag telling the C application that
 * Ens. is now blocked and is not going to look at the "actions-pending" flag
 * until awaken by writing to the pipe.  So in a downcall, see if the flag is
 * set.  If it is, then write to the pipe;  otherwise, just add the actions.
 */

/* Exported interface:  
 */

/* Start Ensemble in the current thread.  
 * If successful, this call will block forever.
 */
hot_err_t hot_ens_Start(char **argv) {
  return hot_ens_Init(argv, HOT_ENS_INIT_SAME_THREAD);
}

/* Initialize/reset an options struct.
 */
void hot_ens_InitJoinOps(hot_ens_JoinOps_t *ops) {
  if (ops == NULL)
    hot_sys_Panic("hot_ens_InitJoinOps: bad argument (NULL)");
  
  memset(ops, 0, sizeof(hot_ens_JoinOps_t));
  ops->heartbeat_rate = HOT_ENS_DEFAULT_HEARTBEAT_RATE;
  strcpy(ops->transports, HOT_ENS_DEFAULT_TRANSPORT);
  strcpy(ops->group_name, HOT_ENS_DEFAULT_GROUP_NAME);
  strcpy(ops->protocol, HOT_ENS_DEFAULT_PROTOCOL);
  strcpy(ops->properties, HOT_ENS_DEFAULT_PROPERTIES);
  ops->use_properties = 1;
  strcpy(ops->params, "");
  ops->groupd = 0;
  ops->dncall_mode = HOT_ENS_DNCALL_SYNC;
  ops->thread_mode = HOT_ENS_THREAD_MULTI;
  ops->argv = NULL ;
  ops->env = NULL ;
}

/* Join a group.  In case of success, the call returns after the first view
 * is delivered.  The group context is returned in *gctxp.
 */
hot_err_t hot_ens_Join(
        hot_ens_JoinOps_t *jops,
	hot_gctx_t *gctxp	/*OUT*/ 
) {
  hot_err_t err = HOT_OK;
  hot_gctx_t g;
  dncall_t *dn;

  /* Initialize global state if not done so already.
   */
  if (!global.initialized) {
    err = hot_ens_Init(jops->argv, 
		       (jops->thread_mode == HOT_ENS_THREAD_SINGLE) ?
		       HOT_ENS_INIT_DONT_START : 
		       HOT_ENS_INIT_SEPARATE_THREAD);
    if (err != HOT_OK)
      return err;
  }
  
  begin_critical();    
  
  /* Consistency check:  All groups should be joined with the same
   * thread mode.
   */
  if (global.thread_mode != jops->thread_mode) {
    end_critical();
    return hot_err_Create(0, "hot_ens_Join:  Can not mix multi-threaded and single-threaded mode");
  }

  /* Consistency check:  Synchronous downcalls (join/leave) are not
   * compatible with single-threaded execution mode.
   */
  if (jops->dncall_mode == HOT_ENS_DNCALL_SYNC &&
      jops->thread_mode == HOT_ENS_THREAD_SINGLE) {
    end_critical();
    return hot_err_Create(0, "hot_ens_Join: Synchronous downcalls are not compatible with single-threaded mode");
  }
  
  /* Allocate a new group context 
   * Initialize the group record.
   */
  g = alloc_gctx();
  g->joining = 1 ;
  g->leaving = 0 ;
  g->conf = jops->conf;
  g->env = jops->env;
  g->dncall_mode = jops->dncall_mode;
  g->vs.members = NULL;
  g->vs.nmembers = 0;
  g->join_sema = NULL ;
  g->leave_sema = NULL ;
  
  /* Enqueue the join request.
   */
  dn = alloc_dn(g,DNCALL_JOIN);

  dn->jops = (hot_ens_JoinOps_t*) hot_malloc(&ch_joinops, sizeof(*dn->jops)) ;
  *dn->jops = *jops ;

  /* We actually set the "return" value here so that
   * it is available when the accepted view call arrives.
   * This is somewhat of a hack.
   */
  *gctxp = g;
  
  /* Block on the semaphore until the first view is delivered.
   */
  if (g->dncall_mode == HOT_ENS_DNCALL_SYNC) {
    err = global.thr->sema_create(0, &g->join_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err)) ;
      
    end_critical();
    err = global.thr->sema_dec(g->join_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    begin_critical();

    err = global.thr->sema_destroy(g->join_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    g->join_sema = NULL ;
  }
  
  end_critical();
  return HOT_OK;
}

/* Leave a group.  This should be the last call made to a given gctx.
 * No more events will be delivered to this gctx after the call
 * returns.  
 */
hot_err_t hot_ens_Leave(hot_gctx_t g) {
  hot_err_t err = HOT_OK;

  begin_critical();

  g->leaving = 1 ;
  
  /* Enqueue the leave request.
   */
  (void) alloc_dn(g, DNCALL_LEAVE);
  
  if (g->dncall_mode == HOT_ENS_DNCALL_SYNC) {
    /* Temporarily release the global lock while we block on the
     * semaphore until the exit event is delivered.  
     */
    assert(!g->leave_sema) ;
    err = global.thr->sema_create(0, &g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err)) ;

    end_critical();
    err = (*global.thr->sema_dec)(g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
    begin_critical();

    err = global.thr->sema_destroy(g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err)) ;
    g->leave_sema = NULL ;

    release_gctx(g);
  }
  
  end_critical();

  return HOT_OK;
}

/* Send a multicast message to the group.
 */
hot_err_t hot_ens_Cast(
         hot_gctx_t g,
	 hot_msg_t msg, 
	 hot_ens_MsgSendView *send_view	/*OUT*/
) {
  dncall_t *dn;
  hot_err_t err;

  begin_critical() ;
  if (send_view != NULL) {
    *send_view = (g->group_blocked) ?
      HOT_ENS_MSG_SEND_NEXT_VIEW : HOT_ENS_MSG_SEND_CURRENT_VIEW ;
  }
  
  /* Enqueue the cast request.
   */
  dn = alloc_dn(g,DNCALL_CAST);
  
  err = hot_msg_Alias(msg, &dn->msg);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  end_critical() ;
  return HOT_OK ;
}

/* Send a point-to-point message to the specified group member.
 */
hot_err_t hot_ens_Send(
        hot_gctx_t g, 
	hot_endpt_t *dest,
	hot_msg_t msg,
	hot_ens_MsgSendView *send_view /*OUT*/
) {
  dncall_t *dn;
  hot_err_t err;

  begin_critical() ;

  if (send_view != NULL) {
    *send_view = (g->group_blocked) ?
      HOT_ENS_MSG_SEND_NEXT_VIEW : HOT_ENS_MSG_SEND_CURRENT_VIEW ;
  }
  
  /* Enqueue the send request.
   */
  dn = alloc_dn(g,DNCALL_SEND);
  dn->dest = *dest;
  err = hot_msg_Alias(msg, &dn->msg);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  end_critical() ;
  return HOT_OK ;
}

/* Report group members as failure-suspected.
 * 
 * NB:  In the initial implementation, this downcall will not be supported.
 *      (if invoked, an exeption will be raised by OCAML).
 */
hot_err_t hot_ens_Suspect(
        hot_gctx_t gctx,
	hot_endpt_t *suspects, 
	int nsuspects
) {
  dncall_t *dn;
  int size ;

  begin_critical();    
  
  /* Enqueue the suspect request.
   */
  dn = alloc_dn(gctx,DNCALL_SUSPECT);
  
  size = sizeof(hot_endpt_t) * nsuspects ;
  dn->suspects = (hot_endpt_t*) hot_malloc(&ch_endpt_array,size);
  memcpy(dn->suspects, suspects, size);
  dn->nsuspects = nsuspects;
  
  end_critical();
  return HOT_OK;
}

/* Request a protocol change.
 */
hot_err_t hot_ens_ChangeProtocol(
        hot_gctx_t gctx,
	char *protocol
) {
  dncall_t *dn;

  begin_critical();

  /* Enqueue the ChangeProtocol request.
   */ 
  dn = alloc_dn(gctx, DNCALL_PROTOCOL);

  assert(!dn->protocol) ;
  dn->protocol = hot_malloc(&ch_protocol,HOT_ENS_MAX_PROTO_NAME_LENGTH) ;
  if (strlen(protocol) >= sizeof(dn->protocol))
    hot_sys_Panic("hot_ens_AddChangeProperties: properties too large") ;
  strcpy(dn->protocol, protocol);

  end_critical();
  return HOT_OK;
}

/* Request a protocol change specifying properties.
 */
hot_err_t hot_ens_ChangeProperties(
        hot_gctx_t gctx,
	char *properties
) {
  dncall_t *dn;
  assert(properties) ;

  begin_critical();

  /* Enqueue the ChangeProperties request.
   */ 
  dn = alloc_dn(gctx,DNCALL_PROPERTIES);

  assert(!dn->protocol) ;
  dn->protocol = hot_malloc(&ch_protocol,HOT_ENS_MAX_PROTO_NAME_LENGTH) ;
  if (strlen(properties) >= sizeof(dn->protocol))
    hot_sys_Panic("hot_ens_AddChangeProperties: properties too large") ;
  strcpy(dn->protocol, properties);

  end_critical();
  return HOT_OK;
}

/* Request a new view to be installed.
 */
hot_err_t hot_ens_RequestNewView(
        hot_gctx_t gctx
) {
  dncall_t *dn;
  begin_critical();
    
  /* Enqueue the ChangeProtocol request.
   */
  dn = alloc_dn(gctx,DNCALL_PROMPT);
    
  end_critical();
  return HOT_OK;
}


/*######################### Callback Dispatchers ############################*/


/* Callback dispatchers:
 *
 * Callback dispatchers are invoked from OCAML. The arguments are extracted and 
 * converted into C types, and the user callback function is invoked.
 *
 * NB:  Conventions for messages:  If a message is to be used outside of
 *      a callback, a copy of it (not alias!) must be created.  The
 *      reason is, a message in an upcall is created using the passed
 *      ocaml string, without copying.  After the upcall returns, the
 *      message string can be garbage collected by ocaml.
 * 
 *      For uniformity, the view structure should also not be used
 *      outside of the callback.  The struct. is allocated and released
 *      by the corresp. dispatcher function (hot_ens_AcceptedView_cbd).  
 *      However, the user callback function can steal the array of members by 
 *      setting the value of that field (in the view-state struct) to NULL.
 */ 

/* Got a multicast message. 
 */
value hot_ens_ReceiveCast_cbd(value gctx_v, value origin_v, value msg_v) {
  hot_endpt_t origin;
  hot_msg_t msg;
  hot_err_t err;
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_ReceiveCast_cback receive_cast;
  
  begin_critical();
  env = gctx->env;
  receive_cast = gctx->conf.receive_cast;
  end_critical();
  
  Endpt_val(origin_v, &origin);
  msg = hot_ens_Msg_of_val(msg_v);
  
  if (receive_cast != NULL) {
    (*receive_cast)(gctx, env, &origin, msg);
  }
  
  err = hot_msg_Release(&msg);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  return Val_unit;
}

/* Got a point-to-point message.
 */
value hot_ens_ReceiveSend_cbd(
        value gctx_v,
	value origin_v, 
	value msg_v
) {
  hot_endpt_t origin;
  hot_msg_t msg;
  hot_err_t err;
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_ReceiveSend_cback receive_send;
  
  begin_critical();
  env = gctx->env;
  receive_send = gctx->conf.receive_send;
  end_critical();
  
  Endpt_val(origin_v, &origin);
  msg = hot_ens_Msg_of_val(msg_v);
  
  if (receive_send != NULL) {
    (*receive_send)(gctx, env, &origin, msg);
  }
  
  err = hot_msg_Release(&msg);
  if (err != HOT_OK)
    hot_sys_Panic(hot_err_ErrString(err));
  
  return Val_unit;
}

/* Accepted a new view.
 */
value hot_ens_AcceptedView_cbd(	/* ML */
        value gctx_v,
	value view_state_v
) {
  hot_view_state_t view_state;
  hot_err_t err;
  hot_gctx_t g = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_AcceptedView_cback accepted_view;
  
  begin_critical();
  env = g->env;
  accepted_view = g->conf.accepted_view;
  
  /* Release the old view if necessary.
   */
  if (g->joining) {
    if ((g->vs.members != NULL) ||
	(g->vs.nmembers != 0))
      hot_sys_Panic("hot_ens_AcceptedView_cbd: internal inconsistency");
  } else {
    if ((g->vs.members == NULL) ||
	(g->vs.nmembers == 0))
      hot_sys_Panic("hot_ens_AcceptedView_cbd: internal inconsistency");
    release_endpt_array(&ch_endpt_array,g->vs.members,g->vs.nmembers) ;
  }
  
  /* The group is unblocked now.
   */
  g->group_blocked = 0;
  
  /* Setup the new view.
   */
  ViewState_of_val(view_state_v, &g->vs); 
  ViewState_of_val(view_state_v, &view_state); 

  end_critical();

  if (accepted_view != NULL) {
    (*accepted_view)(g, env, &view_state);
  }
  
  /* If this is the first view for the given gctx, unblock the join dncall.
   */
  begin_critical();

  if (view_state.members != NULL) {
    release_endpt_array(&ch_endpt_array,view_state.members,view_state.nmembers) ;
  }

  if (g->joining &&
      g->dncall_mode == HOT_ENS_DNCALL_SYNC) {
    assert(g->join_sema) ;
    if ((err = (*global.thr->sema_inc)(g->join_sema)) != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
  g->joining = 0 ;
  
  end_critical();
  
  return Val_unit;
}

/* Got a heartbeat event.
 */
value hot_ens_Heartbeat_cbd(
        value gctx_v,
        value rate_v
) {
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_Heartbeat_cback heartbeat;
  
  begin_critical();
  env = gctx->env;
  heartbeat = gctx->conf.heartbeat;
  end_critical();
  
  if (heartbeat != NULL) {
    (*heartbeat)(gctx, env, Rate_val(rate_v));
  }
  
  return Val_unit;
}

/* The group is about to block.
 */
value hot_ens_Block_cbd(value gctx_v) {
  hot_gctx_t gctx = hot_ens_Gctx_of_val(gctx_v);
  void *env;
  hot_ens_Block_cback block;
  
  /* Set the "group_blocked" flag in the corresp. gctx.
   * The flag is cleared when the next view is delivered.
   */
  begin_critical();
  env = gctx->env;
  block = gctx->conf.block;
  gctx->group_blocked = 1;
  end_critical();
  
  if (block != NULL) {
    (*block)(gctx, env);
  }

  return Val_unit;
}

/* The member has left the group.
 */
value hot_ens_Exit_cbd(value gctx_v) {
  hot_gctx_t g = hot_ens_Gctx_of_val(gctx_v);
  hot_ens_Exit_cback exit_cb;
  hot_err_t err;
  void *env;

  /* Phase 1: get info for callback.
   */
  begin_critical();  
  
  if (!g->leaving)
    hot_sys_Panic("hot_ens_Exit_cbd: mbr state is not leaving");

  env = g->env;
  exit_cb = g->conf.exit;
  
  end_critical();

  if (exit_cb != NULL) {
    (*exit_cb)(g, env);
  }  
  
  /* Phase 2: cleanup.
   */
  begin_critical();  

  /* If leave downcall is asynchronous, release gctx now.
   */
  if (g->dncall_mode == HOT_ENS_DNCALL_ASYNC) {
    release_gctx(g);
  } else {
    /* If leave is synchronous, unblock the leave downcall for the
     * specified gctx.  The gctx is recycled in the leave dncall.  
     */
    assert(g->leave_sema) ;
    err = (*global.thr->sema_inc)(g->leave_sema) ;
    if (err != HOT_OK)
      hot_sys_Panic(hot_err_ErrString(err));
  }
  
  end_critical();  
  
  return Val_unit;
}

/************************** C/Ensemble communication ****************************/

/* Mainloop-status values.
 * The enum type definition must match the corresponding ML type def.:
 *
 * type mainloop_status = 
 * | Active
 * | About_to_block
 */
enum hot_ens_mainloop_status {
    HOT_ENS_MAINLOOP_STATUS_ACTIVE,
    HOT_ENS_MAINLOOP_STATUS_ABOUT_TO_BLOCK
};

/* Return true iff there are pending downcalls.
 */
value hot_ens_DncallsPending(value mainloop_status_v) {
  int pending ;
  
  begin_critical() ;
  
  /* Ensemble blocks if it was about to do it and there were no pending
   * downcalls to give it something to do.
   */
  global.ensemble_blocked = 
    ((Int_val(mainloop_status_v) == HOT_ENS_MAINLOOP_STATUS_ABOUT_TO_BLOCK) && !global.dn_tail) ;
  
  pending = (global.dn_tail != NULL) ;
  
  /* If we are going to tell Ensemble there are pending downcalls, we know 
   * an alarm will be scheduled.
   */ 
  if (pending)
    global.alarm_scheduled = 1 ;
  
  end_critical() ;

  return Val_int(pending) ;
}

/* Return the array of pending downcalls.
 */
/* unit -> c_dncall_t array */
value hot_ens_GetDncalls(void) {
  value ret,tmp ;
  int i, n ;
  dncall_t *d, *next ;
  
  Push_roots(r, 1);
#define dn_v r[0]
  
  begin_critical();
 
  for (n=0, d=global.dn_tail ; d ; n++, d=d->next) ;
  
  dn_v = hot_ml_alloc_tuple("hot_ens_GetDncalls", n) ;

  /* Initialize array because we're going to be allocating
   * soon.
   */
  for (i=0;i<n;i++)
    Field(dn_v,i) = Val_unit ;
  
  for (i = 0, d = global.dn_tail ; 
       d ;
       i++, d = next) {
    assert(i<n) ;
    tmp = Val_of_dn(d) ;
    Field(dn_v,(n-i-1)) = tmp ;
    next = d->next ;
    d->next = NULL ;
    release_dn(d) ;
  }
  assert(i==n) ;
  global.dn_tail = NULL ;
  global.alarm_scheduled = 0 ;
  
  end_critical();

  ret = dn_v ;
#undef dn_v
  Pop_roots() ;
  
  return ret ;
}

/* Return the read fd of the C<->Ensemble communication pipe.
 * The pipe is opened in the initialization function, before starting Ensemble.
 */
inline value hot_ens_OpenAppPipe(void) {
  /*printf("request for pipe id %d\n", global.fd[0]);*/
  return Val_int(global.fd[0]);
}

#ifdef WIN32

void
Perror(char *s) {
  char error[256];
  int errno;
  
  errno = GetLastError();
  (void) FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM ,
		       NULL,	// message format
		       errno,
		       0,		// language
		       error,
		       256,		// buffer size, ignored for ALLOC_BUF
		       NULL	// args for msg format
		       );
  printf("%s: (%d) %s\n", s, errno, error);
  //LocalFree(error);
}

socketpair(
        int af,
        int type,
        int protocol, 
	int fd[2]
) {
  struct sockaddr_in sin;
  int result, sinlen;
  WSADATA wsaData;
  SOCKET	s0, s1;
  
  (void) WSAStartup(MAKEWORD(1, 0), &wsaData);
  
  s0 = socket(AF_INET, SOCK_DGRAM, 0);
  if (s0 < 0) {
    Perror("Socket0");
    return -1;
  }
  
  s1 = socket(AF_INET, SOCK_DGRAM, 0);
  if (s1 < 0) {
    Perror("socket1");
    closesocket(s0);
    return -1;
  }
  
  sin.sin_family      = AF_INET;
  sin.sin_addr.s_addr = ntohl(INADDR_LOOPBACK);
  sin.sin_port        = ntohs(0);
  
  result = bind(s0, (struct sockaddr *)&sin, sizeof(sin));
  if (result < 0) {
    Perror("Bind");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  sinlen = sizeof(sin);
  result = getsockname(s0, (struct sockaddr *)&sin, &sinlen);
  if (result < 0) {
    Perror("getsockname");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  result = connect(s1, (struct sockaddr *)&sin, sinlen);
  if (result < 0) {
    Perror("connect");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }   
  
  if ((fd[0] = _open_osfhandle(s0,0)) == -1) {
    Perror("open_osfhandle s0");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  if ((fd[1] = _open_osfhandle(s1,0)) == -1) {
    Perror("open_osfhandle s1");
    closesocket(s0);
    closesocket(s1);
    return -1;
  }
  
  return 0;
}

#endif
