(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
let name = "GCTX"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
open Util
open Trans
(**************************************************************)

type id = int
type endpt = string
type msg = string


(* These options are passed with application's join() downcalls.
 *)
type ens_join_options = {
  jops_hrtbt_rate	: float ;
  jops_transports	: string ;
  jops_protocol	        : string ;
  jops_group_name	: string ;
  jops_properties	: string ;
  jops_use_properties	: bool ;
  jops_groupd	        : bool ;
  jops_params	        : string ;
  jops_debug            : bool
}

(* Types of C application's downcalls to Ensemble *)
type dncall = 
  | C_Join of ens_join_options 
  | C_Cast of msg
  | C_Send of endpt * msg
  | C_Suspect of endpt array 
  | C_Protocol of string
  | C_Properties of string
  | C_Leave of unit
  | C_Prompt of unit
  | C_Void of unit

type c_dncall = {
  gid	: id;
  dncall : dncall 
}

(* This describes a group member ("group context").
 * The other part of group-context struct is at the C appl. side.
 *)
type gctx = {
  id	                : id ;		(* index in the gctx array *)
  dncalls	        : dncall Queuee.t ; (* pending application downcalls *)
  mutable hb_requested  : bool ;
  mutable vs            : View.state ;
  mutable ls            : View.local ;
  mutable sview	        : endpt array ;	(* current group view *)
  mutable exited 	: bool		(* set after exit event *)
}

(**************************************************************)
(* Use an integer based hashtable to keep track of 
 * members.
 *)

module Int_key =
  struct
    type t = int
    let equal = (==)
    let hash i = abs (i lsr 1)
  end

module Hashtbl = Hashtbl.Make ( Int_key )

let active = Hashtbl.create 10
    
(**************************************************************)

let add id f = 
  try 
    Hashtbl.find active id ;
    failwith "initialize:already in table"
  with Not_found ->
    Hashtbl.add active id f

let get id dc =
  try
    let g = Hashtbl.find active id in
    g dc
  with Not_found ->
    failwith "get:not found in table"

let destroy debug id =
  try
    (* Ensure that the link was in the table.
     *)
    let _ = Hashtbl.find active id in
    
    (* Then remove it.
     *)
    Hashtbl.remove active id
  with Not_found ->
    failwith "destroy:not found in table"
	
(* Print the group state of a gctx.
 *)
let string_of_gctx gctx = 
  sprintf "gctx = %d, exited = %b\n" gctx.id gctx.exited
    
(* Print the state of a gctx record.
 *)
let print s = 
  eprintf "%s\n" (string_of_gctx s)

(* Print states of all gctx records.
 *)
(*
let print_all_gctx () = 
  Hashtbl.iter (fun _ g -> print g) active
*)

(**************************************************************)
