(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* SESSION.ML: session module *)
(* Author: Takako M. Hickey, 4/97 *)
(**************************************************************)
open Ensemble
open Db
(**************************************************************)
type sess_sig =
| SessSigKill
| SessSigSuspend
| SessSigResume

type sess_opname =
| SeCreate
| SeWait
| SeSig
| SeGetState
| SeDestroy
| SeNone
| PrCreate
| PrWait
| PrSig
| PrGetState
| PrNone
| DBGet
| DBDelete
| DBChange
| DShut

type sessproc = {
    seprocname		: string ;
    seprogram		: string ;
    seenv		: string array ;
    seprocsvr		: Ensemble.Endpt.id 
}

type sessreq = {
    reqid		: int ;
    sessop 		: sess_opname ;
    clientrid		: Rpc.Sockio.request_id ;
    mutable ntowait	: int ;
    mutable proctoreturn: sessproc list 
}

(* TODO: add client authorization list *)
type session = {
    sessname		: string ;
    mutable proc_ticket	: int ;
    mutable req_ticket	: int ;
    mutable processes	: sessproc list ;
    mutable requests	: sessreq list 
}

type sesslist = {
    mutable ticket : int ;
    mutable sessions : session list 
}

type command =
| Noop of unit
| Cmd of string

type sessop_flag =
| AllOrNothing of unit
| MultLimit of int
| Unlimited of unit

type sess_op =
| SessCreate of string
| SessWait of string
| SessSig of string * sess_sig
| SessDestroy of string

| SessCreateSuccess of string * string
| SessWaitSuccess of string
| SessSigSuccess of string * sess_sig
| SessDestroySuccess of string
| SessOpFailure of string * sess_opname * string

| ProcCreate of string * command array array * string array
	* (Db.dbop * Db.attrval) list array * (string * Db.dbval) list * sessop_flag
| ProcWait of string * string
| ProcSig of string * string * sess_sig

| ProcCreateSuccess of string * sessproc array
| ProcWaitSuccess of string * string
| ProcSigSuccess of string * string * sess_sig
| ProcOpFailure of string * sess_opname * string
| OpFailure of string

(* administrative operations *)
(* TODO: should do security check *)
| DBGetEntry of Db.attrval
| DBDeleteEntry of Db.attrval
| DBChangeAttributes of Db.attrval * Db.attrval list
| Shutdown of unit

| DBGetEntryReply of Db.attrval * Db.dbentry
| DBDeleteEntryOk of Db.attrval
| DBChangeAttributesOk of Db.attrval
| ShutdownOk of unit


let my_hostname = Unix.gethostname ()

(**************************************************************)
let sesslist_create () = (
  {
    ticket = 0 ;
    sessions = []
  }
)

let sess_create l hint = (
  l.ticket <- l.ticket + 1 ;
  let name = hint ^ "." ^ (string_of_int l.ticket) in
  let sess = {
    sessname = name ;
    proc_ticket = 0 ;
    req_ticket = 0 ;
    processes = [] ;
    requests = [] 
  } in
  l.sessions <- l.sessions @ [sess] ;
  sess
)

let sess_destroy l name = (
  let rec loop = function
    | [] -> raise (Failure("non existent session"))
    | h::t ->
        if h.sessname = name then (
          if h.processes <> [] then
            raise (Failure("session has active processes")) ;
          t
        )
        else
          h::(loop t)
  in
    l.sessions <- loop l.sessions
)

let sess_lookup l name = (
  let rec loop = function
    | [] -> raise Not_found
    | h::t ->
        if h.sessname = name then
          h
        else
          loop t
  in
    loop l.sessions
)

(**************************************************************)
let sess_processes_clear l sessname = (
  let s = sess_lookup l sessname in
  s.processes <- []
)

let sess_proc_add l sessname proc = (
  let s = sess_lookup l sessname in
  s.processes <- s.processes @ [proc] ;
  s
)

let sess_proc_remove l sessname procname = (
  let s = sess_lookup l sessname in
  let rec loop = function
    | [] -> []
    | h::t ->
        if h.seprocname = procname then
          t
        else
          h :: (loop t)
  in
    s.processes <- (loop s.processes) ;
)

let sess_proc_lookup l sessname procname = (
  let s = sess_lookup l sessname in
  let rec loop = function
    | [] -> raise Not_found
    | h::t ->
        if h.seprocname = procname then
          h
        else
          loop t
  in
    loop s.processes
)

(**************************************************************)
let sess_req_add l sessname sessop rid n = (
  let s = sess_lookup l sessname in
  let n =
    if n < 0 then
      (List.length s.processes)
    else
      n
  in
  s.req_ticket <- s.req_ticket + 1 ;
  let req = {
    reqid = s.req_ticket ;
    sessop = sessop ;
    clientrid = rid ;
    ntowait = n ;
    proctoreturn = []
  } in
  s.requests <- s.requests @ [req] ;
  s
)

let sess_req_remove l sessname reqid = (
  let s = sess_lookup l sessname in
  let rec loop = function
    | [] -> []
    | h::t ->
        if h.reqid = reqid then
          t
        else
          h :: (loop t)
  in
    s.requests <- (loop s.requests) ;
)

let sess_req_lookup l sessname reqid = (
  let s = sess_lookup l sessname in
  let rec loop = function
    | [] -> raise Not_found
    | h::t ->
        if h.reqid = reqid then
          h
        else
          loop t
  in
    loop s.requests
)

(**************************************************************)

(* Lookup session.  Increment process ticket before returning.
 *)
let sess_lookup_incticket l sessname n = (
  let s = sess_lookup l sessname in
  s.proc_ticket <- s.proc_ticket + n ;
  s
)

let sess_req_lookup_decntowait l sessname reqid = (
  let r = sess_req_lookup l sessname reqid in
  r.ntowait <- r.ntowait - 1 ;
  r
)

let sess_req_lookup_decntowait_addproc l sessname reqid proc = (
  let r = sess_req_lookup l sessname reqid in
  r.ntowait <- r.ntowait - 1 ;
  r.proctoreturn <- r.proctoreturn @ [proc] ;
  r
)
