(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* RSESSION.ML: replicated session module *)
(* Author: Takako M. Hickey, 4/97 *)
(**************************************************************)
open Repl
open Session
(**************************************************************)
type read_request
  = SeLOOKUP of string * int * (session -> unit) * error
  | PrLOOKUP of string * string * (sessproc -> unit) * error
  | RqLOOKUP of string * int * (sessreq -> unit) * error
  | RqLOOKUPw of string * int * (sessreq -> unit) * error
  | RqLOOKUPwp of string * int * sessproc * (sessreq -> unit) * error

type update_request
  = SeCREATE of string * (session -> unit) * error
  | SeDESTROY of string * (unit -> unit) * error
  | PrCLEAR of string * (unit -> unit) * error
  | PrADD of string * sessproc * (session -> unit) * error
  | PrREMOVE of string * string * (unit -> unit) * error
  | RqADD of string * sess_opname * Rpc.Sockio.request_id * int
	* (session -> unit) * error
  | RqREMOVE of string * int * (unit -> unit) * error

(**************************************************************)
(* Create an instance of a rsession.
 *)
let s init_state =
  let state = match init_state with
    | None -> sesslist_create ()
    | Some s -> s
  in
  let read = function
    | SeLOOKUP(sessname, n, reply, error) ->
	(try reply(sess_lookup_incticket state sessname n)
		with _ -> error "sess lookup failed")
    | PrLOOKUP(sessname, procname, reply, error) ->
	(try reply(sess_proc_lookup state sessname procname)
		with _ -> error "proc lookup failed")
    | RqLOOKUP(sessname, reqid, reply, error) ->
	(try reply(sess_req_lookup state sessname reqid)
		with _ -> error "req lookup failed")
    | RqLOOKUPw(sessname, reqid, reply, error) ->
	(try reply(sess_req_lookup_decntowait state sessname reqid)
		with _ -> error "req lookup_w failed")
    | RqLOOKUPwp(sessname, reqid, proc, reply, error) ->
	(try reply(sess_req_lookup_decntowait_addproc state sessname reqid proc)
		with _ -> error "req lookup_wp failed")
  in
  let update = function
    | SeCREATE(hint, reply, error) ->
	(try reply(sess_create state hint)
		with _ -> error "sess create failed")
    | SeDESTROY(sessname, reply, error) ->
	(try reply(sess_destroy state sessname)
		with _ -> error "sess create failed")
    | PrCLEAR(sessname, reply, error) ->
	(try reply(sess_processes_clear state sessname)
		with _ -> error "proc clear failed")
    | PrADD(sessname, proc, reply, error) ->
	(try reply(sess_proc_add state sessname proc)
		with _ -> error "proc add failed")
    | PrREMOVE(sessname, procname, reply, error) ->
	(try reply(sess_proc_remove state sessname procname)
		with _ -> error "proc remove failed")
    | RqADD(sessname, sessop, rid, n, reply, error) ->
	(try reply(sess_req_add state sessname sessop rid n)
		with _ -> error "req add failed")
    | RqREMOVE(sessname, reqid, reply, error) ->
	(try reply(sess_req_remove state sessname reqid)
		with _ -> error "req remove failed")
  in
  let destroy reply _ = reply() in
  let get_state reply _ = reply state in
  { read = read; update = update; destroy = destroy; get_state = get_state }

(**************************************************************)
(* Install the queue constructor under the name "rsession".
 *)
let _ =
  Repl.install "rsession" s

(**************************************************************)
(* This is the interface to the rsession.
 *)
let rsess_create upcall error =
  Repl.create (Repl.got_request Repl.got_reply) "rsession" upcall error

let rsess_lookup_incticket s sessname n upcall error =
  s.read (SeLOOKUP(sessname, n, upcall, error))
let rsess_proc_lookup s sessname procname upcall error =
  s.read (PrLOOKUP(sessname, procname, upcall, error))
let rsess_req_lookup s sessname reqid upcall error =
  s.read (RqLOOKUP(sessname, reqid, upcall, error))
let rsess_lookup_decntowait s sessname reqid upcall error =
  s.read (RqLOOKUPw(sessname, reqid, upcall, error))
let rsess_req_lookup_decntowait_addproc s sessname reqid proc upcall error =
  s.read (RqLOOKUPwp(sessname, reqid, proc, upcall, error))

let rsess_session_create s hint upcall error =
  s.update (SeCREATE(hint, upcall, error))
let rsess_session_destroy s sessname upcall error =
  s.update (SeDESTROY(sessname, upcall, error))
let rsess_processes_clear s sessname upcall error =
  s.update (PrCLEAR(sessname, upcall, error))
let rsess_proc_add s sessname proc upcall error =
  s.update (PrADD(sessname, proc, upcall, error))
let rsess_proc_remove s sessname procname upcall error =
  s.update (PrREMOVE(sessname, procname, upcall, error))
let rsess_req_add s sessname sessop rid n upcall error =
  s.update (RqADD(sessname, sessop, rid, n, upcall, error))
let rsess_req_remove s sessname reqid upcall error =
  s.update (RqREMOVE(sessname, reqid, upcall, error))

let rsess_destroy s upcall error =
  s.destroy upcall error
