(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
open Printf
open Pack

type id = int

type error = string -> unit

type deliver = string -> unit

type response
  = RESPONSE of Obj.t
  | ERROR of string

type request =
  | CREATE of (id * string) * (unit -> unit) * error
  | DESTROY of id * (unit -> unit) * error
  | READ of (id * int * Obj.t) * (Obj.t -> unit) * error
  | UPDATE of (id * int * Obj.t) * (Obj.t -> unit) * error
  | GET_STATE of id * (Obj.t -> unit) * error

type ('r, 'u, 's) instance = {
  read: 'r -> unit;
  update: 'u -> unit;
  destroy: (unit -> unit) -> error -> unit;
  get_state: ('s -> unit) -> error -> unit
}

let implementations = ref []
let instances = ref []

let install name impl =
  implementations := (name, Obj.repr impl) :: !implementations

(* Take the given element from the list.  Return the element and the
 * list.
 *)
let rec take e = function
  | (a, b) :: tl ->
      if a = e then
	b, tl
      else
	let f, l = take e tl in
	f, ((a, b) :: l)
  | [] ->
      raise Not_found

let handle = function
  | CREATE((id, name), reply, error) ->
      begin try
	let impl = Obj.magic (List.assoc name !implementations) in
	let instance = Obj.repr (impl None) in
	instances := (id, instance) :: !instances;
	reply()
      with _ ->
	error "create failed"
      end
  | DESTROY(id, reply, error) ->
      begin try
	let (instance, rest) = take id !instances in
	instances := rest;
	(Obj.magic instance).destroy reply error
      with _ ->
	error "destroy failed"
      end
  | READ((id, tag, request), reply, error) ->
      begin try
	let instance = Obj.magic (List.assoc id !instances) in
	let upcall x = reply (Obj.repr x) in
	let error x = error (Obj.magic x) in
	instance.read (Obj.magic (pack tag request upcall error))
      with _ ->
	error "read failed"
      end
  | UPDATE((id, tag, request), reply, error) ->
      begin try
	let instance = Obj.magic (List.assoc id !instances) in
	let upcall x = reply (Obj.repr x) in
	let error x = error (Obj.magic x) in
	instance.update (Obj.magic (pack tag request upcall error))
      with _ ->
	error "update failed"
      end
  | GET_STATE(id, reply, error) ->
      begin try
	let instance = Obj.magic (List.assoc id !instances) in
	instance.get_state reply error
      with _ ->
	error "get_state failed"
      end

(* Outstanding requests.
 *)
let outstanding = ref []

let request_seqno = ref 0

let got_reply reply =
  let (o, _) = Obj.unmarshal reply 0 in
  let (rid, r) = Obj.magic o in
  let (upcall, error) = List.assoc rid !outstanding in
  match r with
    | RESPONSE r -> upcall r
    | ERROR r -> error r

let got_request send request =
  let (o, _) = Obj.unmarshal request 0 in
  let (rid, tag, req) = Obj.magic o in
  let upcall r =
    let o = Obj.repr (rid, RESPONSE r) in
    send (Obj.marshal o)
  in
  let error r =
    let o = Obj.repr (rid, ERROR (Obj.magic r)) in
    send (Obj.marshal o)
  in
  let request = pack tag req upcall error in
  handle (Obj.magic request)

let apply send request =
  let rid = !request_seqno in
  incr request_seqno;
  let (tag, req, upcall, error) = unpack (Obj.repr request) in
  let error x = error (Obj.magic x) in
  outstanding := (rid, (upcall, error)) :: !outstanding;
  let o = Obj.repr(rid, tag, req) in
  send (Obj.marshal o)

let id_no = ref 0
let make_new_id() =
  let id = !id_no in
  incr id_no;
  !id_no

let create send name upcall error =
  let id = make_new_id() in
  let destroy id upcall error = apply send (DESTROY(id, upcall, error)) in
  let read id request =
    let (tag, req, upcall, error) = Obj.magic (unpack (Obj.repr request)) in
    apply send (READ((id, tag, req), upcall, error))
  in
  let update id request =
    let (tag, req, upcall, error) = Obj.magic (unpack (Obj.repr request)) in
    apply send (UPDATE((id, tag, req), upcall, error))
  in
  let get_state id upcall error =
    let reply x = upcall (Obj.magic x) in
    apply send (GET_STATE(id, reply, error))
  in
  let response() = upcall {
    destroy = destroy id;
    read = read id;
    update = update id;
    get_state = get_state id
  } in
  apply send (CREATE((id, name), response, error))
