(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PROCESS.ML *)
(* Author: Takako M. Hickey, 4/97 *)
(**************************************************************)
open Ensemble
open Session
(**************************************************************)
type sess_or_proc =
| Sess
| Proc

type process = {
    psessname		: string ;
    procname		: string ;
    program		: string ;
    procenv		: string array ;
    procsvr		: Ensemble.Endpt.id ;

    handle              : Socket.process_handle ;
    starttime		: float ;
    mutable procrid	: Rpc.Sockio.request_id option ;
    mutable waitreqid	: int ;
    mutable waitoptype  : sess_or_proc ;
    mutable waitendpt	: Ensemble.Endpt.id option
}

let my_hostname = Unix.gethostname ()
let proc_exec_list = ref []
let proc_done_list = ref []

(**************************************************************)
(* Create a process entry and add to the execting process list.
 *)
let proc_exec_create sessname program env procsvr handle suffix = (
  let name = my_hostname ^ "." ^ suffix in
  let proc = {
    psessname = sessname ;
    procname = name ;
    program = program ;
    procenv = env ;
    procsvr = procsvr ;
    handle = handle ;
    starttime = 0.0 ;
    procrid = None ;
    waitreqid = 0 ;
    waitoptype = Proc ;
    waitendpt = None 
  } in
  proc_exec_list := !proc_exec_list @ [proc] ;
  proc
)
let proc_done_add proc =
  proc_done_list := !proc_done_list @ [proc]

(* Move a matching entry from exec list to done list, and return
 * the removed entry.
 *)
let proc_move_to_done handle = (
  let proc = ref None in
  let rec loop = function
  | [] -> []
  | h::t ->
      if h.handle = handle then (
        proc := Some h ;
        t
      )
      else
        h :: (loop t)
  in
    proc_exec_list := loop !proc_exec_list ;
    (match !proc with
     | Some p ->
         proc_done_list := !proc_done_list @ [p] ;
         p
     | None ->
         raise Not_found
    )
)

(* Remove a matching entry from a process list and return the entry.
 *)
let proc_remove sessname procname plist = (
  let proc = ref None in
  let rec loop = function
  | [] -> []
  | h::t ->
      if h.psessname = sessname & h.procname = procname then (
        proc := Some h ;
        t
      )
      else
        h :: (loop t)
  in
    plist := loop !plist ;
    (match !proc with
     | Some p -> p
     | None -> raise Not_found
    )
)
let proc_exec_remove sessname procname =
  proc_remove sessname procname proc_exec_list
let proc_done_remove sessname procname =
  proc_remove sessname procname proc_done_list

let rec proc_lookup sessname procname = function
  | [] -> raise Not_found
  | h::t ->
      if h.psessname = sessname & h.procname = procname then
        h
      else
        proc_lookup sessname procname t
let proc_exec_lookup sessname procname =
  proc_lookup sessname procname !proc_exec_list
let proc_done_lookup sessname procname =
  proc_lookup sessname procname !proc_done_list


let proc_list_print s plist = (
  let f proc =
    Util.printf "%s:%s " proc.psessname proc.procname
  in
    Util.printf "%s: [ " s ;
    List.map f plist ;
    Util.printf "]\n" ;
)
let proc_exec_list_print () =
  proc_list_print "proc_exec_list" !proc_exec_list
let proc_done_list_print () =
  proc_list_print "proc_done_list" !proc_done_list


(**************************************************************)
(* Used for shutdown.
 * TODO: send notification to client.
 *)
let kill_all_procs () = (
   let send_kill proc =
     Util.printf "killing process %s\n" proc.procname ;
     Socket.terminate_process proc.handle ;
   in
   List.map send_kill !proc_exec_list ;
   proc_done_list := !proc_done_list @ !proc_exec_list ;
   proc_exec_list := []
)

let toseprocess proc = (
  let seproc = {
    seprocname = proc.procname ;
    seprogram = proc.program ;
    seenv = proc.procenv ;
    seprocsvr = proc.procsvr
  } in
  seproc
)
