(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* DUTIL.ML: utility module *)
(* Author: Takako M. Hickey, 4/97 *)
(**************************************************************)
open Ensemble
open Session
open Db
(**************************************************************)
type svr_type =
| SvrSess
| SvrProc

type svr_state = {
  sttype                        : svr_type ;
  stname                        : string ;
  mutable stendpt               : Ensemble.Endpt.id option ;

  mutable staticentry           : dbentry ;
  mutable nrunning              : int ;
  mutable nwaiting              : int ;
  mutable ncompleted            : int ;
  mutable gossipid              : int ;
  mutable stload                : float ;
  mutable stuptime              : float ;

  mutable strestrictions        : dbrestrictions 
}

type execsvr_msg =
  | DSessWait of string * int * string
  | DSessSig of string * int * string * sess_sig

  | DSessWaitSuccess of string * int * string
  | DSessSigSuccess of string * int * string * sess_sig
  | DSessOpFailure of string * int * sess_opname * string


  | DProcCreate of string * int * string * string array * int
  | DProcWait of string * int * string
  | DProcSig of string * int * string * sess_sig * Rpc.Sockio.request_id

  | DProcCreateSuccess of string * int * sessproc
  | DProcWaitSuccess of string * int * string
  | DProcSigSuccess of string * int * string * sess_sig * Rpc.Sockio.request_id
  | DProcOpFailure of string * int * sess_opname * string

  | DDBDeleteEntry of attrval
  | DDBChangeAttributes of attrval * attrval list

  | DDBDeleteEntryOk of attrval
  | DDBChangeAttributesOk of attrval

  | DShutdown of unit
  | DUpQuery of unit
  | DUpAnswer of svr_type * string * float * float * int
  | DGossip of string * float * dbentry

(**************************************************************)
let convert_uptime time = (
  let remain = ref (Pervasives.truncate time) in
  let days = !remain / 86400 in
  remain := !remain - days * 86400 ;
  let hours = !remain / 3600 in
  remain := !remain - hours * 3600 ;
  let minutes = !remain / 60 in
  remain := !remain - minutes * 60 ;
  let seconds = !remain in
  (days, hours, minutes, seconds)
)

(**************************************************************)
let compose_dbentry st = st.staticentry @
  [("load", Float st.stload);
   ("uptime", Float st.stuptime);
   ("nrunning", Int st.nrunning);
   ("nwaiting", Int st.nwaiting);
   ("ncompleted", Int st.ncompleted);
   ("gossipid", Int st.gossipid)]

(**************************************************************)
let print_uptime svr machine time load n = (
  let (days, hours, minutes, seconds) = convert_uptime time in
  (match svr with
  | SvrSess ->
      Util.printf "sess %s: up: %d days, %d:%2d; load: %.2f nprocessed: %d\n"
          machine days hours minutes load n
  | SvrProc ->
      Util.printf "proc %s: up: %d days, %d:%2d; load: %.2f nexecd: %d\n"
          machine days hours minutes load n
  )
)

let print_msghdr msg = (match msg with
  | DSessWait(_, _, _) ->
      Util.printf "got DSessWait"  ;
  | DSessWaitSuccess(sessname, _, procname) ->
      Util.printf "got DSessWaitSuccess"  ;
  | DSessSig(sessname, _, procname, sesssig) ->
      Util.printf "got DSessSig"  ;
  | DSessSigSuccess(sessname, _, procname, sesssig) ->
      Util.printf "got DSessSigSuccess"  ;
  | DProcCreate(sessname, _, program, env, ticket) ->
      Util.printf "got DProcCreate"  ;
  | DProcCreateSuccess(sessname, _, proc) ->
      Util.printf "got DProcCreateSuccess"  ;
  | DProcWait(sessname, _, procname) ->
      Util.printf "got DProcWait"  ;
  | DProcWaitSuccess(sessname, _, procname) ->
      Util.printf "got DProcWaitSuccess"  ;
  | DProcSig(sessname, _, procname, sesssig, rid) ->
      Util.printf "got DProcSig"  ;
  | DProcSigSuccess(sessname, _, procname, sesssig, rid) ->
      Util.printf "got DProcSigSuccess"  ;
  | DUpQuery() ->
      Util.printf "got DUpQuery"  ;
  | DUpAnswer(svr, machine, time, load, nissued) ->
      Util.printf "got DUpAnswer"  ;
  | DShutdown () ->
      Util.printf "got DShutdown"  ;
  | DGossip(fromhost, time, changes) ->
      Util.printf "got DGossip"  ;
  | _ ->
      Util.printf "got something"  ;
)
