(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* DBEDIT.ML: Machine database editor *)
(* DBedit allows inspection and editing of the machine database
 * managed by the sessvr.  The procsvr gets its initial values from
 * the default database.  DBedit can be used later to change the value
 * of existing attributes or adding in new attribute value pairs.
 * Currently, all attributes are assumed to be of type string.
 * 
 * Supported commands are:
 *   g key=val                               get entry
 *   d key=val                               delete entry
 *   c key=val;attr1=val1;attr2=val2;...     change entry
 *   h                                       help
 *
 * TODO: add in security.
 *)
(* Author: Takako M. Hickey, 4/97 *)
(**************************************************************)
open Ensemble
open Hsys
open View
open Appl_intf 
open Str
open Rpc
open Db
open Session
(**************************************************************)
let name = Trace.source_file "DBEDIT" 
let failwith s = failwith (Util.failmsg name s) 
(**************************************************************)

and default_execsvr_hosts = "eclair0.cs.cornell.edu"
and default_execsvr_port = 8123
and server = (ref None: Rpc.Sockio.rpc_handle option ref)

(**************************************************************)

let init_rpc servername alarm = (
  let get_servernames () =
    let buffer =
      (try
         Sys.getenv "ENS_EXECSVR_HOSTS"
       with _ ->
         default_execsvr_hosts
      )
    in
      Array.of_list (Str.split (regexp "[:]+") buffer)
  in
  let get_serverport () =
    (try
      int_of_string (Sys.getenv "ENS_EXECSVR_PORT")
    with _ ->
      default_execsvr_port
    )
  in
  Sockio.ensemble_register (Alarm.add_sock alarm) (Alarm.rmv_sock alarm) ;
  let s = Sockio.bind_to_service servername in
  let l = (get_servernames ()) in
  let dp = (get_serverport ()) in
  for i = 0 to (Array.length l) - 1 do
    Sockio.add_to_service s l.(i) dp ; 
  done ;
  server := Some s
)

let select_loop interval = (
  try
    while true do
      Sockio.select interval
    done
  with Not_found ->
    Util.printf "got not found while selecting\n"
)

  let recv_failure reply =
    Util.printf "rpc failed" 

  let rec recv_reply reply = (
    (match (Obj.magic reply) with
    | DBGetEntryReply(key, entry) ->
        dbentry_print entry ;
    | DBDeleteEntryOk(key) ->
        Util.printf "got DBDeleteEntryOk\n" ;
    | DBChangeAttributesOk(key) ->
        Util.printf "got DBChangeAttributesOk\n" ;
    | _ -> ()
    ) ;
    Util.printf ">" ;
  )

let dbedit (ls,vs) my_hostname alarm =
  let buffer    = ref ""
  in


  (* Input reading code stole from mtalk application.
   *)
  (* This is set below in the handler of the same name.  Get_input
   * calls the installed function to request an immediate callback.
   *)
  let async = Appl.async (vs.group, ls.endpt) in

  (* Install handler to get input from stdin.
   *)
  let stdin = stdin () in
  let get_input () =
    let buf = String.create 1000 in
    let len = read stdin buf 0 (String.length buf) in
    if len = 0 then exit 0 ;                (* BUG:should leave group *)
    buffer := !buffer ^ (String.sub buf 0 len) ;
    async () 
  in
  Alarm.add_sock alarm stdin (Handler0 get_input) ;

  let print_help () = (
    Util.printf "Type one of the following commands:\n" ;
    Util.printf "\tg key=val\tget entry\n" ;
    Util.printf "\td key=val\tdelete entry\n" ;
    Util.printf "\tc key=val;attr1=val1;attr2=val2;...\tchange entry\n" ;
    Util.printf "\th\thelp\n" ;
  )
  in

  let parse_cmd cmd = (
     let l = Array.of_list (Str.split (regexp "[ \n]+") cmd) in
     if (Array.length l > 1) & (l.(0) <> "") & (l.(1) <> "") then (
       let avpairs  = ref [] in
       let plist = Array.of_list (Str.split (regexp "[;]+") l.(1)) in
       for i = 0 to (Array.length plist) - 1 do
         let entry = Array.of_list (Str.split (regexp "[=]+") plist.(i)) in
         if Array.length entry = 2 then (
(*
Util.printf "attr:'%s' value:'%s'\n" entry.(0) entry.(1) ;
*)
             avpairs := !avpairs @ [(entry.(0), String entry.(1))]
         ) ;
       done ;

       let key = List.hd !avpairs in
       avpairs := List.tl !avpairs ;
       let request = ref (DBGetEntry(key)) in
       let tosend = ref false in

       let c = (l.(0)).[0] in
       if c = 'g' then (
         request := DBGetEntry(key) ;
         tosend := true
       )
       else if c = 'd' then (
         request := DBDeleteEntry(key) ;
         tosend := true
       )
       else if c = 'c' then (
         request := DBChangeAttributes(key, !avpairs) ;
         tosend := true
       )
       else if c = 'h' then (
         print_help () ;
         Util.printf ">" ;
       )
       else (
         Util.printf "Unknown command %c\n" c ;
         print_help () ;
         Util.printf ">" ;
       ) ;
       if !tosend then
         (match !server with
         | Some s ->
             Util.printf "sending rpc request\n" ;
             Sockio.rpc s !request recv_reply recv_failure ; 
             async () 
         | None -> ()
         ) 
     )
     else (
       Util.printf ">" ;
     )
  )
  in

  let check_buf () =
    if !buffer <> "" then (
      let cmd = !buffer in
      buffer := "" ;
      parse_cmd cmd 
    ) ;
    []
  in

  let recv_cast from msg = []
  and recv_send from msg = []
  and block () = []

  and heartbeat _ =
     Sockio.sweep();
     check_buf() 

  and block_recv_cast from msg = ()
  and block_recv_send from msg = ()
  and block_view (ls,vs) = [ls.rank, ()]
  and block_install_view (ls,vs) ms = ()

  and unblock_view (ls,vs) _ = (
    Util.printf ">" ;
    []
  )

  and exit () =
    exit 0

  in full (*(Appl_intf.debug*) {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    heartbeat           = heartbeat ;
    heartbeat_rate	= Time.of_float 1.0 ;
    block               = block ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
  } (*)*)

(**************************************************************)

let run () =
  Sys.signal Sys.sigpipe Sys.Signal_ignore ;

  (*
   * Parse command line arguments.
   *)
  let undoc = "undocumented" in
  Arge.parse [
  ] (Arge.badarg name) "dbedit" ;

  (*
   * Get default transport and alarm info.
   *)
  let (ls,vs) = Appl.default_info "dbedit" in
  let alarm = Alarm.get () in

  let my_hostname = Unix.gethostname () in
  let interface = dbedit (ls,vs) my_hostname alarm in

  (*
   * Initialize the Horus protocol stack, using the
   * interface, transports, and group endpt chosen above.  
   *)
  Appl.config interface (ls,vs) ;

  init_rpc "execsvr" alarm ;

  (*
   * Enter a main loop
   *)
  Appl.main_loop ()
 (* end of run function *)


(* Run the application, with exception handlers to catch any
 * problems that might occur.  
 *)
let _ =
  Appl.exec ["dbedit"] run

(**************************************************************)
