(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* DB.ML: Database  *)
(* Author: Takako M. Hickey, 4/97 *)
(**************************************************************)
open Ensemble
(**************************************************************)

type dbop =
| DBeq
| DBlt
| DBgt
| DBlteq
| DBgteq
| DBinclude of string
| DBrandom of int
| DBmin of int			(* Expensive. Scans entire DB. *)

type dbtype =
| DBmachine
| DBprocess

type dbval =
| String of string
| Int of int
| Float of float
| Endpt of Ensemble.Endpt.id
| Addr of Unix.inet_addr
| StrList of string list
| Noval
type attrval = string * dbval

type dbentry = attrval list
type dbrestrictions = (dbop * dbtype * string * dbval) list

(*
let dbfile = "dbfile"
*)
let db = ref ([]: dbentry list)
let restrictions = ref ([]: (string * dbrestrictions) list)

(**************************************************************)
let printattrval (a, v) =
  Util.printf "%s: " a ;
  (match v with
  | String a -> Util.printf "String \"%s\" " a
  | Int a -> Util.printf "Int %d " a
  | Float a -> Util.printf "Float %f " a
  | Endpt a -> Util.printf "Endpt %s " (Endpt.string_of_id a)
  | _ -> ()
  )

let dbentry_print entry = 
  List.map printattrval entry ;
  Util.printf "\n" 

let db_print () = 
  let rec loop = function
  | [] -> ()
  | h::t ->
      dbentry_print h ;
      loop t 
  in loop !db

(*
let db_read () =
  (try
    let chanin = open_in dbfile in
    db := input_value chanin ;
    close_in chanin ;
    Util.printf "read db:\n";
    dbprint () ;
  with Sys_error _ -> db := [])

let db_write () =
  (try
    Util.printf "writing db:\n";
    dbprint () ;
    let chanout = open_out dbfile in
    output_value chanout !db ;
    close_out chanout ;
    flush chanout ;
  with Sys_error _ -> ())
*)


let rec getdbval ta = function
  | [] -> Noval
  | (a, v)::t ->
      if a = ta then
        v
      else
        getdbval ta t

let getrestrictions name =
  let rec loop = function
    | [] -> raise Not_found
    | (a, l)::t ->
        if a = name then
          l
        else
          loop t
  in
  loop !restrictions

(**************************************************************)
let do_comp op va vb = (match op with
| DBeq -> va = vb
| DBlt -> va < vb
| DBgt -> va > vb
| DBlteq -> va <= vb
| DBgteq -> va >= vb
| _ -> false
)

let db_compval op va vb = (match va with
| String a ->
    (match vb with
    | String b -> do_comp op a b
    | _ -> false)
| Int a ->
    (match vb with
    | Int b -> do_comp op a b
    | _ -> false)
| Float a ->
    (match vb with
    | Float b -> do_comp op a b
    | _ -> false)
| Endpt a ->
    (match vb with
    | Endpt b -> do_comp op a b
    | _ -> false)
| Addr a ->
    (match vb with
    | Addr b -> do_comp op a b
    | _ -> false)
| _ -> false
)

let db_incval item v = (match v with
  | StrList(l) ->
      if (List.mem item l) then
        true
      else
        false
  | _ -> false
)

(**************************************************************)
let restrictions_ok properties machine = (
  let valueok op va vb = (match op with
    | DBrandom(n) ->
        false
    | DBmin(n) ->
        false
    | DBinclude(t) ->
        (db_incval t va)
    | _ ->
        (db_compval op va vb)
  )
  in
  let rec loop = function
    | [] -> true
    | (op, dt, attr, valu)::t ->
        let v = (match dt with
         | DBmachine ->
             getdbval attr machine
         | DBprocess ->
             getdbval attr properties
        )
        in
        (match v with
         | Noval ->
             Util.printf "missing attribute %s\n" attr ;
             false
         | _ ->
            if (valueok op v valu) then
              loop t
            else
              false
        )
  in
  let name = getdbval "name" machine in
  (match name with
  | String n ->
      (try
        let r = (getrestrictions n) in
        loop r
      with Not_found ->
        (* This machine has no restrictions *)
        true)
  | _ ->
      Util.printf "machine missing name attribute\n" ;
      false
  )
)
    
let db_restrict properties machines =
  let rec restrict result = function
    | [] -> result
    | h::t ->
        if restrictions_ok properties h then
          restrict (h::result) t
        else
          restrict result t
  in
  restrict [] machines

(**************************************************************)

let rec db_compentry (op, (ta, tv)) = function
  | [] -> false
  | (a, v)::t ->
      if a = ta then (
        if (db_compval op v tv) then
          true
        else
          false
      )
      else
        db_compentry (op, (ta, tv)) t

let db_comp tav result mlist pass properties =
  let rec loop tav result = function
    | [] -> result
    | h::t ->
        if db_compentry tav h &
          (pass or (restrictions_ok properties h)) then
          loop tav (h::result) t
        else
          loop tav result t
  in
  loop tav result mlist

let db_include item ta result mlist pass properties =
  let rec db_inclentry item ta = function
    | [] -> false
    | (a, v)::t ->
        if a = ta then
          db_incval item v
        else
          db_inclentry item ta t
  in
  let rec loop item ta result = function
    | [] -> result
    | h::t ->
        if (db_inclentry item ta h) &
          (pass or (restrictions_ok properties h)) then
            loop item ta (h::result) t
        else
          loop item ta result t
  in
  loop item ta result mlist

let db_random n blist pass properties =
  if n > List.length blist then (
    Util.printf "not enough machines\n" ;
    blist
  )
  else (
    let barray = Array.of_list blist in
    let il = ref [] in
    let al = ref [] in
    for i = 0 to n - 1 do
      let found = ref false in
      let index = ref 0 in
      (* Only try up to trials times *)
      let trials = ref 5 in
      while not !found & !trials > 0 do
        index := Random.int (Array.length barray) ;
        if not (List.mem !index !il) &
          (pass or (restrictions_ok properties barray.(!index))) then (
          al := !al @ [barray.(!index)] ;
          il := !il @ [!index] ;
          found := true
        )
        else
          trials := !trials - 1
      done ;
    done ;
    !al
  )

let db_min n ta blist pass properties =
  let rec do_min n ta varray aarray = function
    | [] -> (Array.to_list aarray)
    | h::t ->
        let v = getdbval ta h in
        if (db_compval DBlt v varray.(0)) &
          (pass or (restrictions_ok properties h)) then (
          let vl = Array.append [|v|] (Array.sub varray 0 (Array.length varray)) in
          let al = Array.append [|h|] (Array.sub aarray 0 (Array.length aarray)) in
          do_min n ta vl al t
        )
        else
          do_min n ta varray aarray t
  in (
    let barray = Array.of_list blist in
    let found = ref false in
    let al = ref [] in
    let vl = ref [] in
    let i = ref 0 in
    let count = ref 0 in
    while not !found do
      if !i = List.length blist then
        found := true 
      else (
        if (pass or (restrictions_ok properties barray.(!i))) then (
          al := !al @ [barray.(!i)] ;
          vl := !vl @ [(getdbval ta barray.(!i))] ;
          count := !count + 1 ;
          if !count = n then 
            found := true ;
        )
      ) ;
      i := !i + 1 
    done ;
    if !i < Array.length barray then (
      let al = Array.of_list !al in
      let vl = (Array.of_list (Sort.list (db_compval DBlt) !vl)) in
      let len = (Array.length barray) - !i in
      let rl = Array.to_list (Array.sub barray !i len) in
      let al = do_min n ta vl al rl in
      al
    )
    else (
      !al
    )
  )

(* Do selection.  If flag is false, then check properties against
 * restrictions on the first selection.
 *)
let do_select_with_restrict avpairs flag properties = (
  let do_select (op, (ta, tv)) blist pass = (match op with
    | DBinclude(t) ->
        db_include ta t [] blist pass properties
    | DBrandom(n) ->
        db_random n blist pass properties
    | DBmin(n) ->
        db_min n ta blist pass properties
    | _ ->
        db_comp (op, (ta, tv)) [] blist pass properties
  )
  in
  let rec loop i result = function
    | [] -> result
    | h::t ->
        if flag = false & i = 0 then
          loop (i + 1) (do_select h result false) t
        else
          loop (i + 1) (do_select h result true) t
  in
    loop 0 !db avpairs
)

let db_select_with_restrict avpairs properties =
  do_select_with_restrict avpairs false properties

let db_select avpairs =
  do_select_with_restrict avpairs true []

let db_select_project sav pattr =
  let l = db_select sav in
  List.map (getdbval pattr) l 

(**************************************************************)

let db_restrictions_add name rlist =
  if rlist <> [] then
    restrictions := !restrictions @ [(name, rlist)] 

let db_restrictions_delete name =
  let rec filterout = function
    | [] -> []
    | (n, l)::t ->
        if n = name then
          t
        else
          (n, l)::(filterout t)
  in
    restrictions := filterout !restrictions 

let db_restrictions_lookup name =
  let rec loop = function
    | [] -> (name, [])
    | (n, l)::t ->
        if n = name then
          (n, l)
        else
          loop t
  in
    loop !restrictions 

(**************************************************************)

let db_add avlist =
  if avlist <> [] then
    db := !db @ [avlist] 


(* TODO: may want to support selection on all dbops. *)
let db_delete keyav =
  let rec filterout = function
    | [] -> []
    | h::t ->
        if db_compentry keyav h then
          t
        else
          h::(filterout t)
  in
    db := filterout !db 


(* Change the value associated with attribute for entry assoicated with
 * key.  If attribute does not exist, add.
 *)
let db_change keyav newavlist =
  let rec change (na, nv) = function
    | [] -> [(na, nv)]
    | (a, v)::t ->
        if a = na then
          (na, nv)::t
        else
          (a, v)::(change (na, nv) t)
  in
  let rec changeall result = function
    | [] -> result
    | h::t ->
        changeall (change h result) t
  in
  let rec find = function
    | [] ->
        if List.mem keyav newavlist then
          [newavlist]
        else
          [keyav::newavlist]
    | h::t ->
        if db_compentry (DBeq, keyav) h then
          (changeall h newavlist)::t
        else
          h::(find t)
  in
    db := find !db 

(**************************************************************)
let db_init i =
  Random.init i
