/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
#ifndef ENSEMBLE_MAESTRO_HH
#define ENSEMBLE_MAESTRO_HH

//
// $Id: EnsembleAD.hh,v 1.0 1997/02/08 15:03:47 sachdeva,atre Exp $
//
// Authors: Anand Atre, Anil Sachdeva.  atre@cs.cornell.edu, 
// anil@cs.cornell.edu.
// Copyright (c) 1997 by Anand Atre, Anil Sachdeva.  anand@cs.cornell.edu, 
// anil@cs.cornell.edu.
// All rights reserved.
//
// THIS IS FREE SOFTWARE.
// Permission to use, copy, modify, and distribute this software and 
// documentation in all settings is hereby granted, provided that no fee is 
// charged for this software and provided that this copyright notice appears 
// in all copies of any software which is or includes a copy or modification 
// of this software. In all advertising materials, documentation, and 
// publications mentioning features or use of this software you must credit 
// the author.
//
// This software is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of merchantability 
// or fitness for a particular purpose. 
//
//

#include "vmachine/Electra.hh"
#include "Maestro_Types.h"
#include "Maestro_CSX.h"

extern int gArgc;
extern char **gArgv;
extern electra_main(int, char **);
extern Maestro_CSX_Options gOps;
extern void quit_adaptors();

// the default stack associated with the default protocol policy:
//#define GOSSIP_STACK "Top:Heal:Switch:Leave:Inter:Intra:Elect:Merge:Sync:Suspect:Top_appl:Pt2pt:Frag:Stable:Mnak:Bottom"
//#define GROUPD_STACK "Top:Migrate:Top_appl:Frag:Pt2ptw:Mflow:Pt2pt:Stable:Mnak:Bottom"

// max. numbers of monitors per object reference:
const int MAX_MONITORS = 8;
const int MAX_GROUPNAME = 128;

// set the amount of time (seconds) we are willing to wait for the merge
// layer to find server objects:
const int VIEW_TIMEOUT=7;
const int VIEW_TIMEOUT_NOTIFY=3;

class RequestData;
class SenderIdent;

// EntityAddrLocal represents local information associated with a group
class EntityAddrLocal: public LocalData {
  friend class Ensemble_MaestroAD;
public:
  class Member : public Maestro_CSX {
    friend class EntityAddrLocal;
    friend class Ensemble_MaestroAD;
  public:
    Member(Maestro_CSX_Options &Ops, 
	   const char *Addr, EntityAddrLocal *Owner, 
	   const char *GroupName=0,
	   tFunc ViewChange=0, void *VcEnv=0, 
	   tFunc4 GetState=0, void *GsEnv=0, 
	   tFunc4 SetState=0, void *SsEnv=0);
    ~Member();
  protected:
    // Maestro callbacks:
    void csx_ReceiveScast_Callback(Maestro_EndpID &Origin, 
				   Maestro_Message &Msg);
    void csx_ReceiveSend_Callback(Maestro_EndpID &Origin, 
				  Maestro_Message &Msg);
    void stateTransfer_Callback(Maestro_XferID &XferID);
    void askState_Callback(Maestro_EndpID &Origin, Maestro_XferID &XferID,
			   Maestro_Message &RequestMsg);
    void csx_AcceptedView_Callback(Maestro_CSX_ViewData &ViewData,
				   Maestro_Message &Msg);
    void Print();
  private:
    Boolean           *receivedAcceptedView;    // necessary for viewTimeout
                                                // which takes possession of it
    tFunc             viewChange;                // view change upcall.
    void              *vcEnv;                    // (its parameter)
    tFunc4            getStateUpcall;            // get state upcall.
    void              *gsEnv;                    // (its parameter)
    tFunc4            setStateUpcall;            // set state upcall.
    void              *ssEnv;                    // (its parameter)
    Maestro_EndpList  oldServers;                // servers in previous view.
    Short             choiceMember;              // for Electra::ChoiceCall.
    Boolean           clientBlockedOnBind;
    Sema              clientBindViewSema;
    View              currView;   
    char              addr[MAX_GROUPNAME];       // the group that we belong to
    EntityAddrLocal   *owner;

    // these structures are required for passing arguments to Maestro threads
    // which can take a single void * argument only
    struct LocalThreadArgs {
      RequestData *RData;
      SenderIdent *SendId;
    };
    struct ViewTimeoutWrap {
      ViewTimeoutWrap(EntityAddrLocal::Member *Me, Boolean *RcvdAccView)
      { Memb = Me; RcvdAcceptedView = RcvdAccView; }
      EntityAddrLocal::Member *Memb;
      Boolean *RcvdAcceptedView;
    };

    THREAD static void viewTimeout(void *);
    THREAD static void gotRequestThread(void *);
    THREAD static void gotReplyThread(void *);
  };
  class MemberList {
  public:
    MemberList(Member *Membr, MemberList *Nxt = NULL);
    ~MemberList();
    void Destroy();
    void Print();
    Member *MemberElem;
    MemberList *Next;
  };

  EntityAddrLocal(Maestro_CSX_Options &Ops, 
		  const char *Addr, const char *GroupName=0,
		  tFunc ViewChange=0, void *VcEnv=0, tFunc4 GetState=0, 
		  void *GsEnv=0, tFunc4 SetState=0, void *SsEnv=0);
  ~EntityAddrLocal();
  void Print();
  Boolean AlreadyJoined(const char *groupName);
  Status AddMember(Maestro_CSX_Options &Ops, 
		   const char *Addr, const char *GroupName=0,
		   tFunc ViewChange=0, void *VcEnv=0, tFunc4 GetState=0, 
		   void *GsEnv=0, tFunc4 SetState=0, void *SsEnv=0);
  EntityAddrLocal::Member *GetMember(const char *groupName);
  Status LeaveGroup(const char *groupName=NULL);
protected:
  MemberList *members;
  MsgReceiveData *msgRcvData;               // msg receive upcall.
  Monitor monitors[MAX_MONITORS];    // for monMonitor.
};

class RequestData {
  friend class EntityAddrLocal;
public:
  RequestData(ElectraRequest *Req, tFunc Usr, void *Env)
    : req(Req), usr(Usr), env(Env) {}
  ~RequestData() {}
private:
  ElectraRequest *req;
  tFunc           usr;
  void           *env;
};

// to remember to whom we shall return a reply
class SenderIdent {
  friend class Ensemble_MaestroAD;
public:
  SenderIdent(Maestro_EndpID &Origin, EntityAddrLocal::Member *Group) 
    : origin(Origin), group(Group) {}
private:
  Maestro_EndpID origin;
  EntityAddrLocal::Member *group;
};

class ThreadWrap: public LocalData {
  friend class Ensemble_MaestroAD;
public:
  struct AdaptorThreadArgs {
    tFunc F;         // the function to be executed in the thread
    void *Env;
    void *Param;
  };
  ThreadWrap() { threadArg = new AdaptorThreadArgs; }
  ~ThreadWrap() { delete threadArg; }
private:
  static void threadExecuteFunc(void *Data);
  AdaptorThreadArgs *threadArg;
  tPrio p;
  int stackSize;
};

class SemaWrap: public LocalData {
  friend class Ensemble_MaestroAD;
public:
  SemaWrap(int InitValue) { sema = new Maestro_Semaphore(InitValue); }
  ~SemaWrap() { delete sema; }  
private:
  Maestro_Semaphore *sema;
};

class LockWrap: public LocalData {
  friend class Ensemble_MaestroAD;
public:
  LockWrap() { lock = new Maestro_Lock; }
  ~LockWrap() { delete lock; }  
private:
  Maestro_Lock *lock;
};

// global entity identifier: (Goes over the network.)
class EntityAddrGlobal: public GlobalData {
  friend class Ensemble_MaestroAD;
public:
  EntityAddrGlobal() { ::memset(addr, 0, MAX_GROUPNAME); }
  virtual ULong marshalSize(ULong &Align) const;
  virtual void marshal(Octet *&Buf) const;
  virtual void unmarshal(Octet *&Buf, Boolean);
  void Print();
private:  
  void makeGroupName();
  char addr[MAX_GROUPNAME];
};

// The Adaptor Object itself.
class Ensemble_MaestroAD : public VirtualMachine {
  friend class EntityAddrLocal;
public:
  Ensemble_MaestroAD(char *Name);
  ~Ensemble_MaestroAD();
  
  // Initialization:
  Status machInit();
  Status machQuit();
  void electraInit(const AppPolicy &);
  
  // Entities:
  Status  entityCreateImpl(Entity &, tPrio, const ProtocolPolicy&);
  Status  entityCreateRef(Entity &);
  Status  entityBind(RpcHandle *);
  Status  entityDestroyImpl(Entity &);
  Status  entityDestroyRef(Entity &) { return ST_OK; }
  Status  entityFlush(Entity &) { return ST_OK; }
  Boolean entityEqual(const Entity &, const Entity &);

  // Asynchronous/synchronous reliable message passing:
  Status msgReceive(Entity &Src, tFunc Usr, void *UEnv);
  Status msgSendRequest(Entity &Dest, ElectraMessage &M, tFunc SendDone, 
			void *SEnv);
  Status msgSendReply(ElectraMessage &M, tFunc SendDone, void *SEnv, 
		      void *AdaptorData);
  Status msgReceiveDone(void *);
  
  // Group management:
  Status grpCreate(Entity &Grp, const ProtocolPolicy &);
  Status grpJoin(Entity &Grp, Entity &E, tFunc Mon, void *,
		 tFunc4 GetState, void *, tFunc4 SetState, void *,
		 View *V);
  Status grpLeave(Entity &Grp, Entity& E);
  Status grpDestroy(Entity &Grp) { return ST_OK; }
  
  // "Thread" management:
  void threadDeclare(Thread &, tFunc F,
		     void *Env = 0,
		     tPrio P  = eMid,
		     int Stack = MIN_STACK);
  void threadCreate(Thread &T, void *Param);
  void threadYield() { return; }
  Status threadSweep(Thread &, long MSec){ return ST_NO_IMPLEMENT;};

  // Failure Suspectors:
  virtual Status monMonitor(Monitor &, Entity &, Monitor::monFunc);
  virtual Status monCancel(Monitor &);
  
  //    - semaphores:
  void semaCreate(Sema &S, int Value);
  void semaDec(Sema &S);
  void semaInc(Sema &S);
  void semaGetValue(Sema &S, int &) { return; }

  //   - locks:
  void lockCreate(Lock &L);
  void lockAcquire(Lock &L);
  void lockRelease(Lock &L);
  void lockGetValue(Lock &L, int &) { return; }

  static Boolean isLocalEntity(const Maestro_EndpID &E1, 
			       const Maestro_EndpID &E2);

  // accessor function for groupd flag
  Boolean UseGroupd() { return useGroupd; }
  
protected:
  Boolean useGroupd;
  // this class is necessary because the Maestro_EndpID does not export
  // the actual hot_endp_t field
  class Electra_EndpID : public Maestro_EndpID {
  public: 
    void GetEndpID(char Name[HOT_ENDP_MAX_NAME_SIZE]) const
    { strncpy(Name, endpID.name, HOT_ENDP_MAX_NAME_SIZE); }
  };
  
  static void extractLocation(const Electra_EndpID *, 
			      char[HOT_ENDP_MAX_NAME_SIZE]);
};

#endif
