/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
//
// $Id: Ensemble_MaestroAD.cc,v 1.0 1997/02/08 15:03:47 sachdeva,atre Exp $
//
// Authors: Anand Atre, Anil Sachdeva.  atre@cs.cornell.edu, 
// anil@cs.cornell.edu.
// Copyright (c) 1997 by Anand Atre, Anil Sachdeva.  anand@cs.cornell.edu, 
// anil@cs.cornell.edu.
// All rights reserved.
//
// THIS IS FREE SOFTWARE.
// Permission to use, copy, modify, and distribute this software and 
// documentation in all settings is hereby granted, provided that no fee is 
// charged for this software and provided that this copyright notice appears 
// in all copies of any software which is or includes a copy or modification 
// of this software. In all advertising materials, documentation, and 
// publications mentioning features or use of this software you must credit 
// the author.
//
// This software is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of merchantability 
// or fitness for a particular purpose. 
//
//
#include <sys/param.h>

#include <sys/time.h>
#include <sys/types.h>

#include </usr/include/netdb.h>
#include <signal.h>
#include <unistd.h>

#include <time.h>

#include "orb/CORBA.h"
#include "vmachine/NList.hh"
#include "Ensemble_MaestroAD.hh"

//#define DEBUG
#define XXX_DEBUG
//#define TRACE

#ifdef PURIFY
extern "C" {
  void purify_printf_with_call_chain(...);
};
#endif

extern const char *gUtilName; 

#ifdef TRACE
char *Trace[] = {"arg1","-trace","stacke","-trace","manage",NULL};
#endif

const int StackSize=128*1024;

// ************************************************************************
// The EntityAddrLocal::Member Class
// ************************************************************************
EntityAddrLocal::
Member::Member(Maestro_CSX_Options &Ops, const char *Addr,
	       EntityAddrLocal *Owner, const char *GroupName, 
	       tFunc ViewChange, void *VcEnv, 
	       tFunc4 GetState, void *GsEnv, tFunc4 SetState, 
	       void *SsEnv) : Maestro_CSX(Ops) 
{	
  strncpy(addr, Addr, MAX_GROUPNAME);
  if(GroupName) 
    currView.group_name(GroupName);
  else 
    currView.group_name(addr);
  viewChange = ViewChange;
  vcEnv = VcEnv;
  getStateUpcall = GetState;
  gsEnv = GsEnv;
  setStateUpcall = SetState;
  ssEnv = SsEnv;
  owner = Owner;
  choiceMember = -1;
  receivedAcceptedView=0;

  ad->semaCreate(clientBindViewSema, 0);
  oldServers.clear();
  join();
}

EntityAddrLocal::Member::~Member() {}

void EntityAddrLocal::Member::Print() {
  cout << "NumServers=" << oldServers.size();
  cout << ", Electra GroupName = " << currView.group_name();
  cout << ", Ensemble GroupName = " << addr;
}

// got a request message cast to servers only
void EntityAddrLocal::
Member::csx_ReceiveScast_Callback(Maestro_EndpID &Origin, 
				  Maestro_Message &Msg) {
#ifdef DEBUG
  cerr << "csx_ReceiveScast_Callback: Origin = " << Origin << endl;
#endif 

  ElectraRequest *Req = new ElectraRequest;

  ULong MsgLen;
  Msg.read(&MsgLen, sizeof(ULong));
  Req->allocate(MsgLen);

  Msg.read((char *)Req->data(), MsgLen);

  LocalThreadArgs *ThreadArg = new LocalThreadArgs;
  RequestData *RData = new RequestData(Req, owner->msgRcvData->usr_, 
				       owner->msgRcvData->uEnv_);
  ThreadArg->RData = RData;
  ThreadArg->SendId = new SenderIdent(Origin, this);

  Maestro_ThreadOps ThreadOps;
  ThreadOps.stackSize = StackSize;

  // create a new thread to handle request
  Maestro_Thread::create(gotRequestThread, (void *)ThreadArg, &ThreadOps);
}

// got a reply message:
void EntityAddrLocal::
Member::csx_ReceiveSend_Callback(Maestro_EndpID &Origin, 
				 Maestro_Message &Msg) {
#ifdef DEBUG
  cerr << "csx_ReceiveSend_Callback\n";
#endif

  ElectraReply *Rep = new ElectraReply;
  ULong MsgLen;
  Msg.read(&MsgLen, sizeof(ULong));
  Rep->allocate(MsgLen);
  Msg.read(Rep->data(), MsgLen);
  Octet *Tmp=0;
  Rep->unmarshalMessage(Tmp);

  if(Rep->isType(Electra::ReplyCall)) {
    Maestro_ThreadOps ThreadOps;
    ThreadOps.stackSize = StackSize;
    Maestro_Thread::create(gotReplyThread, (void *)Rep, &ThreadOps);
    // rpc->rpcGotReply(Rep);
  } else {
    Rep->freeData(FALSE);

    ElectraRequest *Req = new ElectraRequest;
    Req->allocate(Rep);
    Req->freeData(TRUE);
    Rep->_release();

    LocalThreadArgs *ThreadArg = new LocalThreadArgs;
    ThreadArg->RData = new RequestData(Req, owner->msgRcvData->usr_, 
				       owner->msgRcvData->uEnv_);

    ThreadArg->SendId = new SenderIdent(Origin, this);

    Maestro_ThreadOps ThreadOps;
    ThreadOps.stackSize = StackSize;

    Maestro_Thread::create(gotRequestThread, (void *)ThreadArg, 
   			   &ThreadOps);
  }
}

// XFER_Upcall upcall requests the state from another member
// piece wise.
//
void EntityAddrLocal::
Member::stateTransfer_Callback(Maestro_XferID &XferID) { 
#ifdef DEBUG
  cerr << "stateTransfer_Callback" << endl; 
#endif
  CORBA::AnySeq *State = new CORBA::AnySeq;
  Maestro_Message RequestMsg, StateMsg;
  Maestro_XferStatus Stat;
  Char *Buf, *BufP;
  Boolean Done = TRUE;
  Boolean StateByteOrd;
  
  do {
    RequestMsg.reset();
    getState(XferID, RequestMsg, StateMsg, Stat);
    if(Stat == MAESTRO_XFER_TERMINATED) { 
      delete State;
      return;
    }

    // received a part of the state
    ULong Size;
    StateMsg.read(&Size, sizeof(ULong));

    Buf  = new Octet [Size]; 
    BufP = Buf;
    StateMsg.read(Buf, Size);
    ::unmarshal(StateByteOrd, BufP, gByteOrder);
    ::unmarshal(*State, BufP, StateByteOrd);
    ::unmarshal(Done, BufP, StateByteOrd);
    delete[] Buf;
    if(setStateUpcall) {
      (*setStateUpcall)(ssEnv, *State, &currView, Done);
    } else {
      gLog->panic("EntityAddrLocal::Member::stateTransfer_callback: "
		  "setStateUpcall not defined");
    }
  } while(!Done);
  
  xferDone(XferID); 
  delete State;
}

// handle incoming state request:
void EntityAddrLocal::
Member::askState_Callback(Maestro_EndpID &Origin, Maestro_XferID &XferID,
			  Maestro_Message &RequestMsg) {
#ifdef DEBUG
  cerr << "askState_Callback " << endl;
#endif
  Maestro_Message StateMsg;
  CORBA::AnySeq *State = new CORBA::AnySeq;
  ULong Size, Align = 8ul;
  Char *Buf, *BufP;
  Boolean Done = TRUE;
  
  State->length(0);
  if(getStateUpcall)
    (*getStateUpcall)(gsEnv, *State, &currView, Done);
  else {
    gLog->panic("EntityAddrLocal::Member::askState_Callback: getStateUpcall "
		"not defined.");
  }
  
  Size = 
    ::marshalSize(gByteOrder, Align) 
    + ::marshalSize(*State, Align) 
    + ::marshalSize(Done, Align);
  Buf = new Octet [Size]; 
  BufP = Buf;
  ::marshal(gByteOrder, BufP);
  ::marshal(*State, BufP);
  ::marshal(Done, BufP);
  StateMsg.reset();
  StateMsg.write(Buf, Size);
  StateMsg.write(&Size, sizeof(ULong));
  delete[] Buf;
  sendState(Origin, XferID, StateMsg);
  delete State;
}

void EntityAddrLocal::
Member::csx_AcceptedView_Callback(Maestro_CSX_ViewData& ViewData,
				  Maestro_Message &Msg) {
#ifdef DEBUG
  cerr << "csx_AcceptedView_Callback" << endl;
#endif

  int i;
  Short ItsRank = -1;
  Short NumJoined = 0, NumLeft = 0;

  // check whether there is something to do at all:
  if(isServer() &&
     (ViewData.myServerRank != 0) && (ViewData.newServers.size() == 0) &&
     (ViewData.departedServers.size() == 0) &&
     (ViewData.servers.size() == this->oldServers.size()))
    return;
  
  if(isClient() && 
     (ViewData.newServers.size() == 0) &&
     (ViewData.departedServers.size() == 0) &&
     (ViewData.servers.size() == this->oldServers.size()))
    return;
  
#ifdef DEBUG
  cerr << "----------------------------------------------------" << endl;
  if(isServer())
    cerr << "I am a server" << endl;
  else cerr << "I am a client" << endl;
  cerr << "servers         = " << ViewData.servers.size() << endl
       << "departedServers = " << ViewData.departedServers.size() << endl
       << "newServers      = " << ViewData.newServers.size() << endl
       << "xferServers     = " << ViewData.xferServers.size() << endl
       << "clients         = " << ViewData.clients.size() << endl;
  if(isServer())
    cerr << "my Server rank  = " << ViewData.myServerRank << endl;
  cerr << "oldServers      = " << oldServers.size() << endl;
#endif

#ifdef XXX_DEBUG
  cout << "----------------------------------------------------" << endl;
  cout << "VIEW CHANGE for " << currView.group_name() << endl;
  if(isServer())
    cout << "I am a server" << endl;
  else cout << "I am a client" << endl;
  cout << "servers         = " << ViewData.servers.size() << endl
       << "departedServers = " << ViewData.departedServers.size() << endl
       << "newServers      = " << ViewData.newServers.size() << endl
       << "xferServers     = " << ViewData.xferServers.size() << endl
       << "clients         = " << ViewData.clients.size() << endl;
  if(isServer())
    cout << "my Server rank  = " << ViewData.myServerRank << endl;
  cout << "oldServers      = " << oldServers.size() << endl;
#endif

  // Ensemble might pack several join/leave events into the same view change 
  // notification. In Electra, we want one notification per join/leave
  // event, for simplicity:
  if((NumJoined = ViewData.newServers.size()) > 1) {

    Maestro_CSX_ViewData VD = ViewData;
    for(i=0; i < NumJoined; i++) {
      VD.newServers.clear();
      VD.newServers += ViewData.newServers[i];
      Maestro_Message M;
      csx_AcceptedView_Callback(VD, M);
    }
    return;
  }
  if((NumLeft = ViewData.departedServers.size()) > 1) {
    Maestro_CSX_ViewData VD = ViewData;
    for(i=0; i < NumLeft; i++) {
      VD.departedServers.clear();
      VD.departedServers += ViewData.departedServers[i];
      Maestro_Message M;
      csx_AcceptedView_Callback(VD, M);
    }
    return;
  }

  assert(ViewData.departedServers.size() < 2);
  assert(ViewData.newServers.size() < 2);
  
  // determine the rank of the departed servers:
  if(ViewData.departedServers.size() > 0) {
    for(i=0; i < oldServers.size(); i++) {
      if(oldServers[i] == ViewData.departedServers[0]) {
	ItsRank = i; 
	break;
      }
    }
  }
  
  // select choiceMember for Electra::ChoiceCall:
  choiceMember = -1;

  if(isClient() && (ViewData.servers.size() > 0)) {
    Maestro_EndpID MyEid;
    myEndpID(MyEid);
    for(i=0; i < ViewData.servers.size(); i++) {
      if(Ensemble_MaestroAD::isLocalEntity(MyEid, ViewData.servers[i])) { 
#ifdef DEBUG
	cerr << "csx_AcceptedView_Callback: CHOOSING LOCAL MEMBER " << i 
	     << endl;
#endif
	choiceMember = i; 
	break; 
      }
    }
    if(choiceMember == -1) {
      choiceMember = rand() % ViewData.servers.size();
#ifdef DEBUG
      cerr << "csx_AcceptedView_Callback: CHOOSING REMOTE MEMBER " 
	   << choiceMember << endl;
#endif
    }
  }
  
  // build view object:
  View V;
  V.num_members(ViewData.servers.size());
  V.num_left(ViewData.departedServers.size());
  V.num_joined(ViewData.newServers.size());
  V.my_rank(isClient() ? -1 : ViewData.myServerRank);
  V.its_rank(ItsRank);
  // don't set the V.group_name field.  It'll be set by the higher layer

  // invoke monitors:
  if(ViewData.departedServers.size() > 0 && ViewData.servers.size() == 0) {
    for(i=0; i < MAX_MONITORS; i++)
      if(owner->monitors[i].mf_)
	CORBA::ORB::monitorWrap(owner->monitors[i], V, 
				owner->monitors[i].env_);
  }
  
  // remember this view:
  oldServers = ViewData.servers;
  
  // When running atop of the Heal layer, the first view will be
  // a singleton view containing only the client. We give the Heal layer some
  // time (VIEW_TIMEOUT) to find server objects for a freshly created
  // object reference.
  if(isClient() && (ViewData.newServers.size() > 0) && 
     (receivedAcceptedView != NULL)) {
    // found at least one server objects. Unblock the client.
    // viewTimeout takes possession of receivedAcceptedView.
    // the following tells viewTimeout that we have received an Accepted View
    // and it need not bother to do anything
    *receivedAcceptedView = TRUE;
    receivedAcceptedView = 0;
    ad->semaInc(clientBindViewSema);
  }
  
  // copy this view into the EntityAddrLocal's view member
  currView.num_members(V.num_members());
  currView.num_left(V.num_left());
  currView.num_joined(V.num_joined());
  currView.my_rank(V.my_rank());
  currView.its_rank(V.its_rank());

  // copy our actual group name into the View data structure
  V.group_name(currView.group_name());

  // Notify the RPC layer of the view change:
  if(isClient()) {
    rpc->moreMembers((RpcHandle*)vcEnv, ViewData.servers.size(), ItsRank);
    return;
  } 

  // deliver view change to object implementations:
  if(viewChange) 
    (*viewChange)(vcEnv, (void*)&V);
}

void EntityAddrLocal::Member::gotRequestThread(void *Arg) {
  
  LocalThreadArgs *ThreadArg = (LocalThreadArgs *)Arg;
  RequestData *RData = ThreadArg->RData;
  SenderIdent *SendId = ThreadArg->SendId;
  
  rpc->rpcGotRequest(RData->req, RData->usr, RData->env, SendId);
  delete ThreadArg;
}

void EntityAddrLocal::Member::gotReplyThread(void *Env) {
  rpc->rpcGotReply((ElectraReply *)Env);
}

THREAD void EntityAddrLocal::Member::viewTimeout(void *Arg) {

  ViewTimeoutWrap *VArg = (ViewTimeoutWrap *)Arg;
  EntityAddrLocal::Member *Me = VArg->Memb;
  Boolean *RcvdAcceptedView = VArg->RcvdAcceptedView;

  if(RcvdAcceptedView == NULL)
    gLog->panic("Error! No argument for accepted view signaling");

  sleep(VIEW_TIMEOUT_NOTIFY);
  if(*RcvdAcceptedView == FALSE) {
#ifdef DEBUG
    cerr << "Ensemble: Heal layer: roaming for group members..." << endl;
#endif
    sleep(VIEW_TIMEOUT - VIEW_TIMEOUT_NOTIFY);
    if(*RcvdAcceptedView == FALSE) {
      // the Member still hasn't received a view Timeout
      // it is safe to assume that it still exists
#ifdef DEBUG
      cerr << "Ensemble: Heal layer: no group members found." << endl;
#endif
      ad->semaInc(Me->clientBindViewSema);
      Me->receivedAcceptedView = 0;
    }
  }
  delete RcvdAcceptedView;
}

// ************************************************************************
// The EntityAddrLocal::MemberList Class
// ************************************************************************
EntityAddrLocal::MemberList::MemberList(Member *Membr, MemberList *Nxt) {
  MemberElem = Membr;
  Next = Nxt;
}

EntityAddrLocal::MemberList::~MemberList() {
  delete MemberElem;
  MemberElem = 0;
}

void EntityAddrLocal::MemberList::Destroy() {
  if(Next) {
    Next->Destroy();
    Next = 0;
  }
  delete this;
}

void EntityAddrLocal::MemberList::Print() {
  if(MemberElem)
    MemberElem->Print();
  else {
    cout << "Fatal Error! MemberList without a Maestro Object" << endl;
    abort();
  }
}
    
// ************************************************************************
// The EntityAddrLocal Class
// ************************************************************************
EntityAddrLocal::EntityAddrLocal(Maestro_CSX_Options &Ops, 
				 const char *Addr, const char *GroupName,
				 tFunc ViewChange, void *VcEnv, 
				 tFunc4 GetState, void *GsEnv,
				 tFunc4 SetState, void *SsEnv) {
  Member *Mbr = new Member(Ops, Addr, this, GroupName, ViewChange, VcEnv, 
			   GetState, GsEnv, SetState, SsEnv);
  members = new MemberList(Mbr);
  msgRcvData = new MsgReceiveData;
}

EntityAddrLocal::~EntityAddrLocal() {
  if(members)
    members->Destroy();
  delete msgRcvData;
}

void EntityAddrLocal::Print() {
  int i=0;
  MemberList *Tmp=members;
  cout << "EntityAddrLocal: " << endl;
  while(Tmp) {
    cout << "Member " << i << ":";
    Tmp->Print(); 
    cout << endl;
    i++;
    Tmp=Tmp->Next;
  }
}

Boolean EntityAddrLocal::AlreadyJoined(const char *groupName) {
  MemberList *Tmp = members;
  while(Tmp) {
    if(!strcmp(Tmp->MemberElem->addr, groupName)) {
      return TRUE;
    }
    Tmp=Tmp->Next;
  }
  return FALSE;
}

Status 
EntityAddrLocal::AddMember(Maestro_CSX_Options &Ops, 
			   const char *Addr, const char *GroupName,
			   tFunc ViewChange, void *VcEnv, tFunc4 GetState, 
			   void *GsEnv, tFunc4 SetState, void *SsEnv) {
  Member *Mbr = new Member(Ops, Addr, this, GroupName, ViewChange, VcEnv, 
			   GetState, GsEnv, SetState, SsEnv);
  if(members) {
    MemberList *Tmp = new MemberList(Mbr, members);
    members = Tmp;
  } else {
    members = new MemberList(Mbr);
  }
  return ST_OK;
}

EntityAddrLocal::Member *EntityAddrLocal::GetMember(const char *groupName) {
  MemberList *Tmp = members;
  while(Tmp) {
    if(!strcmp(Tmp->MemberElem->addr, groupName)) {
      return Tmp->MemberElem;
    }
    Tmp=Tmp->Next;
  }
  return NULL;
}

// if the groupName is NULL, we want to leave all the groups we are a 
// part of
Status EntityAddrLocal::LeaveGroup(const char *groupName) {
  MemberList *Tmp = members;
  if(groupName == NULL) {
    while(Tmp) {
      MemberList *Curr = Tmp;
      Tmp = Tmp->Next;
      delete Curr;
    }
    members = NULL;
    return ST_OK;
  }
  
  // we want to leave a particular group
  MemberList *Prev = NULL;
  while(Tmp) {
    // we only want to leave one group, see if this is the one
    if(!strcmp(Tmp->MemberElem->addr, groupName)) {
      if(Prev) {
	Prev->Next = Tmp->Next;
      } else {
	// this element was the head
	members = Tmp->Next;
      }
      delete Tmp;
      return ST_OK;
    }
    Prev = Tmp;
    Tmp=Tmp->Next;
  }
  return ST_UNKNOWN;
}

// *************************************************************************
// Class EntityAddrGlobal 
// *************************************************************************
ULong EntityAddrGlobal::marshalSize(ULong &Align) const {
  int Size = MAX_GROUPNAME * sizeof(char);
  Align += Size;
  return Size;
}

void EntityAddrGlobal::marshal(Octet *&Buf) const {
  int Size = MAX_GROUPNAME * sizeof(char);
  ::memcpy(Buf, addr, Size);
  Buf += Size;
}

void EntityAddrGlobal::unmarshal(Octet *&Buf, Boolean) {
  int Size = MAX_GROUPNAME * sizeof(char);
  ::memcpy(addr, Buf, Size);
  Buf += Size;
}

void EntityAddrGlobal::Print() {
  cout << "EntityAddrGlobal: addr = " << addr;
}

void EntityAddrGlobal::makeGroupName() {
  static int Count = 0;
  struct timeval tv;
  char Temp[25];
  
  tmpnam(Temp);
  
  if(gettimeofday(&tv, 0) < 0)
    gLog->panic("EntityAddrGlobal::makeGroupName: gettimeofday");
  
  sprintf(addr, "%s/%d/%d/%d", Temp, tv.tv_sec, tv.tv_usec, Count++);
}

// *************************************************************************
// Class Ensemble_MaestroAD 
// *************************************************************************
Ensemble_MaestroAD::Ensemble_MaestroAD(char *Name) : VirtualMachine(Name) { 
  if(getenv("ENS_GROUPD")) {
    cout << "Starting Ensemble in groupd mode" << endl;
    useGroupd=TRUE;
  } else {
    cout << "Starting Ensemble in gossip mode" << endl;
    useGroupd=FALSE;
  }
}

Ensemble_MaestroAD::~Ensemble_MaestroAD() {
#ifdef PURIFY
  purify_printf_with_call_chain("Ensemble_MaestroAD::~Ensemble_MaestroAD");
#endif
}

Status Ensemble_MaestroAD::machInit() {
  VirtualMachine::machInit();
  electra_main(gArgc, gArgv);
  return ST_OK;
}

Status Ensemble_MaestroAD::machQuit() {
#ifdef PURIFY
  purify_printf_with_call_chain("Ensemble_MaestroAD:machQuit");
#endif
  VirtualMachine::machQuit();
  return ST_OK;
}

void Ensemble_MaestroAD::electraInit(const AppPolicy& ap) {
  init_ = TRUE;
  if(ap.threadModel() != Policy::eNonPreemptive) {
    gLog->panic("Ensemble_MaestroAD::electraInit: AppPolicy not supported");
  }
}

// Create an entity for an object implementation.
// entityCreateImpl always creates a group, so a singleton object is
// represented as a ensemble group with one member:
Status 
Ensemble_MaestroAD::entityCreateImpl(Entity &E, tPrio Prio, 
				     const ProtocolPolicy &P) {
  EntityAddrGlobal *EaddrG = new EntityAddrGlobal;
  EntityAddrLocal  *EaddrL;
  Maestro_CSX_Options Ops;
  
//   if(P.stack()) {
//     cout << "entityCreateImpl: USING Stack provided by ELECTRA" << endl;
//     cout << "Stack = " << (char *)P.stack() << endl;
//     Ops.protocol = (char*)P.stack();
//   } else {
//     if(useGroupd == TRUE) {
//       Ops.protocol = GROUPD_STACK;
//     } else {
//       Ops.protocol = GOSSIP_STACK;
//     }
//   }
#ifdef TRACE
  Ops.argv = Trace;  // necessary for printing traces etc.
#endif
  Ops.groupdFlag = (useGroupd? 1:0);
  //Ops.properties = "Gmp:Sync:Heal:Switch:Frag:Suspect:Flow";
  //Ops.params = "suspect_max_idle=3:int;suspect_sweep=1.000:time";

  EaddrG->makeGroupName();
  Ops.groupName = EaddrG->addr;
  Ops.xferType = MAESTRO_ATOMIC_XFER;
  Ops.mbrshipType = MAESTRO_SERVER;

  //Ops.programName = (char *)gUtilName;
  Ops.threadMode = MAESTRO_MODE_MULTI_THREADED;

  Ops.transports = "UDP";
  EaddrL = new EntityAddrLocal(Ops, EaddrG->addr);

  E.global(EaddrG);
  E.local(EaddrL);
  
  return ST_OK;
}

// Create an entity to for an object reference.
Status Ensemble_MaestroAD::entityCreateRef(Entity &E) {

#ifdef DEBUG
  cerr << "Ensemble_MaestroAD::entityCreateRef" << endl;
#endif

  EntityAddrGlobal *EaddrG = new EntityAddrGlobal;
  EaddrG->makeGroupName();
  E.global(EaddrG);
  E.local(0);
  return ST_OK;
}

// NOTE: binding is performed by msgSendRequest when the first message is sent:
Status Ensemble_MaestroAD::entityBind(RpcHandle *H) {
  EntityAddrGlobal *EaddrG = (EntityAddrGlobal*)H->entity().global();
  EntityAddrLocal  *EaddrL = (EntityAddrLocal*)H->entity().local();
  Entity &Dest = H->entity();

  if(EaddrG == NULL)
    gLog->panic("No address specified for entityBind");

#ifdef XXX_DEBUG
  cout << "entityBind: ";
  EaddrG->Print();
  cout << endl;
#endif

  // have to create a new Member
  Maestro_CSX_Options Ops;

  // dest was retrieved from the name server and thus has no
  // associated ENSEMBLE object yet. Create ENSEMBLE object:

  // overwrite the group id with the data retrieved from the name server:
  Ops.groupName = EaddrG->addr;
  //Ops.programName = (char *)gUtilName;
  Ops.threadMode = MAESTRO_MODE_MULTI_THREADED;
  Ops.transports = "UDP";
//   if(Dest.proto().stack()) {
//     cout << "entityBind: USING Stack provided by Electra" << endl;
//     cout << "Stack = " << (char *)Dest.proto().stack() << endl;
//     Ops.protocol = (char*)Dest.proto().stack();
//   } else {
//     if(useGroupd == TRUE) {
//       Ops.protocol = GROUPD_STACK;
//     } else {
//       Ops.protocol = GOSSIP_STACK;
//     }
//   }
#ifdef TRACE
  Ops.argv = Trace;  // necessary for printing traces etc.
#endif
  Ops.groupdFlag = (useGroupd? 1:0);
  //Ops.properties = "Gmp:Sync:Heal:Switch:Frag:Suspect:Flow";
  //Ops.params = "suspect_max_idle=3:int;suspect_sweep=1.000:time";

  Ops.mbrshipType = MAESTRO_CLIENT;

  // create a new member and add it to the EntityAddrLocal,
  // also set its vcEnv field
  if(EaddrL == NULL) {
    EaddrL = new EntityAddrLocal(Ops, EaddrG->addr, NULL, NULL, (void *)H);
    Dest.local(EaddrL);
  } else {
    EaddrL->AddMember(Ops, EaddrG->addr, NULL, NULL, (void *)H);
  }
 
  // wait for a view that contains at least one server. 
  // time out if it takes too long.  
  EntityAddrLocal::Member *Memb = EaddrL->GetMember(EaddrG->addr);
  if(Memb == NULL) {
    gLog->panic("Ensemble_MaestroAD::entityBind: Error retrieving member");
  }
  
  Maestro_ThreadOps ThreadOps;
  ThreadOps.stackSize = StackSize;
  if(Memb->receivedAcceptedView != NULL) {
    gLog->panic("Already waiting for a view Timeout");
  }
  Memb->receivedAcceptedView = new Boolean(FALSE);
  EntityAddrLocal::Member::ViewTimeoutWrap *Arg = 
    new EntityAddrLocal::Member::ViewTimeoutWrap(Memb, 
						 Memb->receivedAcceptedView);
  Maestro_Thread::create(EntityAddrLocal::Member::viewTimeout, Arg, 
  			 &ThreadOps);
  // this semaphore may be incremented by AcceptedView callback, or 
  // viewTimeout if the callback doesn't come back in time (times out)
  ad->semaDec(Memb->clientBindViewSema);

  return ST_OK;
}

Status Ensemble_MaestroAD::entityDestroyImpl(Entity &E) {
#ifdef DEBUG
  cerr << "entityDestroyImpl" << endl;
#endif
  // we want to leave all the groups we are a part of
  EntityAddrLocal *EaddrL = (EntityAddrLocal *)E.local();
  // leave all groups
  Status st = EaddrL->LeaveGroup(NULL);
  EaddrL->_release();
  return st;
}

// check whether a and b denote the same destination:
Boolean Ensemble_MaestroAD::entityEqual(const Entity &A, const Entity &B) {
  return ::memcmp(((EntityAddrGlobal*)A.global())->addr, 
		  ((EntityAddrGlobal*)B.global())->addr, 
		  MAX_GROUPNAME*sizeof(char)) == 0;
}

// Registers a thread which will be invoked when a message
// arrives on Entity listenOn:
Status Ensemble_MaestroAD::msgReceive(Entity &ListenOn, 
				      tFunc Usr, void *UEnv) {
  EntityAddrLocal *EaddrL = (EntityAddrLocal*)ListenOn.local();
  EntityAddrGlobal *EaddrG = (EntityAddrGlobal *)ListenOn.global();
  EaddrL->msgRcvData->usr_    = Usr;
  EaddrL->msgRcvData->uEnv_   = UEnv;

  return ST_OK;
}

inline Status Ensemble_MaestroAD::msgReceiveDone(void *AdaptorData) {
  if(AdaptorData) {
    delete (SenderIdent *) AdaptorData;
    AdaptorData = NULL;
  }
  return ST_OK;
}

// deliver a request message:
Status Ensemble_MaestroAD::msgSendRequest(Entity &Dest, ElectraMessage &M, 
					  tFunc SendD, void *SEnv) {
  Status st;

  // bind to remote object implementation, if necessary:
  if(!Dest.isBound() || !Dest.local()) {
    Dest.setBound(TRUE);
    if((st = entityBind(M.handle())) != ST_OK)
      return st;
  }

  Maestro_Message Msg;
  EntityAddrGlobal *EaddrG  = (EntityAddrGlobal*)Dest.global();
  EntityAddrLocal  *EaddrL = (EntityAddrLocal*)Dest.local();
  EntityAddrLocal::Member *Mbr = EaddrL->GetMember(EaddrG->addr);
  if(Mbr == NULL)
    gLog->panic("Ensemble_MaestroAD::msgSendRequest: Error finding "
		"Maestro object");
  Msg.reset();
  ULong MsgSize = M.dataSize();
  Msg.write(M.data(), MsgSize);
  Msg.write(&MsgSize, sizeof(ULong));

#ifdef XXX_DEBUG
  cout << "--------------------------------------------------" << endl;
  cout << "Before sending msg in msgSendRequest" << endl;
  EaddrL->Print();
#endif

  // check whether the target group is empty
  if(Mbr->oldServers.size() == 0) {
    if(!gQuiet)
      cerr << "Ensemble_MaestroAD::msgSendRequest: TARGET GROUP IS EMPTY"
	   << endl;
#ifdef PURIFY
    purify_printf_with_call_chain("empty group");
#endif
    return ST_COMM_FAILURE;
  }
  
  if((Electra::CallType)M.message_type() & Electra::ChoiceCall) {
#ifdef DEBUG
    cerr << "CHOICECALL" << endl;
#endif
    if(Mbr->choiceMember == -1) 
      gLog->panic("Ensemble_MaestroAD::msgSendRequest internal error 1");
    Mbr->send(Mbr->oldServers[Mbr->choiceMember], Msg);
  } else {
#ifdef DEBUG
    cerr << "Ensemble_MaestroAD::msgSendRequest: Performing an scast" << endl;
#endif
    Maestro_CSX_MsgOptions Mops;
    Mops.msgXferSafety = MAESTRO_MSG_GENERIC;
    Mops.destList.clear();
    Mops.destList += Mbr->oldServers;
    Maestro_MsgSendView SendView;
    Mbr->scast(Msg, SendView, Mops);
  }
  
  // tell the upper layer that we are done with the request
  if(SendD)
    (*SendD)(SEnv, 0);
  else {
    gLog->panic("Ensemble_MaestroAD::msgSendRequest: no method to inform "
		"upper layer");
  }
  return ST_OK;
}

// deliver a reply message:
Status Ensemble_MaestroAD::msgSendReply(ElectraMessage &M, tFunc SendD, 
					void *SEnv, void *AdData) {
#ifdef DEBUG
  cerr << "msgSendReply: " << ((SenderIdent *)AdData)->origin << endl;
#endif
  Maestro_Message Msg;
  Msg.reset();
  ULong MsgSize = M.dataSize();
  Msg.write(M.data(), MsgSize);
  Msg.write(&MsgSize, sizeof(ULong));

  SenderIdent *SendId = (SenderIdent *)AdData;
  SendId->group->send(SendId->origin, Msg);

  // tell the upper layer that we're done with the reply
  if(SendD)
    (*SendD)(SEnv, 0);
  else {
    gLog->panic("Ensemble_MaestroAD::msgSendReply: no method to inform "
		"upper layer");
  }
  return ST_OK;
}

Status Ensemble_MaestroAD::grpCreate(Entity& Grp, const ProtocolPolicy &P) {
#ifdef DEBUG
  cerr << "grpCreate" << endl;
#endif
  EntityAddrGlobal *EaddrG = new EntityAddrGlobal;
  EaddrG->makeGroupName();
  Grp.global(EaddrG);
  Grp.name(EaddrG->addr);
  return ST_OK;
}

Status Ensemble_MaestroAD::grpJoin(Entity &G, Entity &E,
				   tFunc ViewChange, void *VcEnv, 
				   tFunc4 GetState, void *GsEnv,
				   tFunc4 SetState, void *SsEnv, View *V) {
  
  Maestro_CSX_Options Ops;
  
  EntityAddrGlobal *Grp = (EntityAddrGlobal *)G.global();
  EntityAddrLocal  *EaddrL = (EntityAddrLocal *) E.local(); 

#ifdef XXX_DEBUG
  cout << "--------------------------------------------------" << endl;
  cout << "Before joining in grpJoin" << endl;
  EaddrL->Print();
  cout << "Grp->addr = " << Grp->addr;
  cout << ", G.name() = " << G.name() << endl;
#endif

  // see if we already belong to this group
  if(EaddrL->AlreadyJoined(Grp->addr)) {
#ifdef DEBUG
    cerr << "Ensemble_MaestroAD::grpJoin: Already belong to this group" 
	 << endl;
#endif
    return ST_OK;
  }

//   if(G.proto().stack()) {
//     cout << "grpJoin: USING Stack provided by ELECTRA" << endl;
//     cout << "Stack = " << (char *)G.proto().stack() << endl;
//     Ops.protocol = (char*)G.proto().stack();
//   } else {
//     if(useGroupd) {
//       Ops.protocol = GROUPD_STACK;
//     } else {
//       Ops.protocol = GOSSIP_STACK;
//     }
//   }
  
#ifdef TRACE
  Ops.argv = Trace;  // necessary for printing traces etc.
#endif
  Ops.groupdFlag = (useGroupd? 1:0);
  //Ops.properties = "Gmp:Sync:Heal:Switch:Frag:Suspect:Flow";
  //Ops.params = "suspect_max_idle=3:int;suspect_sweep=1.000:time";
  //Ops.programName = (char *)gUtilName;

  Ops.groupName = Grp->addr;
  Ops.threadMode = MAESTRO_MODE_MULTI_THREADED;
  Ops.transports = "UDP";

  Ops.xferType = MAESTRO_ATOMIC_XFER;
  Ops.mbrshipType = MAESTRO_SERVER;
  // create a new Member object setting its true group name also
  EaddrL->AddMember(Ops, Grp->addr, V->group_name(), ViewChange, VcEnv, 
		    GetState, GsEnv, SetState, SsEnv);
  
  EntityAddrLocal::Member *Mbr = EaddrL->GetMember(Grp->addr);
  if(Mbr == NULL)
    gLog->panic("Ensemble_MaestroAD::grpJoin: Error retrieving member");

  // copy the fields of the view we received from Ensemble into the argument 
  // View except the group_name
  V->num_members(Mbr->currView.num_members());
  V->num_left(Mbr->currView.num_left());
  V->num_joined(Mbr->currView.num_joined());
  V->my_rank(Mbr->currView.my_rank());
  V->its_rank(Mbr->currView.its_rank());

#ifdef DEBUG
  cerr << "grpJoin, Ensemble Group = " << Grp->addr << ", Electra Group = " 
       << V->group_name() << endl;
#endif
  
#ifdef XXX_DEBUG
  cout << "--------------------------------------------------" << endl;
  cout << "After joining group in grpJoin" << endl;
  EaddrL->Print();
#endif
  return ST_OK;
}

Status Ensemble_MaestroAD::grpLeave(Entity &G, Entity &E) {

  EntityAddrGlobal *EaddrG = (EntityAddrGlobal*)G.global();
  EntityAddrLocal *EaddrL = (EntityAddrLocal*)E.local();

#ifdef XXX_DEBUG
  cout << "--------------------------------------------------" << endl;
  cout << "Before leaving group in grpLeave" << endl;
  EaddrL->Print();
#endif

#ifdef DEBUG
  cerr << "grpLeave" << endl;
#endif
  Status st = EaddrL->LeaveGroup(EaddrG->addr);
#ifdef XXX_DEBUG
  cout << "--------------------------------------------------" << endl;
  cout << "After leaving group in grpLeave" << endl;
  EaddrL->Print();
#endif
  return st;
}

// threadDeclare declare a thread to the MUTS layer. Electra Thread objects 
// consist of a visible (public) part defining the function, stacksize, 
// priority etc., and of an invisible part (Thread::priv_) ti which the 
// adaptor attaches its data. HorusHotAD attaches a MUTS thread to it:
void Ensemble_MaestroAD::threadDeclare(Thread &T, tFunc F, void *Env, 
				       tPrio P, int Stack) {
  
  // fill out the Thread object:
  threadGet(T, F, Env, P, Stack);
  
  ThreadWrap *ThreadOut = new ThreadWrap;
  ThreadOut->threadArg->F = F;
  ThreadOut->stackSize = Stack;
  ThreadOut->p = P;
  
  T.local(ThreadOut);
}

void ThreadWrap::threadExecuteFunc(void *Data) {

  if(!Data) {
    gLog->panic("threadExecuteFunc: thread not initialized.");
  }

  ThreadWrap::AdaptorThreadArgs *ThreadArg = 
    (ThreadWrap::AdaptorThreadArgs *)Data;
  ThreadArg->F(ThreadArg->Env, ThreadArg->Param);
}

void Ensemble_MaestroAD::threadCreate(Thread &T, void *Param) {

  ThreadWrap *ThreadOut = (ThreadWrap *)T.local();
  if(!ThreadOut) {
    gLog->panic("Ensemble_MaestroAD::threadCreate: received uninitialized "
		"thread.");
  }
  ThreadWrap::AdaptorThreadArgs *ThreadArg = 
    (ThreadWrap::AdaptorThreadArgs *)ThreadOut->threadArg;

  ThreadArg->Env = T.getEnv();
  ThreadArg->Param = Param;

  Maestro_ThreadOps ThreadOps;
  ThreadOps.stackSize = ThreadOut->stackSize;

  Maestro_Thread::create(ThreadWrap::threadExecuteFunc, (void *)ThreadArg, 
			 &ThreadOps);
}

// monitor group membership changes in e:
Status Ensemble_MaestroAD::monMonitor(Monitor &M, Entity &E, 
				      Monitor::monFunc) {
#ifdef DEBUG
  cerr << "Ensemble_MaestroAD::monMonitor" << endl;
#endif
  int i;
  EntityAddrLocal *EaddrL = (EntityAddrLocal*)E.local();
  if(!EaddrL) 
    gLog->panic("Ensemble_MaestroAD::monMonitor: entity not initialized");
  
  for(i = 0; i < MAX_MONITORS && EaddrL->monitors[i].mf_; i++);
  if(i == MAX_MONITORS)
    gLog->panic("Ensemble_MaestroAD::monMonitor: too many monitors");
  EaddrL->monitors[i] = M;

  return ST_OK;
}

// cancel a monitor:
Status Ensemble_MaestroAD::monCancel(Monitor& M) {

#ifdef DEBUG
  cerr << "Ensemble_MaestroAD::monCancel" << endl;
#endif
  int i;
  EntityAddrLocal *EaddrL = (EntityAddrLocal*)M.entity_.local();
  if(!EaddrL) 
    gLog->panic("Ensemble_MaestroAD::monCancel: monitor undefined");

  for(i = 0; i < MAX_MONITORS && EaddrL->monitors[i] != M; i++);
  if(i == MAX_MONITORS) 
    gLog->panic("Ensemble_MaestroAD::monCancel: no such monitor");

  EaddrL->monitors[i].mf_ = 0;
  return ST_OK;
}

// Create a semaphore. Ensemble_MaestroAD sets attaches its own semaphore data
// to Sema::priv_. This is transparent to the upper layers:
void Ensemble_MaestroAD::semaCreate(Sema &S, int Value) {
  if(S.getInitialized())
    gLog->panic("Ensemble_MaestroAD::semaCreate: semaphore already "
		"initialized");
  S.setInitialized();
  
  SemaWrap *SemaW = new SemaWrap(Value);
  S.local(SemaW);
}

// increment a semaphore. If the value was 0 and one or more threads are 
// blocked on this semaphore, unblock one thread:
void Ensemble_MaestroAD::semaInc(Sema &S) {
#ifdef DEBUG
  cerr << "Ensemble_MaestroAD::semaInc" << endl;
#endif
  if(!S.getInitialized())
    gLog->panic("Ensemble_MaestroAD::semaInc: semaphore uninitialized");
  
  Maestro_Semaphore *Semaphore = ((SemaWrap *)S.local())->sema;
  Semaphore->inc();
}

// decrement a semphore. Trying to decrement a semaphore whose value is 0
// blocks the issuing thread.
void Ensemble_MaestroAD::semaDec(Sema &S) {
#ifdef DEBUG
  cerr << "Ensemble_MaestroAD::semaDec" << endl;
#endif
#ifdef PURIFY
  purify_printf_with_call_chain("Ensemble_MaestroAD::semaDec");
#endif
  
  if(!S.getInitialized())
    gLog->panic("Ensemble_MaestroAD::semaDec: semaphore uninitialized");
  
  Maestro_Semaphore *Semaphore = ((SemaWrap *)S.local())->sema;
  Semaphore->dec();
}

// Create a lock. 
void Ensemble_MaestroAD::lockCreate(Lock &L) {
  if(L.getInitialized())
    gLog->panic("Ensemble_MaestroAD::lockCreate: Lock already "
		"initialized");
  L.setInitialized();
  
  LockWrap *Lck = new LockWrap;
  L.local(Lck);
}

// acquire a lock
void Ensemble_MaestroAD::lockAcquire(Lock &L) {
  if(!L.getInitialized())
    gLog->panic("Ensemble_MaestroAD::lockAcquire: Lock uninitialized");

  Maestro_Lock *Lck = ((LockWrap *)L.local())->lock;
  Lck->lock();
}

// release a lock
void Ensemble_MaestroAD::lockRelease(Lock &L) {

#ifdef PURIFY
  purify_printf_with_call_chain("Ensemble_MaestroAD::lockRelease");
#endif
  if(!L.getInitialized()) {
    LockWrap *LW = (LockWrap *)L.local();
    if(!LW) {
      gLog->panic("Ensemble_MaestroAD::lockRelease: lock uninitialized");
    } else {
      gLog->panic("Ensemble_MaestroAD::lockRelease: init and LW don't agree");
    }
  }

  Maestro_Lock *Lck = ((LockWrap *)L.local())->lock;
  Lck->unlock();
}

void Ensemble_MaestroAD::
extractLocation(const Electra_EndpID *Id, 
		char Location[HOT_ENDP_MAX_NAME_SIZE]) {

  char Name[HOT_ENDP_MAX_NAME_SIZE];

  Id->GetEndpID(Name);

  // remove the all the characters preceding the first ':'
  char *FirstRemoved = ::strchr(Name, ':');

  // increment past the ':'
  FirstRemoved++;  

  char *SecondRemoved = ::strchr(FirstRemoved, ':');
  int Length = SecondRemoved - FirstRemoved;
  strncpy(Location, FirstRemoved, Length);
  Location[Length]='\0';
}

Boolean Ensemble_MaestroAD::isLocalEntity(const Maestro_EndpID &E1, 
					  const Maestro_EndpID &E2) {

  char Loc1[HOT_ENDP_MAX_NAME_SIZE], Loc2[HOT_ENDP_MAX_NAME_SIZE];
    
  extractLocation((Electra_EndpID *)&E1, Loc1);
  extractLocation((Electra_EndpID *)&E2, Loc2);

  return (::strcmp(Loc1, Loc2) == 0);
}
