(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* RAND.ML: Randomly multicast/send messages and fail *)
(* Author: Mark Hayden, 6/95 *)
(**************************************************************)
open Ensemble
open Util
open Arge
open View
open Appl_intf
(**************************************************************)
let name = Trace.source_file "RAND"
let failwith s = failwith (failmsg name s)
(**************************************************************)

let quiet	= ref false
let size	= ref 100

(**************************************************************)

type action = ACast | ASend | ALeave

let policy nmembers thresh =
  let next = (Random.float 1.0) *. (float nmembers)
  and p = Random.float 1.0 in
  let action =
    if nmembers >= thresh 
    && p < 0.1
    then ALeave
    else if p < 0.5
    then ACast
    else ASend
  in (action,(Time.of_float next))

(**************************************************************)

type state = {
  mutable ls		: View.local ;
  mutable vs		: View.state ;
  mutable time		: Time.t ;
  mutable next_action 	: Time.t ;
  mutable last_view	: Time.t
}

(**************************************************************)

let dump (ls,vs) s =
  eprintf "RAND:dump:%s\n" ls.name ;
  eprintf "  time=%s, last_view=%s\n" 
    (Time.to_string s.time) (Time.to_string s.last_view) ;
  eprintf "  rank=%d, nmembers=%d\n" ls.rank ls.nmembers

let max_time = ref Time.zero

(**************************************************************)

let interface policy thresh on_exit (init_ls,init_vs) time merge =
  let s = {
    ls			= init_ls ;
    vs			= init_vs ;
    time		= time ;
    last_view		= time ;
    next_action 	= Time.zero
  } in
(*
  let msg () = 
    let s = String.create (Random.int !size) in
    let d = Digest.string s in
    (s,d)
  in

  let check (s,d) =
    let d' = Digest.string s in
    if d <> d' then
      failwith "bad message"
  in
*)
  let msg () = () in
  let check _ = () in

  let recv_cast o m = 
(*    printf "RAND:%s:got from %d\n" s.ls.name o ;
*)
    check m ; []
  and recv_send _ m = check m ; []
  and block () 	     =
(*
    printf "RAND:%s:(blocking)\n" s.my_name ; flush std_out ;
*)
    []

  and heartbeat time =
    max_time := max time !max_time ;
    s.time <- time ;
    let acts = ref [] in
(* BUG?*)
    if s.last_view <> Time.invalid
    && time >= Time.add s.last_view (Time.of_float 600.0) 
    then (
      dump (s.ls,s.vs) s ;
      s.last_view <- time ;
      acts := Dump :: !acts
    ) ;
(**)
    if time >= s.next_action then (
      let (action,next) = policy s.ls.nmembers thresh in
      s.next_action <- Time.add time next ;

      match action with
      | ACast ->
(*
	  eprintf "RAND:%s:casting\n" s.ls.name ;
*)
      	  acts := [Cast(msg())] @ !acts
      | ASend ->
	  let dest = Random.int s.ls.nmembers in
	  if dest <> s.ls.rank then
	    acts := [Send([dest],(msg()))] @ !acts
      | ALeave ->
	  acts := [Cast(msg());Cast(msg());Cast(msg());Leave] @ !acts ;
	  if !verbose then (
    	    printf "RAND:%s:Leaving(nmembers=%d)\n" s.ls.name s.ls.nmembers
	  )
    ) ;
    !acts

  and block_recv_cast _ _ 	= ()
  and block_recv_send _ _ 	= ()
  and block_view (ls,vs)        = 
    let merge = sequence merge in
    [ls.rank,merge]
  and block_install_view _ states =
    s.last_view <- !max_time ;
    states
  and unblock_view (ls,vs) _ =
(*
  printf "RAND:max_time=%f\n" !max_time ;
*)
(*
    eprintf "RAND:%s: view_id=%s\n" ls.name (View.string_of_id vs.view_id) ;
*)
    s.last_view		<- !max_time ;
    s.ls		<- ls ;
    s.vs		<- vs ;
    s.next_action	<- Time.zero ;
    if (not !quiet) 
    && (!verbose || ls.rank = 0) then
      printf "RAND:%s:(time=%s) View=(xfer=%s)%s\n"
	ls.name (Time.to_string !max_time(*s.time*)) (string_of_bool vs.xfer_view)
	(View.to_string vs.view) ;

    if vs.xfer_view then 
      [XferDone] 
    else 
      [Cast(msg())]
  and exit () =
    on_exit ()
  in full ((*Appl_intf.debug*) {
    recv_cast 		= recv_cast ;
    recv_send 		= recv_send ;
    heartbeat 		= heartbeat ;
    heartbeat_rate	= Time.of_float 0.3 ;
    block 		= block ;
    block_recv_cast 	= block_recv_cast ;
    block_recv_send 	= block_recv_send ;
    block_view  	= block_view ;
    block_install_view 	= block_install_view ;
    unblock_view 	= unblock_view ;
    exit 		= exit
  })

(**************************************************************)

let thresh   	= ref 5
let size	= ref 100
let merge       = ref 0

(**************************************************************)

let run () =
  let props = Property.Drop :: Property.Debug :: Property.vsync in
  let props = List.map Property.string_of_id props in
  let props = String.concat ":" props in

(*
  Trace.set_trace "merge" (open_out "rand.merge") ;
*)
  Arge.set_default alarm "Netsim" ;
  Arge.set_default modes "Netsim" ;
  Arge.set_default nmembers "7" ;
  Arge.set_default properties props ;
  Endpt.short_strings () ;

  let undoc = ": undocumented" in
  Arge.parse [
    "-merge",   Arg.Int(fun i -> merge := i),undoc ;
    "-n",	Arg.String(Arge.set_default nmembers),undoc ;
    "-t",	Arg.Int(fun i -> thresh := i),undoc ;
    "-s",	Arg.Int(fun i -> size := i),undoc ;
    "-quiet",	Arg.Set(quiet),undoc
  ]  (Arge.badarg name) 
  "rand: random failure generation test program" ;

  let gettime () = 
    let time = Alarm.gettime (Alarm.get ()) in
    if !max_time = Time.zero then 
      max_time := time ;
    time
  in

  let rec instance () =
    let vs = Appl.default_info "rand" in
    let time = gettime() in
    let interface = interface policy !thresh instance vs time !merge in
    Appl.config interface vs
  in

  for i = 1 to Arge.get nmembers do
    instance ()
  done ;

  Sys.catch_break true ;
  Appl.main_loop ()

let _ = Appl.exec ["rand"] run

(**************************************************************)
