(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PERF.ML: round-trip performance demo *)
(* Author: Mark Hayden, 8/95 *)
(**************************************************************)
open Ensemble
open Util
open Appl
open View
open Appl_intf
open Proxy
(**************************************************************)
let name = Trace.source_file "PERF"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
external (=|) : int -> int -> bool = "%eq"
external (<>|) : int -> int -> bool = "%noteq"
external (>=|) : int -> int -> bool = "%geint"
external (<=|) : int -> int -> bool = "%leint"
external (>|) : int -> int -> bool = "%gtint"
external (<|) : int -> int -> bool = "%ltint"
(**************************************************************)

let meter name often max callb =
  let roll = Queue.create () in
  for i = 0 to 10 do
    Queue.add None roll
  done ;
  let cntr = ref 0 in
  let time = ref 0.0 in
  let init = Hsys.gettimeofday () in
  let min_len  = ref 10000 in
  fun () ->
    if (!cntr mod often) =| 0 then (
      let now = Hsys.gettimeofday () in
      let roll_rate =
	match Queue.take roll with
	|  None -> 0.0
	|  Some(roll_cntr,roll_time) ->
	        (now -. roll_time) /. (float (!cntr - roll_cntr))
      in
      Queue.add (Some(!cntr,now)) roll ;

      let rate = 
      	if !cntr = 0 then 0.0 else
	  ((now -. !time) /. (float often))
      in
      let s = callb () in
      printf "PERF:%s #=%06d rate=%8.6f roll=%8.6f ctime=%8.6f %s\n" 
        name !cntr rate roll_rate (now -. init) s ;
      time := now ;
      match max with 
      |	None -> ()
      |	Some max ->
	  if !cntr >= max then (
	    Timestamp.print () ;
	    exit 0
	  )
    ) ;
    incr cntr

(**************************************************************)

type iov_action = (Iovecl.t,Iovecl.t) action

type test = {
  start : iov_action list ;
  hbt : Time.t -> iov_action list ;
  cast : origin -> iov_action list ;
  send : origin -> iov_action list
}  

type testrun = View.full -> test option

let fast_iov len =(* Hack! *)
  let len = ceil_word len in
  let msg = Iovec.create name (len + 8) in
  let msg = Iovec.read name msg (fun buf ofs len ->
    Iovec.sub name msg (ofs+8) (len-8))
  in
  [|msg|]

(**************************************************************)
(**************************************************************)

let rt size nrounds (ls,vs) =
  let start_t = Hsys.gettimeofday() in
  let round= ref 0 in

  let partner =
    match ls.rank with
    |  0 -> 1
    |  1 -> 0
    |  _ -> -1
  in

  let msg = fast_iov size in
  let msg_ref () = Iovecl.ref name msg in
  let msg = [Cast msg] in
  
  let print_result () =
    let end_t = Hsys.gettimeofday () in
    let time = end_t -. start_t in
    Timestamp.print () ;
(*
    Arge.stats () ;
    Profile.end () ;
*)
    printf "real-time: %.6f\n"time ;
    printf "latency/round: %.6f\n"      (time /. (float nrounds)) ;
    printf "msgs/sec: %.3f\n"((float nrounds) /. time)
  in

  let cast origin =
    if origin =| partner then (
      incr round ;
      if !round >=| nrounds then (
        print_result () ;
	msg_ref () ;
        msg @ [Leave]
      ) else (
	msg_ref () ; 
	msg
      )
    ) else []
  and send _ = failwith "error1"
  and hbt _ = []
  and start = 
    if ls.rank =| 0 then (
      msg_ref () ;
      msg 
    ) else []
  in 

  Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
} 

(**************************************************************)
(**************************************************************)

let latency size nrounds (ls,vs) =
  let start_t = Hsys.gettimeofday() in
  let round = ref 0 in

  let src = (pred ls.rank + ls.nmembers) mod ls.nmembers in
  let msg = fast_iov size in
  let msg_ref () = Iovecl.ref name msg in
  let msg = [Cast msg] in
  
  let print_result () =
    let end_t = Hsys.gettimeofday () in
    let time = end_t -. start_t in
    Timestamp.print () ;
    printf "real-time: %.6f\n"time ;
    printf "latency/round: %.6f\n" (time /. (float nrounds))
  in

  let cast origin =
    incr round ;
    if origin =| src then (
      if !round >=| nrounds then (
        print_result () ;
	msg_ref () ;
        msg @ [Leave]
      ) else (
	msg_ref () ; 
	msg
      )
    ) else []
  and send _ = failwith "sanity"
  and hbt _ = []
  and start = 
    if ls.am_coord then (
      msg_ref () ;
      msg 
    ) else []
  in 

  Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
  } 

(**************************************************************)
(**************************************************************)

let sleep = ref 0.0

let ring nmsgs size nrounds (ls,vs) =
  let start_t = Hsys.gettimeofday () in
  let end_t = ref zero in
  let rnd = ref 0 in
  let got = ref 0 in
  let got_end = ref false in
  let gc = Gc.stat () in

  let msg = fast_iov size in
  let msgs = Array.to_list (array_create name nmsgs (Cast msg)) in
  let msg_ref () =
    for i = 1 to nmsgs do
      Iovecl.ref name msg
    done
  in
  msg_ref () ;
  let msgs_per_rnd = (pred ls.nmembers) * nmsgs in

  if not (Arge.get Arge.quiet) then
    printf "RING:got view, nmembers=%d, rank=%d\n" ls.nmembers ls.rank ;
  if not (Arge.get Arge.quiet) then
    printf "RING:starting\n" ; 
    (*Profile.start () ;*)

  let print_result () =
    end_t := Hsys.gettimeofday () ;
    let gc' = Gc.stat () in
    let time = !end_t -. start_t in
    let total_msgs = nrounds * nmsgs * ls.nmembers in
    (*Profile.end () ;*)
    (*Timestamp.print () ;*)

    let latency = time /. (float nrounds) in
    let msgs_sec = (float total_msgs) /. time in
    let msgs_mbr_sec = (float total_msgs) /. time /. (float ls.nmembers) in
    if not (Arge.get Arge.quiet) then (
      Timestamp.print () ;
      printf "latency/round: %.6f\n"    latency ;
      printf "msgs/sec: %.3f\n"msgs_sec ;
      printf "msgs/mbr/sec: %.3f\n"     msgs_mbr_sec ;
      printf "real-time: %.6f\n"time
(*
  printf "#gcs: %d\n"(gc'.minor_collections - gc.minor_collections) ;
  printf "minorwords: %d\n"(gc'.minor_words - gc.minor_words) ;
*)
    ) else (
      printf "OUTPUT:n=%d, rank=%d, protocol=%s, latency=%.6f\n"
      ls.nmembers ls.rank (Proto.string_of_id vs.proto_id) latency
    )
  in

  let cast origin =
    if origin <> ls.rank then (
      incr got ;
      if !got = msgs_per_rnd then (
	got := 0 ;
	incr rnd ;
	if !rnd >= nrounds then (
	  print_result ();
	  if !sleep = zero then
	    [Leave]
	  else (
	    (*        printf "\nRING: going to idle for %f sec...\n" !sleep;*)
	    got_end := true ;
	    []
	  )
	) else ( msg_ref () ; msgs )
      ) else []
    ) else []
  and send _ = []
  and hbt _ =
    if (Arge.get Arge.quiet) then
      printf "RING:heartbeat (%d/%d of %d)\n" !got msgs_per_rnd !rnd ;
    if !got_end then (
      let dif = Hsys.gettimeofday () -. !end_t in
(* printf "RING heartbeat: %f elapsed since end\n" dif ;*)
      if dif > !sleep then
        [Leave]
      else (
(*        printf "RING: %f elapsed since end\n" dif ;*)
 	[]
      )
    )
    else []
  and start = msg_ref () ; msgs 
  in Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
} 

(**************************************************************)
(**************************************************************)

let chain nrounds size nchains (ls,vs) =
  let round = ref 0 in
  let start_t = Hsys.gettimeofday() in
  let end_t= ref zero in

  let chain = array_create name nrounds 0 in
  (* Initialize the chain.
   *)
  for i = 0 to pred nrounds do
    chain.(i) <- i mod ls.nmembers
  done ;
 
  (* Scramble the entries.
   *)
  Random.init 1023 ;
  for i = 1 to pred nrounds do
    let j = Random.int i in
    let tmp = chain.(i) in
    chain.(i) <- chain.(j) ;
    chain.(j) <- tmp
  done ;
  printf "CHAIN:%s\n" (string_of_int_array chain) ;

  let msg = [Cast (fast_iov size)] in

  let print_result () =
    end_t := Hsys.gettimeofday () ;
    let time = !end_t -. start_t in
    let latency = time /. (float nrounds) in
    Timestamp.print () ;
    if not (Arge.get Arge.quiet) then (
      printf "real-time: %.6f\n"time ;
      printf "latency/round: %.6f\n"    latency
    ) else (
      printf "OUTPUT:n=%d, rank=%d, c=%d, protocol=%s, latency=%.6f\n"
      ls.nmembers ls.rank nchains (Proto.string_of_id vs.proto_id) latency
    )
  in

  if not (Arge.get Arge.quiet) then
    printf "CHAIN:got view, nmembers=%d, rank=%d\n" ls.nmembers ls.rank ;

  if not (Arge.get Arge.quiet) then
    printf "CHAIN:starting\n" ; 
  let start_t = Hsys.gettimeofday () in

  let cast origin =
    incr round ;
    printf "round=%d, %d\n" !round (Array.length chain) ;
    if !round = nrounds then (
      print_result () ;
      [Leave]
    ) else if chain.(pred !round) = ls.rank then (
      msg
    ) else (
      []
    )
  and hbt _ = 
    if not (Arge.get Arge.quiet) then
      printf "CHAIN:heartbeat:round:%d/%d\n" !round nrounds ;
    []
  and send _ = []
  and start = 
    let m = array_create name (nchains / ls.nmembers) msg in
    let m = Array.to_list m in
    let m = List.flatten m in
    if ls.rank < nchains then
      msg @ m
    else
      m
  in Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
} 

(**************************************************************)
(**************************************************************)

let one_to_n size rate terminate_time (ls,vs) =
  let msg = Cast (fast_iov size) in
  let next = ref (Time.of_float (Hsys.gettimeofday ())) in
  let stop = Time.add !next terminate_time in
   let ctr = meter "1-n" 100 None (fun () -> if ls.rank=0 then "coord" else "") in
  let start =
    if ls.rank = 0 then 
      [msg]
    else []
  and cast _ =
    ctr () ;
    []
  and send _ = []
  and hbt time =
    if time > stop then [Leave]
    else
      if ls.rank = 0 then (
      	let msgs = ref [] in
      	while time >= !next do
    	  ctr () ;
	  msgs := msg :: !msgs ;
	  next := Time.add !next rate ;
      	done ;
      	!msgs
    	) else []
  in Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
} 

(**************************************************************)
(**************************************************************)

let m_to_n size rate sender vs =
  let msg = Cast (fast_iov size) in
  let next = ref (Time.of_float (Hsys.gettimeofday ())) in
  let ctr = meter "m-n" 100 None (fun () -> if sender then "sender" else "") in
  let start =
    if sender then 
      [msg]
    else []
  and cast _ =
    ctr () ;
    []
  and send _ = []
  and hbt time =
    if sender then (
      let msgs = ref [] in
      while time >= !next do
    	ctr () ;
	msgs := msg :: !msgs ;
	next := Time.add !next rate ;
      done ;
      !msgs
    ) else []
  in Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
} 

(**************************************************************)
(**************************************************************)

let rpc size rounds (ls,vs) =
  let msg = fast_iov size in
  let request = [Cast(msg)] in
  let reply = [Send([0],msg)] in
  let count = ref 0 in
  let ctr = meter "rpc" 100 (Some rounds) (fun () -> if ls.rank=0 then "coord" else "") in
  let start =
    if ls.rank = 0 then (
      count := pred ls.nmembers ;
      request 
    ) else []
  and cast =
    if ls.rank <> 0 then (
      fun _ -> 
    	ctr () ;
	reply 
    ) else (fun _ -> [])
  and send =
    if ls.rank = 0 then (
      fun _ ->
	decr count ;
	if !count = 0 then (
	  ctr () ;
	  count := pred ls.nmembers ;
	  request
	) else []
    ) else (fun _ -> [])
  and hbt _ = []
  in Some{
    start = start ;
    cast = cast ;
    send = send ;
    hbt = hbt
  } 

(**************************************************************)
(**************************************************************)

let switch () =
  let count = ref 0 in
  let start = ref (-1.0) in
  fun (ls,vs) ->
    if !start = (-1.0) then
      start := Hsys.gettimeofday () ;
    incr count ;
    if ls.rank = 0 && (!count mod 100) = 0 then (
      let now = Hsys.gettimeofday () in
      printf "PERF:switch: %5d views %6.3f sec/view\n" 
        !count ((now -. !start) /. float !count) ;
    ) ;
    
    let start = 
      if ls.rank = 0 then (
(*
	eprintf "PERF:switching\n" ;
*)
	[Protocol vs.proto_id]
      ) else
	[]
    and cast _ = []
    and send _ = []
    and hbt time = []
    in Some{
      start = start ;
      cast = cast ;
      send = send ;
      hbt = hbt
  } 

(**************************************************************)
(**************************************************************)

let wait nmembers f =
  let correct_number = ref false in
  fun (ls,vs) ->
    if ls.nmembers = nmembers then (
      correct_number := true ;
      if !verbose then
      	printf "PERF:wait:got all members:cur=%d\n" ls.nmembers ;
      f (ls,vs)
  ) else if !correct_number
  then (
    eprintf "New view formed before my timer expired.\n" ;
    exit 0
    ) else (
    if !verbose then
      printf "PERF:wait:wrong # members:cur=%d:goal=%d\n"
      ls.nmembers nmembers ;
    None
    ) 

let once f =
  let ran = ref false in
  fun vs ->
    if !ran then (
      Timestamp.print () ;
      exit 0
    ) ;
    let i = f vs in
    if i <> None then
      ran := true ;
    i

(**************************************************************)

let interface rate xfer test =
  let null = fun _ -> [] in
  let cast = ref null in
  let send = ref null in
  let hbt = ref null in

  let recv_cast o _ = !cast o
  and recv_send o _ = !send o
  and block () = []
  and heartbeat t = !hbt t
  and block_recv_cast _ _ = ()
  and block_recv_send _ _ = ()
  and block_view (ls,vs) = 
    match xfer,ls.rank with
    |  false, 0 -> 
	List.map (fun i -> (i,[|Iovec.empty name|])) (Util.sequence ls.nmembers)
    |  false, _ -> []
    |  true, _ -> [ls.rank,[|Iovec.empty name|] ]
  and block_install_view _ _ = [| Iovec.empty name |]
  and unblock_view (ls,vs) _ =
    eprintf "PERF:view:%s:nmembers=%d\n" ls.name ls.nmembers ;
(*
    printf "PERF:view change:nmembers=%d, rank=%d\n" ls.nmembers ls.rank ;
    printf "PERF:view=%s\n" (Endpt.string_of_id_list vs.view) ;
    printf "PERF:view_id=%s\n" (View.string_of_id vs.view_id) ;
*)
    match test (ls,vs) with
    |  None -> 
	cast := null ;
	send := null ;
	hbt := null ;
	[]
    |  Some(cb) ->
	cast := cb.cast ;
	send := cb.send ;
	hbt := cb.hbt ;
	cb.start
  and exit () = exit 0
  in iovl ((*Appl_intf.debug*){
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    heartbeat           = heartbeat ;
    heartbeat_rate      = rate ;
    block               = block ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
} )

(**************************************************************)

let empty =
  let recv_cast _ () = []
  and recv_send _ () = []
  and block () = []
  and heartbeat _ = []
  and block_recv_cast _ _ = ()
  and block_recv_send _ _ = ()
  and block_view (ls,vs) = [ls.rank,()]
  and block_install_view _ _ = ()
  and unblock_view _ () = []
  and exit () = ()
  in full {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    heartbeat           = heartbeat ;
  heartbeat_rate= Time.of_float 100.0 ;(* a big number *)
    block               = block ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
} 

(**************************************************************)

let groupd wait group endpt counter send =
  let log = Trace.log "GROUPD" (Group.string_of_id (um_group group)) in
  let recv msg = match msg with
  |  View(_,view) ->
      log (fun () -> "View()") ;
      let n = List.length view in
      if (not wait) 
      || n = Arge.get Arge.nmembers then (
      	counter n ;
      	if List.hd view = endpt then
	    send (Fail[])
      )
  |  Sync ->
      log (fun () -> "Sync") ;
      send Synced
  |  Failed _ ->
      ()
  in
  recv

(**************************************************************)
(**************************************************************)

let use_locator file am_server =
  if am_server then (
    let host = Hsys.gethostname () in
    (* Had problems opening for append mode.
     *)
    Sys.command (sprintf "echo %s >>%s" host file) ;
(*
	let co = open_out_gen [Open_append;Open_text;Open_creat] 0o666 file in
        output_string co (sprintf "%s\n" host) ;
	close_out co
*)
    ) ;
  
  (* Sleep for 5 seconds for everyone who wants to 
   * to write their names in the file.
   *)
  Unix.sleep 5 ;

  (* Now read from the file.
   *)
  let hosts =
    let l = ref [] in
    let ci = open_in file in
    begin try while true do
      let host = input_line ci in
      l := host :: !l
    done with End_of_file -> () end ;
    close_in ci ;
    (String.concat ":" !l)
  in

  (* Override the environment with these host
   * lists.
   *)
  Arge.set_default Arge.groupd_hosts hosts ;
  Arge.set_default Arge.gossip_hosts hosts ;

  (* If I'm a server then also initialize the 
   * server stacks.
   *)
  if am_server then (
    let vs = Appl.default_info "groupd" in
    let port = Arge.check name Arge.groupd_port in
    let m, vs, intf = Manage.create vs in
    Appl.config intf vs ;
    Manage.proxy_server m port ;
    
    let vs = Appl.default_info "gossip" in
    let port = Arge.check name Arge.gossip_port in
    let vs, interface = Reflect.init vs port true in
    Appl.config interface vs ;
  )

(**************************************************************)
(**************************************************************)

let size        = ref 0
let nrounds     = ref 100
let msgs_per_round = ref 1
let nchains     = ref 1
let prog        = ref ""
let nlocal      = ref 1
let mlocal      = ref false
let sender      = ref false
let rate_r      = ref (Time.of_float 0.01)
let heartbeat_rate = ref (Time.of_float 10.0)
let wait_r      = ref true
let xfer_r      = ref true
let ngroups     = ref 1
let once_r      = ref false
let stoptime    = ref (Time.of_float 0.00)
let locator     = ref None
let locator_server = ref false
let proto_id    = ref None
let terminate_time = ref (Time.of_float 40.0)

let run () =
  let undoc = "undocumented" in
  Arge.parse [
    "-n",Arg.String(Arge.set_default Arge.nmembers), ": # of members" ;
    "-s",Arg.Int(fun i -> size := i), ": size of messages" ;
    "-r",Arg.Int(fun i -> nrounds := i), ": # of rounds" ;
    "-c",Arg.Int(fun i -> nchains := i),undoc ;
    "-k", Arg.Int(fun i -> msgs_per_round := i),undoc ;
    "-local",   Arg.Int(fun i -> nlocal := i), ": # of local members" ;
    "-rate",    Arg.Float(fun i -> rate_r := Time.of_float i),undoc ;
    "-stoptime",    Arg.Float(fun i -> stoptime := Time.of_float i),undoc ;
    "-sleep",Arg.Float(fun i -> sleep := i),undoc ;
    "-nowait",  Arg.Clear(wait_r),undoc ;
    "-noxfer",  Arg.Clear(xfer_r),undoc ;
    "-ngroups", Arg.Int(fun i-> ngroups :=i),"";
    "-prog",    Arg.String(fun s -> prog := s), undoc ;
    "-sender",  Arg.Set sender, " : set this process as a sender";
    "-mlocal",  Arg.Set mlocal, " : use local manager" ;
    "-once",    Arg.Set once_r, " : only run test once" ;
    "-locator", Arg.String(fun s -> locator := Some s), " : use locator (for SP2)" ;
    "-locator_server", Arg.Set locator_server, " : act as locator server (for SP2)" ;
    "-protocol",Arg.String(fun s -> proto_id := Some s), " : set protocol" ;
    "-terminate_time", Arg.Float(fun f -> terminate_time := Time.of_float f), undoc  ]  (Arge.badarg name) "perf: Ensemble performance testing" ;

  if not (Arge.get Arge.quiet) then
    printf "nmembers=%d size=%d round=%d rate=%s\n"
      (Arge.get Arge.nmembers) !size !nrounds (Time.to_string !rate_r) ;

  Alarm.choose (Arge.get Arge.alarm) Appl.root_sched ;
  let alarm = Alarm.get () in  
  let sleep t f =
    let now = Alarm.gettime alarm in
    Alarm.schedule (Alarm.alarm alarm f) (Time.add now t)
  in

  (* Check if we are using the file system to meet up.
   *)
  begin match !locator with
  | Some file -> use_locator file !locator_server
  | None -> ()
  end ;

  let ready () =
    for i = 1 to !nlocal do
      let (ls,vs) = Appl.default_info "perf" in
      let vs =
	match !proto_id with
	| None -> vs
	| Some proto -> View.set vs [Vs_proto_id (Proto.id_of_string proto)] 
      in

      let prog = match !prog with
      | "ring"  -> ring !msgs_per_round !size !nrounds
      | "rt"    -> rt !size !nrounds
      | "chain" -> chain !nrounds !size !nchains
      | "latency" -> latency !size !nrounds
      | "rpc"   -> 
	  once_r := true ;
	  rpc !size !nrounds
      | "switch" -> 
	  xfer_r := false ;
	  switch ()
      | "1-n" -> 
	  heartbeat_rate := !rate_r ;
	  one_to_n !size !rate_r !terminate_time
      | "m-n"   -> 
	  heartbeat_rate := !rate_r ;
	  m_to_n !size !rate_r !sender
      | "empty" ->
	  let start_t = Hsys.gettimeofday () in
	  for i = 1 to !nrounds do
	    let (ls,vs) = Appl.default_info "perf" in
	    Appl.config empty (ls,vs) ;
	  done ;
	  let end_t = Hsys.gettimeofday () in
	  eprintf "PERF:empty: #views=%5d %8.5f sec/view\n" 
	    !nrounds ((end_t -. start_t) /. float !nrounds) ;
	  exit 0
      | "groupd" ->
	  let m = 
	    if !mlocal then (
	      let m, vs, intf = Manage.create (ls,vs) in
	      Appl.config intf vs ;
	      m
	    ) else (
	      Appl.init_groupd ()
	    )
	  in
	  for i = 1 to !nlocal do
  (*
    let gbl_counter = rate "Groupd" 50 None (fun () -> "") in
   *)
	    for g = 1 to !ngroups do
	      let name = sprintf "Perf%03d" g in
	      let group = Group.named name in
	      let group = ma_group group in
	      let nmem = ref 0 in
	      let counter = meter name (if g = 1 then 25 else 400) None 
		(fun () -> sprintf "nmembers=%d" !nmem)
	      in
	      let counter n =
		nmem := n ;
		counter ()
	      in

	      let endpt = Endpt.id () in
	      let endpt = (endpt,(Appl.addr (Arge.get Arge.modes))) in
	      let endpt = ma_endpt endpt in

	      let send_r = ref (fun _ -> failwith "sanity") in
	      let send msg = !send_r msg in
	      let recv = groupd !wait_r group endpt counter send in
	      send_r := Manage.join m group endpt recv
	    done
	  done ;
	  Appl.main_loop () ;
	  exit 1

      |  _ -> failwith "unknown performance test"
      in

      let prog =
	if !wait_r then
	  wait (Arge.get Arge.nmembers) prog
	else prog
      in

      let prog =
	if !once_r then
	  once prog
	else prog
      in

      if Arge.get Arge.perturb then (
	let disable = 
	  Appl.perturb alarm
	    (Arge.get Arge.perturb_doze)
	    (Arge.get Arge.perturb_rate)
	in ()
      ) ;

      let interface = interface !heartbeat_rate !xfer_r prog in
      Appl.config interface (ls,vs) ;
    done ;

    (* timeout code, as suggested by Mark.... *)
    let timeout = Time.add (Alarm.gettime alarm) (Time.of_float 123.0) in 
    let alarm = Alarm.alarm alarm (fun _ -> exit 0 ) in 
    if (Time.to_float !stoptime) <> 0.0 then
      Alarm.schedule alarm 
      (Time.add (Time.of_float (Hsys.gettimeofday ())) !stoptime);
    (*  end of mark's suggestion *) 
  in

  ready () ;
  Appl.main_loop ()

let _ = 
  Appl.exec ["perf"] run

(**************************************************************)
