(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* LENSEMBLE.ML: text-based interface to Ensemble *)
(* Author: Mark Hayden, 8/95 *)
(**************************************************************)
open Ensemble
open Hsys
open Util
open View
open Appl_intf
(**************************************************************)
let name = Trace.source_file "ENSEMBLE"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type taction =
  | TCast of string
  | TSend of string * string
  | TLeave
  | TBlock

(* This function returns the interface record that defines
 * the callbacks for the application.
 *)
let intf (ls,vs) alarm sync =

  (* This is the buffer used to store typed input prior
   * to sending it.
   *)
  let buffer 	= ref [] in
  
  (* View_inv is a hashtable for mapping from endpoints to ranks.
   *)
  let view_inv	= ref (Hashtbl.create 10) in
  let view	= ref [||] in

  (* This is set below in the handler of the same name.  Get_input
   * calls the installed function to request an immediate callback.
   *)
  let async = Appl.async (vs.group,ls.endpt) in

  let action act =
    buffer := act :: !buffer
  in

  (* Install handler to get input from stdin.
   *)
  let get_input () =
    try 
      List.iter (fun line ->
	let com,rest = strtok line " \t" in
	begin match com with
	  | "send" -> 
	      let dest,rest = strtok rest " \t" in
	      begin try Hashtbl.find !view_inv dest with
	      | _ -> (
		  eprintf "ENSEMBLE:bad endpoint name:'%s'\n" dest ;
		  failwith "bad endpoint name"
		)
	      end ;
	      action (TSend(dest,rest))
	  | "cast" ->
	      action (TCast(rest))
	  | "leave" ->
	      action (TLeave)
	  | "blocked" ->
	      action (TBlock)
	  | _ -> 
	      eprintf "ENSEMBLE:bad command:line='%s'\n" line ;
	      failwith "bad command"
	end ;
	async ()
      ) (read_lines ())
    with e -> (
      eprintf "ENSEMBLE:error:%s\n" (string_of_exn e) ;
      exit 1
    )
  in
  Alarm.add_sock alarm (Hsys.stdin()) (Handler0 get_input) ;

  (* If there is buffered data, then return a Cast action to
   * send it.
   *)
  let check_actions () =
    let actions = ref [] in
    List.iter (function
      | TCast msg -> 
      	  actions := Cast msg :: !actions
      | TSend(dest,msg) -> (
	  try 
      	    let dest = Hashtbl.find !view_inv dest in
	    actions := Send([dest],msg) :: !actions
	  with Not_found -> ()
      	)
      | TLeave -> 
      	  actions := Leave :: !actions
      |	TBlock ->
	  actions := Block true :: !actions
    ) !buffer ;
    buffer := [] ;
    !actions
  in

  (* Print out the name.
   *)
  printf "endpt %s\n" (Endpt.string_of_id ls.endpt) ;

  (* Various application interface handlers.
   *)
  let recv_cast origin msg =
    printf "cast %d %s\n" origin msg ;
    check_actions ()
  and recv_send origin msg = 
    printf "send %d %s\n" origin msg ;
    check_actions ()
  and block () = 
    if sync then (
      (* Disable blocking.
       *)
      printf "block\n" ;
      [Block false]
    ) else (
      check_actions ()
    )
  and heartbeat _ =
    check_actions ()
  and block_recv_cast origin msg =
    printf "cast %d %s\n" origin msg
  and block_recv_send origin msg =
    printf "send %d %s\n" origin msg
  and block_view (ls,vs) = [ls.rank,()]
  and block_install_view _ _ = ()
  and unblock_view (ls,vs) names' =
    view := Array.map Endpt.string_of_id vs.view ;
    view_inv := Hashtbl.create 10 ;
    printf "view %d %d %s\n"
      ls.nmembers ls.rank
      (String.concat " " (Array.to_list !view)) ;
    let view = vs.view in
    for i = 0 to pred ls.nmembers do
      Hashtbl.add !view_inv (Endpt.string_of_id view.(i)) i
    done ;
    check_actions ()
  and exit () =
    printf "exit\n" ;
    exit 0
  in full (Appl_intf.debug {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    heartbeat           = heartbeat ;
    heartbeat_rate      = Time.of_float 10.00 ;
    block               = block ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
  })

(**************************************************************)

let run () =
  let nice = ref false in
  let sync = ref false in

  (*
   * Parse command line arguments.
   *)
  Arge.parse [
    (*
     * Extra arguments go here.
     *)
    "-nice", Arg.Set(nice), ": use minimal resources" ;
    "-sync", Arg.Set(sync), ": synchronize on view changes"
  ] (Arge.badarg name) "line based ascii interface to ensemble" ;

  (*
   * Get default transport and alarm info.
   *)
  let (ls,vs) = Appl.default_info (Arge.get Arge.group_name) in
  let alarm = Alarm.get () in
  let endpt = ls.endpt in

  let vs =
    if not !nice then vs else 
      View.set vs [
        Vs_params [
	  "pr_stable_sweep",Param.Time(Time.of_float 3.0) ;
	  "pr_stable_fanout",Param.Int(5) ;
	  "pr_suspect_sweep",Param.Time(Time.of_float 3.0) ;
	  "pr_suspect_max_idle",Param.Int(20) ;
	  "pr_suspect_fanout",Param.Int(5) ;
	  "heal_wait_stable",Param.Bool(true) ;
	  "merge_sweep",Param.Time(Time.of_float 3.0) ;
	  "merge_timeout",Param.Time(Time.of_float 150.0) ;
	  "top_sweep",Param.Time(Time.of_float 10.0)
        ]
      ]
  in

  (*
   * Initialize the application interface.
   *)
  let interface = intf (ls,vs) alarm !sync in

  (*
   * Initialize the protocol stack, using the interface and
   * view state chosen above.  
   *)
  Appl.config interface (ls,vs) ;

  (*
   * Enter a main loop
   *)
  Appl.main_loop ()
  (* end of run function *)


(* Run the application, with exception handlers to catch any
 * problems that might occur.
 *)
let _ = Appl.exec ["ensemble"] run

(**************************************************************)
