(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* FIFO.ML *)
(* Randomly multicast/send messages, but in a way that will cause
 * group to deadlock if FIFO is not live. *)
(* Author: Mark Hayden, 7/95 *)
(**************************************************************)
open Ensemble
open Util
open Arge
open View
open Appl_intf
(**************************************************************)
let name = Trace.source_file "FIFO"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type message = NoMsg
| Token of Endpt.id * int
| Spew
| Ignore
| Other

(**************************************************************)

let nrecd = ref 0
and nsent = ref 0

(**************************************************************)

let intf my_endpt =
  let nmembers = ref (-1)
  and view     = ref [||]
  and time     = ref Time.zero
  and last_msg = ref Time.zero
  in

  let rec random_member () =
    let rank = Random.int !nmembers in
    if !view.(rank) = my_endpt then
      random_member ()
    else (
(*    eprintf "RANDOM_MEMBER:%s , %s\n" (Endpt.string_of_id endpt) (Endpt.string_of_id my_endpt) ;*)
      rank
    )
  in

  let recv_cast_send origin msg =
    let from = !view.(origin) in
    let act = ref [] in
    last_msg := !time ;
    incr nrecd ;
    if !verbose & (!nrecd mod 113) = 0 then (
      eprintf "FIFO:nsent=%d, nrecd=%d\n" !nsent !nrecd
(*    if !nrecd = 10000 then
	exit 0
*)
    ) ;

    (match msg with
    | Token(next,seqno) -> (
(*
	eprintf "TOKEN:%s -> %s (me=%s)\n" 
      	  (Endpt.string_of_id from) 
      	  (Endpt.string_of_id next) 
      	  (Endpt.string_of_id my_endpt) ;
*)
	if next = my_endpt then (
	  if Random.float 1.0 < 0.1 then (
	    nsent := !nsent + !nmembers - 1 ;
	    act := Cast(Spew) :: !act
	  ) ;

	  let next = random_member () in
	  let kind =
	    if Random.float 1.0 < 0.5 then (
	      incr nsent ;
	      act := Send([next],Token(!view.(next),seqno+1)) :: !act ; "send"
	    ) else (
	      nsent := !nsent + !nmembers - 1;
	      act := Cast(Token(!view.(next),seqno+1)) :: !act ; "cast"
	    )
	  in
	  if (seqno mod 87) = 0 then (
	     eprintf "FIFO:token=%d, %d->%d->%d (%s)\n"
	     seqno origin (array_index my_endpt !view) next kind
	  )
	)
      )
    | Spew -> (
	for i = 1 to Random.int 5 + 5 do
	  if Random.float 1.0 < 0.5 then (
	    incr nsent ;
	    act := Send([random_member()],Ignore) :: !act
	  ) else (
	    nsent := !nsent + !nmembers - 1 ;
	    act := Cast(Ignore) :: !act
	  )
	done
      )
    | Ignore -> ()
    | _ -> ()
  ) ;
  !act

  and block ()			= []
  and heartbeat_rate		= (Time.of_float 1.0)
  and heartbeat time'		= 
    let old_time = !time in
    time := time' ;
(*
    eprintf "FIFO:heartbeat:time=%s, last_msg=%s\n" (Time.to_string !time) (Time.to_string !last_msg) ;
*)
(*
    if Time.sub old_time !last_msg > (Time.of_float 100.0) then (
      last_msg := time' ;
      [Dump]
    ) else
*)
      []
  and block_recv_cast _ _	= ()
  and block_recv_send _ _	= ()
  and block_view (ls,vs)        = [ls.rank,()]
  and block_install_view _ _	= ()
  and unblock_view (ls,vs) () =
    eprintf "FIFO:view=%s\n" (View.to_string vs.view) ;
    nmembers := ls.nmembers ;
    view := vs.view ;
    if my_endpt = vs.view.(0) && !nmembers > 1 then (
      nsent := !nsent + !nmembers - 1 ;
      [Cast(Token(!view.(1),0))]
    ) else []
  and exit () = exit 0
  in full {
    recv_cast		= recv_cast_send ;
    recv_send		= recv_cast_send ;
    heartbeat		= heartbeat ;
    heartbeat_rate	= heartbeat_rate ;
    block		= block ;
    block_recv_cast	= block_recv_cast ;
    block_recv_send	= block_recv_send ;
    block_view          = block_view ;
    block_install_view	= block_install_view ;
    unblock_view	= unblock_view ;
    exit		= exit
  }

(**************************************************************)

let run () =
  let props = Property.Drop :: Property.Debug :: Property.vsync in
  let props = List.map Property.string_of_id props in
  let props = String.concat ":" props in

  Arge.set_default alarm "Netsim" ;
  Arge.set_default modes "Netsim" ;
  Arge.set_default nmembers "5" ;
  Arge.set_default properties props ;

  Arge.parse [] (Arge.badarg name) 
  "fifo: fifo ordering test program" ;

  let instance () =
    let (ls,vs) = Appl.default_info "fifo" in
    let endpt = ls.endpt in
    let interface = intf endpt in
    Appl.config interface (ls,vs) ;
  in

  for i = 1 to Arge.get nmembers do
    instance ()
  done ;

  Appl.main_loop ()

let _ = Appl.exec ["fifo"] run

(**************************************************************)
