(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* REFLECT.ML: server side of gossip transport *)
(* Author: Mark Hayden, 8/95 *)
(**************************************************************)
open Util
open View
open Appl_intf
(**************************************************************)
let name = Trace.source_file "REFLECT"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

(* This function returns the interface record that defines
 * the callbacks for the application.
 *)
let intf (ls,vs) alarm sock =

  let max_msg_len = Hsys.max_msg_len () in
  let buf = String.create max_msg_len in
  let clients = Hashtbl.create 10 in

  (* The current view state.
   *)
  let vs = ref vs in
  let ls = ref ls in

  let update_client inet port =
    let time = Hsys.gettimeofday () in
    begin 
      try 
      	Hashtbl.remove clients (inet,port)
      with Not_found -> () 
    end ;
    Hashtbl.add clients (inet,port) time
  in

  let send_gossip msg =
    if !verbose then
      printf "REFLECT:send_gossip\n" ;
    let now = Hsys.gettimeofday () in

    let dests = ref [] in
    Hashtbl.iter (fun (inet,port) time ->
      dests := (inet,port,time) :: !dests
    ) clients ;
    
    List.iter (fun (inet,port,time) ->
      if time +. 60.0 > now then (
	let info = Hsys.preprocess sock [|inet,port|] in
        Hsys.sendopt info msg 0 (String.length msg) ; ()
      ) else (
      	Hashtbl.remove clients (inet,port)
      )
    ) !dests
  in

  let recv_gossip () =
    try
      let (len,inet,port) = Hsys.recvfrom sock buf 0 max_msg_len in
      if !verbose then
      	printf "REFLECT:recv_gossip from %s\n" (Hsys.string_of_inet inet) ;
      let msg = String.sub buf 0 len in
      update_client inet port ;
      if !ls.rank = 0 then (
	send_gossip msg
      )
    with e -> 
      if !verbose then
      	printf "REFLECT:warning:%s\n" (Hsys.error e)
  in
  Alarm.add_sock alarm sock (Hsys.Handler0 recv_gossip) ;

  (* Various application interface handlers.
   *)
  let recv_cast _ _ = failwith "error"
  and recv_send _ _ = failwith "error"
  and block () = []
  and heartbeat time = []
  and block_recv_cast _ _ = failwith "error"
  and block_recv_send _ _ = failwith "error"
  and block_view (ls,vs) =
    if ls.rank = 0 then (
      List.map (fun r -> r,()) (sequence ls.nmembers)
    ) else []
  and block_install_view _ _ = ()
  and unblock_view (ls',vs') () =
    vs := vs' ;
    ls := ls' ;
    if not (Arge.get Arge.quiet) then
      printf "REFLECT:view:%d:%s\n" !ls.nmembers (View.to_string !vs.view) ;
    []
  and exit () = ()
  in full {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    heartbeat           = heartbeat ;
    heartbeat_rate      = Time.of_float 10.0 ;
    block               = block ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
  }

(**************************************************************)

let init view_state port force =
  (*
   * Get default transport and alarm info.
   *)
  let alarm = Alarm.get () in

  (* Figure out port information and bind to it.
   *)
  let host = Hsys.gethostname () in
  let host = Hsys.inet_of_string host in
  if not (Arge.get Arge.quiet) then
    printf "REFLECT:server starting on %s\n" (Hsys.string_of_inet host) ;

  let hosts = Arge.check name Arge.gossip_hosts in

  (* Check that I'm in my list of gossip hosts.
   *)
  if (not force) 
  && (not (List.mem host hosts)) 
  then (
    if not (Arge.get Arge.quiet) then (
      printf "REFLECT:I'm not in the list of gossip hosts, exiting\n" ;
      printf "  (the hosts are %s)\n" 
        (string_of_list Hsys.string_of_inet hosts) ;
    ) ;
    exit 1
  ) ;

  if not (Arge.get Arge.quiet) then
    printf "REFLECT:server will use port %d\n" port ;

  (* Create datagram socket.
   *)
  let sock = Hsys.socket_dgram () in

  (* Bind it to the gossip port.
   *)
  begin
    try Hsys.bind sock host port with e ->
      if not (Arge.get Arge.quiet) then
      	printf "REFLECT:error:%s, exiting \n"
      	  (Hsys.error e) ;
      exit 1
  end ;

  if not (Arge.get Arge.quiet) then
    printf "REFLECT:server ready\n" ;

  (*
   * Initialize the application interface.
   *)
  let interface = intf view_state alarm sock in

  (view_state,interface)

(**************************************************************)
