(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PGP: interface to PGP *)
(* Authors: Zhen Xiao, Mark Hayden, 4/97 *)
(**************************************************************)
open Util
(**************************************************************)
let name = "PGP"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)
type principal = string

type cipher_text = string

type clear_text = string

let flatten_error s =
  string_map (fun c ->
    if c = '\n' then '|' else c
  ) s

(* sign the clear_text with the sender's secret key and encrypt it
 * with receiver's public key into ASCII format. The sender's pass
 * phrase, "pass", is needed to sign the message.
 * 
 * This function uses temporary file "tin" to store "clear_text", 
 * and "tin.asc" to store the encrypted file. They would be overwritten
 * if already exist.
 *)
let encrypt sender receiver clear_text =
  let command = sprintf "pgp -feas +batch %s -u %s" receiver sender in
  let (exit0,cipher,error) = Hsys.open_process command clear_text in
  if exit0 then (
    log (fun () -> "encrypt succeeded") ;
    Some(cipher)
  ) else (
    log (fun () -> "encrypt failed") ;
    log (fun () -> sprintf "pgp info=%s" (flatten_error error)) ;
    None
  )

(* decrypt the cipher_text with my secret key. My pass phrase, "pass"
 * is needed to decrypt the message. 
 *)

(* Open: Neither "sender" nor "receiver" is actually needed. First,
 * "sender" is not needed because it doesn't matter who sent the
 * cipher_text as long as decryption is concerned. Second, "receiver"
 * is not needed because the decryption is always done using the
 * user's pass phras.  
 * 
 * This function uses temporary file "tout.asc" to store "cipher_text",
 * and "tout" to store the decrypted file. They would be overwritten if 
 * already exist.
*)

let decrypt sender receiver cipher_text =
  let command = sprintf "pgp -f +batch %s -u %s" receiver sender in
  let (exit0,clear,error) = Hsys.open_process command cipher_text in
  if exit0 then (
    log (fun () -> "decrypt succeeded") ;
    Some(clear)
  ) else (
    log (fun () -> "decrypt failed") ;
    log (fun () -> sprintf "pgp info=%s" (flatten_error error)) ;
    None
  )

(* The following two functions are for debugging purpose. 
 * They are not part of this module.
 * 
 * Here I assume the pass phrase for "xiao" and "hayden" are just their
 * respective user id.
 *)

(*
let prstr s =
  match decrypt "xiao" "hayden" s with
  | Some s -> Printf.printf "%s\n" s
  | None -> Printf.printf "error\n"

let _ =  
 match encrypt "xiao" "hayden" "pgp is fun" with
 | Some s -> prstr s
 | None -> Printf.printf "error\n"
*)

let _ =
  let princ id nm =
    if id <> Addr.Pgp then 
      failwith "sanity" ;
    Addr.PgpA(nm)
  in

  let seal id src dst clear =
    let src = Addr.project src id in
    let dst = Addr.project dst id in
    match src,dst with
    | Addr.PgpA(src),Addr.PgpA(dst) ->
	encrypt src dst clear
    | _,_ -> None
  in

  let unseal id src dst clear =
    let src = Addr.project src id in
    let dst = Addr.project dst id in
    match src,dst with
    | Addr.PgpA(src),Addr.PgpA(dst) ->
	decrypt src dst clear
    | _,_ -> None
  in

  let auth =
    Auth.create
      name
      princ
      seal
      unseal
  in
  
  Auth.install Addr.Pgp auth

(**************************************************************)
