(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* MANAGE.ML *)
(* Author: Mark Hayden, 11/96 *)
(* Designed with Roy Friedman *)
(**************************************************************)
(* BUGS:

 * Does not provide EVS: members can fail and rejoin next view.

 *)
(**************************************************************)
open Util
open View
open Event
open Proxy
(**************************************************************)
let name = Trace.source_file "MANAGE"
let failwith s = failwith (name^":"^s)
(**************************************************************)

type ('m,'g) t = 
  | Actual of ('m,'g) Actual.t
  | Proxy of Proxy.t

let server_of name = function
  | Actual s -> s
  | Proxy _ -> failwith (name^":can't use proxy")

(**************************************************************)

let create (ls,vs) =
  let t,vs,intf = Actual.create (ls,vs) in
  Actual t, vs, intf

let proxy_server s p = 
  let s = server_of "server" s in
  Proxy.server p (Actual.join s)
		     
let proxy s = Proxy (Proxy.create s)

(**************************************************************)

let join s g e send =
  let join = 
    match s with
    | Actual s -> 
      	Actual.join s g e send
    | Proxy s ->
 	Proxy.join s g e send
  in
  join Join ;
  join

(**************************************************************)

let config s (ls,vs) stack =
  let log = Trace.log name ls.name in
  let endpt = ls.endpt in
  let endpt_full = View.endpt_full (ls,vs) in
  let ls = () in			(* bind over def of ls *)

  let view_r = ref vs.view in
  let to_server_r = ref (fun _ -> failwith "sanity") in
  let to_server_o m = !to_server_r m in

  let up ev = match getType ev with
  | EPrompt -> 
      log (fun () -> "Fail") ;
      to_server_o (Fail[])
  | ELeave -> 
      log (fun () -> "Fail(leave)") ;
      to_server_o (Fail([ma_endpt endpt_full]))
  | EBlockOk -> 
      log (fun () -> "Synced") ;
      to_server_o Synced
  | _ -> failwith "sanity"
  in
    
  let dn = stack up in

  let to_member msg = match msg with
  | Sync -> 
      log (fun () -> "Sync") ;
      dn (Event.create name EBlock[])
  | Failed(failed) ->
      log (fun () -> "Failed()") ;
      let failed = List.map um_endpt failed in
      let failede = List.map fst failed in
      let exiting = List.mem endpt failede in
      let ranks = 
	List.fold_left (fun l endpt -> 
	  try
	    (array_index endpt !view_r) :: l
	  with Not_found ->
	    log (fun () -> sprintf "warning:failure(%s) not in view(%s)"
	      (Endpt.string_of_id endpt) (View.to_string !view_r)) ;
	    l
	) [] failede
      in
      if ranks <> [] then (
	if exiting then (
	  dn (Event.create name EExit[]) ;
	  to_server_r := (fun _ -> eprintf "MANAGE:got msg for server after exit\n") ;
	) else (
	  log (fun () -> sprintf "view=%s" (View.to_string !view_r)) ;
	  log (fun () -> sprintf "failed=%s" (string_of_list Endpt.string_of_id failede)) ;
	  log (fun () -> sprintf "ranks=%s" (string_of_int_list ranks)) ;
	  log (fun () -> sprintf "endpt=%s" (Endpt.string_of_id endpt)) ;
	  dn (Event.create name EFail[Failures ranks]) ;
	)
      ) ;
  | View(ltime,view) ->
      log (fun () -> sprintf "View (ltime=%d,nmembers=%d)" ltime (List.length view)) ;
      let view = List.map um_endpt view in
      let view, address = List.split view in
      let view = Array.of_list view in
      let address = Array.of_list address in
      view_r := view ;
      let view_id = (ltime, view.(0)) in
      let vs = View.set vs [
	Vs_view view ;
	Vs_address address ;
	Vs_view_id view_id
      ] in
      dn (Event.create name EView[ViewState vs])
  in

  let endpt = ma_endpt endpt_full in
  let group = ma_group vs.group in
  to_server_r := join s group endpt to_member

(**************************************************************)

let set_properties s member handler =
  let handler group g = handler (um_group group) g in
  let s = server_of "properties" s in
  Actual.set_properties s member handler

let announce s group filter properties =
  let group = ma_group group in
  let s = server_of "announce" s in
  Actual.announce s group filter properties

let destroy s group =
  let group = ma_group group in
  let s = server_of "destroy" s in
  Actual.destroy s group

(**************************************************************)
