(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* ACTUAL.ML *)
(* Author: Mark Hayden, 11/96 *)
(* Designed with Roy Friedman *)
(**************************************************************)
(* BUGS:
 
 *  Orders members in views lexicographicly

 *)
(**************************************************************)
open Util
open View
open Proxy
open Trans
open Appl_intf
(**************************************************************)
let name = Trace.source_file "ACTUAL"
let failwith s = failwith (name^":"^s)
let log = Trace.log name ""
(**************************************************************)

let list_split4 l =
  List.fold_left (fun (la,lb,lc,ld) (a,b,c,d) ->
    (a::la,b::lb,c::lc,d::ld)
  ) ([],[],[],[]) l

let insert i l = Lset.union l [i]

let foreach l f = List.iter f l

let list_union l = List.fold_left Lset.union [] l

let list_filter2 f l1 l2 = 
  let l =
    List.fold_left2 (fun l a b ->
      if f a then b :: l else l
    ) [] l1 l2
  in Lset.sort l


(*begin					
    let tmp = Lset.subtract c.c_alive c.c_syncd in
  if List.length tmp < 5 then
    log (fun () -> sprintf "unsynced:%s"
      (string_of_list string_of_endpt tmp)) ;
  end ;*)	


(* LaTeX:begin *)
(* COORD: Record containing the state of a coordinator.
 * There is one such coordinator per group.  
 *)

type coord = {
  mutable c_syncing : bool ;            (* am I sync'ing? *)
  mutable c_ltime : int ;               (* view's logical time *)
  mutable c_alive : endpt list ;        (* endpoints in the group *)
  mutable c_syncd : endpt list ;        (* who is sync'ed *)
  c_broadcast : coord_msg -> unit	(* send a message to members **)
}

(* COORD_CHECK_VIEW: Helper function for coord_recv.  Check
 * if it is time for the coordinator to install a new view.
 *)

let coord_check_view c =
  log (fun () -> sprintf "#syncd=%d, #alive=%d"  (**)
    (List.length c.c_syncd) (List.length c.c_alive)) ;(**)
      
  if c.c_syncing && Lset.super c.c_syncd c.c_alive then ( 
    log (fun () -> sprintf "Coord:View:%s" (**)
      (string_of_list string_of_endpt c.c_alive)) ;(**)
    c.c_syncd <- [] ;
    c.c_ltime <- succ c.c_ltime ;
    c.c_syncing <- false ;
    c.c_broadcast (View(c.c_ltime,c.c_alive)) ;
  )
(* LaTeX:end *)

(**************************************************************)
(*let log = Trace.f "ACTUAL" (Proxy.string_of_endpt endpt)*)

(* LaTeX:begin *)
(* COORD_RECV: Handler for coordinator receiving a
 * membership message from a member.  
 *)

let coord_recv c (endpt,msg) =    
  (* Synchronize, if haven't done so already.
   *)
  if not c.c_syncing then (
    c.c_syncing <- true ;
    log (fun () -> sprintf "Coord:Cast(Sync)") ;(**)
    c.c_broadcast Sync
  ) ;

  (* Process the message.
   *)
  begin match msg with
  | Join ->
      (* Add the member to the group.
       *)
      log (fun () -> sprintf "Coord:Join") ;(**)
      c.c_alive <- insert endpt c.c_alive ;
      c.c_syncd <- insert endpt c.c_syncd ;

  | Synced ->
      (* Mark the member as being synchronized.
       *)
      log (fun () -> sprintf "Coord:Synced") ;(**)
      c.c_syncd <- insert endpt c.c_syncd ;

  | Fail(failed) ->
      (* Remove the members from the group.
       *)
      log (fun () -> sprintf "Coord:Fail") ;(**)
      c.c_alive <- Lset.subtract c.c_alive failed ;
      c.c_syncd <- Lset.subtract c.c_syncd failed ;
      if failed <> [] then
      	c.c_broadcast (Failed failed) ;
  end ;                                 (**)

  (* Check if the view is now ready.
   *)
  coord_check_view c
(* LaTeX:end *)

(**************************************************************)
(* STATE: A member is in one of 3 states:
 * Normal: normal state
 * Syncing: in the process of synchronizing.
 * Syncd: finished synchronizing.
 *)

(* LaTeX:begin *)
type state = Normal | Syncing | Syncd

(* MEMBER: Record containing the state of a member.  Each
 * client has one such record.  
 *)

type member = {
  m_endpt : endpt ;                     (* my endpoint *)
  mutable m_state : state ;             (* my state *)
  mutable m_ltime : int ;               (* my logical time *)
  mutable m_view : endpt list ;         (* view of group *)
  mutable m_coord : rank ;              (* who do I think is coordinator **)
  m_send_to_client : coord_msg -> unit ;(* send message to client **)
  m_send_to_coord : member_msg -> unit ;(* send message to coord **)
  m_disable : unit -> unit              (* disable this member **)
}               
(* LaTeX:end *)

(**************************************************************)

(* LaTeX:begin *)
(* MEMBER_RECV_CLIENT: Handler for a member to receive a
 * message from its client.  It just passes the message on
 * to the coordinator and updates the member's state.  
 *)

let member_recv_client m msg =
  m.m_send_to_coord msg ;
  match msg with
  | Join -> 
      m.m_state <- Syncd
  | Synced ->
      m.m_state <- Syncd
  | Fail(failed) -> () (* disable myself when I hear from Coord *)
(* LaTeX:end *)
 
(**************************************************************)
(*let loge = Trace.f "ACTUAL" (Proxy.string_of_endpt m.m_endpt)*)

(* LaTeX:begin *)
(* MEMBER_RECV_COORD: Handler for a member to receive a message
 * from coordinator.  The member checks the data, and passes
 * it on to its client.
 *)

let member_recv_coord m msg =
  match msg with
  | Sync ->
(* I think this case must be allowed, though I'm not sure. --MH
      if m.m_state = Syncing then
        failwith "Sync when state is Syncing" ;  (* Syncd is Ok *)
*)

      (* If in Normal state, then start synchronizing.
       *)
      if m.m_state = Normal then (
        log (fun () -> sprintf "Sync(accepted)") ;(**)
        m.m_state <- Syncing ;
        m.m_send_to_client msg
      )

  | Failed(failed) ->
      if m.m_state = Normal then
        failwith "Fail when state is Normal" ;

      (* If the member being failed is in my view,
       * then pass the message to the client.
       *)
      let in_view = Lset.intersect failed m.m_view in
      if in_view <> [] then (
      	log (fun () -> sprintf "Failed()") ;(**)
        m.m_send_to_client (Failed(in_view)) ;
      ) ;

      (* If I'm the one being failed, then disable me.
       *)
      if List.mem m.m_endpt failed then
        m.m_disable ()

  | View(ltime,view) ->
      log (fun () -> sprintf "View(ltime=%d,view=%s)"(**)
        ltime (string_of_list string_of_endpt view)) ;(**)
      if m.m_state <> Syncd then        (**)
        failwith "View when state isn't Syncd" ;(**)

      (* If I'm listed in the view, then install the
       * information.
       *)
      if List.mem m.m_endpt view then (
        if m.m_state <> Syncd then      (**)
          failwith "got View but not Syncd" ;(**)
        m.m_state <- Normal ;
        m.m_ltime <- ltime ;
        m.m_view <- view ;
        m.m_send_to_client msg ;
      )
(* LaTeX:end *)

(**************************************************************)

(* LaTeX:begin *)
(* RECONSTRUCT: Given lists of the fields of the members,
 * reinitialize a coordinator, and send out messages to
 * update members on the state of the group.
 *)

let coord_reconstruct c (endpts,states,ltimes,views) =
  (* Reconstruct fields as follows:
   *
   *  alive: sorted list of endpoints
   *  ltime: maximum of ltimes
   *  syncd: all members in Syncd state
   *  syncing: if any members is not in normal state
   *    or any member disagrees about the view.
   *)
  c.c_alive <- Lset.sort endpts ;
  c.c_ltime <- List.fold_left max 0 ltimes ;
  c.c_syncd <- list_filter2 (fun state -> state = Syncd) states endpts ;
  c.c_syncing <- 
     List.exists (fun state -> state <> Normal) states ||
     List.exists (fun view -> view <> c.c_alive) views ||
     List.exists (fun ltime -> ltime <> c.c_ltime) ltimes ;

  (* Send out Sync if necessary.
   *)
  if c.c_syncing then (
    log (fun () -> sprintf "recon:Sync") ;
    c.c_broadcast Sync ;
  ) ;
  
  (* Take the union of all views.
   *)
  let total = list_union views in

  (* Any of members not alive are failed.
   *)
  let failed = Lset.subtract total c.c_alive in
  if failed <> [] then (
    log (fun () -> sprintf "recon:Failed") ;
    c.c_broadcast (Failed failed) ;
  ) ;

  (* Check if view is ready.
   *)
  coord_check_view c

(* LaTeX:end *)
(**************************************************************)
(**************************************************************)
(**************************************************************)

let list_join l =
  let l = Lset.sort l in
  let l = 
    List.fold_left (fun cur (k,d) ->
      match cur with
      | [] -> [k,[d]]
      | (k',l) :: tl ->
          if k = k' then 
            (k',(d::l)) :: tl
          else
            (k,[d]) :: cur
    ) [] l
  in
  l

(**************************************************************)

type 'g message = 
  | ToCoord of group * endpt * rank(*of coord*) * member_msg
  | ToMembers of group * coord_msg
  | Announce of group * 'g
  | Destroy of group

type ('m,'g) t = {
  async : unit -> unit ;
  mutable ls : View.local ;
  mutable vs : View.state ;
  mutable announce : (group -> 'g -> unit) option ; (* hack! *)
  mutable blocking : bool ;
  ops : ('g message,'g message) Appl_intf.action Queue.t ;
  members : (group, member list) Hashtbl.t ;
  coords : (group, coord) Hashtbl.t ;
  groups : (group, 'g) Hashtbl.t
}

(**************************************************************)
(**************************************************************)

let string_of_msg = function
  | ToMembers(_,msg) ->
      let msg = match msg with
      | View(_,_) -> "View"
      | Failed(_) -> "Failed"
      | Sync -> "Sync"
      in sprintf "ToMembers(%s)" msg
  | ToCoord(_,_,rank,msg) -> 
      let msg = match msg with
      |	Join -> "Join"
      |	Synced -> "Synced"
      |	Fail(_) -> "Fail"
      in sprintf "ToCoord(%s)" msg
  | Announce _ -> "Announce"
  | Destroy _ -> "Destroy"

(**************************************************************)
(**************************************************************)

let coord s group f =
  let cast msg =
    if not (s.blocking) then (
      let msg = ToMembers(group,msg) in
(*    eprintf "sending %s\n" (string_of_msg msg) ; *)
      Queue.add (Cast msg) s.ops ;
      s.async ()
    )
  in

  let coord = 
    try Hashtbl.find s.coords group with Not_found ->
(*
      if s.vs.rank <> 0 then 
        failwith "non-coord is default" ;
*)
      let coord = {
        c_syncing = true ;
        c_syncd = [] ;
        c_ltime = 0 ;
        c_alive = [] ;
        c_broadcast = cast 
      } in
      Hashtbl.add s.coords group coord ;
      coord
  in
  f coord
  
let members s group f = 
  let members = 
    try Hashtbl.find s.members group
    with Not_found -> []
  in List.iter f members

(**************************************************************)

let join s group endpt to_client =
  (* BUG: there is a dependency below on having the coordinator
   * manage all the groups.
   *)
  let coord = 0 in

  let to_coord msg =
    let msg = ToCoord(group,endpt,coord,msg) in
    Queue.add (Send([coord],msg)) s.ops ;        (*BUG:PERF*)(* CAREFUL!! *)
    s.async ()
  in

  let disable () =
    log (fun () -> sprintf "disable:removing member") ;
    let members = Hashtbl.find s.members group in
    Hashtbl.remove s.members group ;
    let members = list_filter (fun m -> m.m_endpt <> endpt) members in
    Hashtbl.add s.members group members ;
  in
  
  let m = {
    m_endpt = endpt ;
    m_send_to_client = to_client ;
    m_send_to_coord = to_coord ;
    m_disable = disable ;
    m_view = [endpt] ;
    m_coord = coord ;
    m_state = Syncd ;
    m_ltime = 0 
  } in
  
  let members =
    try 
      let ret = Hashtbl.find s.members group in
      Hashtbl.remove s.members group ;
      ret
    with Not_found -> []
  in
  let members = m :: members in
  Hashtbl.add s.members group members ;
  log (fun () -> sprintf "#groups=%d" (hashtbl_size s.members)) ;
  member_recv_client m
  
(**************************************************************)
  
let create (ls_init,vs_init) =
  Util.disable_sigpipe () ;
  
  (* Ensure that we are using Total protocol.
   * Need local delivery.
   *)
  let properties = 
    if vs_init.groupd then
      Property.Total :: Property.fifo
    else
      Property.Total :: Property.vsync
  in

  let vs_init = View.set vs_init [
    Vs_proto_id (Property.choose properties)
  ] in
  
  (* Create the state of the group.
   *)
  let s = {
    vs = vs_init ;
    ls = ls_init ;
    announce = None ;
    async = Async.find (vs_init.group,ls_init.endpt) ;
    blocking = true ;
    ops = Queue.create () ;
    coords = Hashtbl.create 10 ;
    members = Hashtbl.create 10 ;
    groups = Hashtbl.create 10
  } in
  
  let announce group prop =
    try 
      (* BUG: should check that properties are the same. *)
      Hashtbl.find s.groups group ;
    with _ ->
      Hashtbl.add s.groups group prop ;
      if_some s.announce (fun handler ->
        handler group prop
      )
  in

  (* Handle a message receipt.
   *)
  let handle_msg msg = 
(*
    eprintf "got:%s\n" (string_of_msg msg) ;
*)
    match msg with
    | Announce(group,prop) ->
        announce group prop
    | Destroy(group) ->
        (try Hashtbl.remove s.groups group with _ -> ())
    | ToMembers(group,msg) ->
        (* Just pass message directly to the member.
         *)
        members s group (fun m -> member_recv_coord m msg)
    | ToCoord(group,endpt,rank,msg) ->
        (* Only handle message if I'm the requested rank.
         *)
        if s.ls.rank = rank then
          coord s group (fun c -> coord_recv c (endpt,msg))
  in

  let clean_ops () =
    let msgs = list_of_queue s.ops in
    Queue.clear s.ops ;
    msgs
  in
  
  let recv_cast _ msg       = handle_msg msg ; clean_ops ()
  and recv_send _ msg       = handle_msg msg ; clean_ops ()
  and block ()              =
    s.blocking <- true ;
    clean_ops ()
  and heartbeat time        =                  clean_ops ()
  and block_recv_cast _ msg = handle_msg msg
  and block_recv_send _ msg = handle_msg msg
    
  and block_view (ls,vs) =
    (* Remove all membership messages from the send
     * queue.  The data we're sending now should replace them.
     *)
    let l = list_of_queue s.ops in
    let l = list_filter (fun msg -> 
      match msg with
      | Cast(ToCoord _) | Cast(ToMembers _)
      | Send(_,ToCoord _) | Send(_,ToMembers _) -> false
      | _ -> true
    ) l in
    Queue.clear s.ops ;
    List.iter (fun m -> Queue.add m s.ops) l ;

    let members = list_of_hashtbl s.members in
    let members = 
      List.map (fun (group,members) ->
        let members =
          List.map (fun m -> (m.m_endpt,m.m_state,m.m_ltime,m.m_view)) members
        in
        (group, members)
      ) members
    in

    (* Collect all the group properties.
     *)
    let groups = list_of_hashtbl s.groups in
    let groups = Lset.sort groups in

    [ls.rank, (members, groups)]

  and block_install_view (ls,vs) comb =
    let (members,groups) = List.split comb in

    (* Merge, sort, & organize the group information.
     *)
    let members = List.flatten members in
    let members = 
      List.map (fun (group,members) -> 
        List.map (fun member -> group,member) members
      ) members 
    in
    let members = List.flatten members in
    let members = list_join members in

    (* Combine all the groups' propery lists.
     *)
    let groups = List.fold_right Lset.union groups [] in

    (members,groups)

  and unblock_view (ls,vs) (info,groups) =
    s.ls <- ls ;
    s.vs <- vs ;
    s.blocking <- false ;
    if not (Arge.get Arge.quiet) then
      eprintf "ACTUAL:view:%d:%s\n" ls.nmembers (View.to_string s.vs.view) ;

    (* Strip all groups from the table.
     *)
    hashtbl_clean s.coords ;

    (* Reconstruct coordinators.
     *)
    List.iter (fun (group,members) ->
      if ls.rank = 0 then (
        coord s group (fun c -> 
          let lists = list_split4 members in
          coord_reconstruct c lists ;
        )
      ) ;
    ) info ;

    List.iter (fun (group,prop) ->
      announce group prop
    ) groups ;

    clean_ops ()
  and exit () = () in

  let interface = {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    heartbeat           = heartbeat ;
    heartbeat_rate      = Time.of_float 
      (if Arge.get Arge.aggregate then 0.05 else 10.0) ;
    block               = block ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
  } in

  let interface =
    if Arge.get Arge.aggregate then aggr interface else full interface
  in
  
  (s,(ls_init,vs_init(* BUG? *)),interface)
  
let set_properties s member handler =
  s.announce <- Some handler ;
  ()
  
let announce s group filter properties =
  let msg = Announce(group,properties) in
  Queue.add (Cast msg) s.ops ;
  s.async ()
  
let destroy s group =
  let msg = Destroy(group) in
  Queue.add (Cast msg) s.ops ;
  s.async
 ()
  
(**************************************************************)
