(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* DEBUG.ML *)
(* Author: Mark Hayden, 11/96 *)
(**************************************************************)
open Trans
open Util
(**************************************************************)
let name = Trace.source_file "DEBUG"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)

type message =
  | Add of name
  | Remove of name
  | Logs_request
  | Logs_reply of name list      
  | Data of name * string * string
      
let marsh,unmarsh = Util.make_marsh name true

(**************************************************************)

let server host port =
  let alarm = Alarm.get () in

  let host' = Hsys.inet_of_string (Hsys.gethostname ()) in
  if host <> host' then
    failwith "not on right host" ;

  let sock = Hsys.socket_stream () in
  Hsys.setsockopt sock Hsys.Reuse ;
  Hsys.bind sock host port ;
  Hsys.listen sock 5 ;
  let names = ref [] in
  let conns = ref [] in
  
  let client_init sock =
    let clients = Hashtbl.create 100 in
    let connected = ref true in
    
    let send,recv = Hsyssupp.tcp sock in
    let send msg =
      if !connected then (
	let msg = marsh msg in
	send msg 0 (String.length msg) ;
	()
      )
    in
    List.iter (fun name ->
      send (Add(name))
    ) !names ;

    conns := (sock,send) :: !conns ;

    let recv () =
      match recv () with
      |	Some msgs ->
	  List.iter (fun msg ->
	    let msg = unmarsh msg 0 (String.length msg) in
	    match msg with
	    | Data(kind,id,data) ->
	    	printf "%s:%s:%s\n" kind id data
	    | Logs_reply(names) ->
	    	printf "DEBUGS:logs:%s\n" (String.concat "," names)
	    | _ -> failwith "sanity"
	  ) msgs ;
      |	None ->
	  if !verbose then
	    eprintf "DEBUG:lost connection\n" ;
	  connected := false ;
	  Alarm.rmv_sock alarm sock ;
	  Hsys.close sock ;
	  conns := List.fold_right (fun (sock',send) cnew ->
	    if sock <> sock' then 
	      (sock',send) :: cnew
	    else cnew
          ) !conns []
    in
    recv
  in
  
  let svr_handler () =
    let client,host,_ = Hsys.accept sock in
    eprintf "DEBUG:got connection from %s\n" (Hsys.string_of_inet host) ;
    Alarm.add_sock alarm client (Hsys.Handler0 (client_init client))
  in
  Alarm.add_sock alarm sock (Hsys.Handler0 svr_handler) ;

  let stdin = Hsys.stdin () in
  let input_handler () =
    try 
      List.iter (fun line ->
	let toks = string_split " \t" line in
	match toks with
	| ["add";name] -> 
	    let name = String.uppercase name in
	    List.iter (fun (sock,send) ->
	      send (Add(name))
	    ) !conns ;
	    printf "DEBUG:added '%s' to debugged names\n" name
	| ["remove";name] -> 
	    let name = String.uppercase name in
	    List.iter (fun (sock,send) ->
	      send (Remove(name))
	    ) !conns ;
	    printf "DEBUG:removed '%s' from debugged names\n" name
	| ["logs"] ->
	    List.iter (fun (sock,send) ->
	      send Logs_request
	    ) !conns ;
	| _ -> 
	    eprintf "DEBUG:bad command:line='%s'\n" line
      ) (read_lines ())
    with e ->
      eprintf "DEBUG:error:%s\n" (string_of_exn e)
  in
  Alarm.add_sock alarm stdin (Hsys.Handler0 input_handler)

(**************************************************************)

let client host port =
  let alarm = Alarm.get () in
  let sock = Hsys.socket_stream () in
  if !verbose then
    eprintf "DEBUG:connecting to server\n" ;
  Hsys.connect sock host port ;
  if !verbose then
    eprintf "DEBUG:connected to server\n" ;
  let logs = ref [] in
  
  let connected = ref true in

  let send,recv = Hsyssupp.tcp sock in
  let send msg =
    if !connected then (
      let msg = marsh msg in
      send msg 0 (String.length msg) ;
      ()
    )
  in
  
  let recv () =
    match recv () with
    | Some msgs ->
    	List.iter (fun msg ->
	  let msg = unmarsh msg 0 (String.length msg) in
	  match msg with
	  | Add(name) ->
	      let send id data = send (Data(name,id,data)) in
	      Trace.add_log name send ;
	      logs := name :: !logs
	  | Remove(name) ->
	      Trace.rmv_log name ;
	      logs := except name !logs
	  | Logs_request ->
	      let names = Trace.list_of_logs () in
	      send (Logs_reply names)
	  | _ -> failwith "sanity"
        ) msgs ;
    | None ->
      	eprintf "DEBUG:lost connection to server\n" ;
      	Hsys.close sock ;
      	Alarm.rmv_sock alarm sock ;
	List.iter (fun name -> Trace.rmv_log name) !logs ;
      	connected := false
  in
  Alarm.add_sock alarm sock (Hsys.Handler0 recv)

(**************************************************************)
