(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* EARG.ML *)
(* Author: Mark Hayden, 4/96 *)
(**************************************************************)
open Proto
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "ARGE"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
let logu = Trace.log (name^"U") ""
(**************************************************************)

let table_default = ref []

let get_default key =
  try
    Some(List.assoc key !table_default)
  with Not_found -> None

let table_comline = ref []

let get_comline key =
  try
    Some(List.assoc key !table_comline)
  with Not_found -> None

let get_env key =
  let environ_prefix = "ENS_" in
  let key = String.uppercase key in
  let key = environ_prefix^key in
  try 
    Some (Sys.getenv key) 
  with Not_found -> None 

(**************************************************************)

type 'a t = 
  { name : name ;
    convert : string -> 'a ;
    default : 'a } 

let create name conv default = {
  name = name ; 
  convert = conv ;
  default = default
}

let tables = ref [get_comline;get_default;get_env]

let get a = 
  let bindings = 
    List.map (fun f -> f a.name) !tables
  in
  let bindings = Util.filter_nones bindings in
  let binding = 
    match bindings with
    | value :: _ -> 
  	logu (fun () -> sprintf "%s = %s" 
	  (String.lowercase a.name) value) ;
  	a.convert value
    | [] -> a.default
  in  
  binding

(**************************************************************)

let check debug v =
  match get v with
  | None ->
      eprintf "%s:error:%s has not been set\n" debug v.name ;
      eprintf "  (set with [-%s val] command-line argument or\n" v.name ;
      eprintf "   the ENS_%s environment variable)\n" (String.uppercase v.name) ;
      exit 1
  | Some v -> v

(**************************************************************)

let set_comline t data =
  table_comline := (t.name,data) :: !table_comline

let set_default t data =
  table_default := (t.name,data) :: !table_default

let param_args = ref (Some [])

(* General case argument manager.  All the others use this.
 *)
let general name default info conv unit_arg =
  let v = create name (conv name) default in

  (* Then add the command-line argument.
   *)
  begin
    let com_set s = set_comline v s in
    let com_set =
      if unit_arg then
      	Arg.Unit(fun () -> com_set "true")
      else 
	Arg.String(com_set)
    in
    let com_name = "-"^name in
    let com_info = ": "^info in
    match !param_args with
    | None -> failwith "sanity"
    | Some a ->
    	param_args := Some((com_name,com_set,com_info) :: a) ;
  end ;

  v

(**************************************************************)

(* Handle string-based args.
 *)
let string name default info conv =
  general name default info conv false

(* Handle int-based args.
 *)
let int name default info conv =
  let conv name s = 
    let i = 
      try int_of_string s with Failure _ ->
      	eprintf "ARGE:bad value for %s:'%s' not integer, exiting\n" name s ;
      	exit 1
    in
    conv name i 
  in
  general name default info conv false

(* Handle bool-based args.
 *)
let bool name default info conv =
  let conv name s = 
    let b = 
      try bool_of_string s with Failure _ ->
      	eprintf "ARGE:bad value for %s:'%s' not boolean, exiting\n" name s ;
      	exit 1
    in
    conv name b 
  in
  general name default info conv true

(**************************************************************)

let set_ident name = ident

let set_f f name = f

let set_option name a = Some a

let set_properties name prop =
  let prop = string_split ":" prop in
  let prop = List.map Property.id_of_string prop in
  prop

let set_modes name modes =
  let modes = string_split ":" modes in
  let modes = 
    try List.map Addr.id_of_string modes with e ->
      eprintf "ARGE:bad value for %s, exiting\n" name ;
      eprintf "  (modes={%s}, error=%s\n" 
        (String.concat ", " modes) (Hsys.error e) ;
      exit 1
  in	
  modes

let set_string_list name = Util.string_split ":"

let set_inet_list_option name hosts =
  let hosts = Util.string_split ":" hosts in
  let hosts = 
    try List.map Hsys.inet_of_string hosts with e -> 
      eprintf "ARGE:bad value for %s, exiting\n" name ;
      eprintf "  (hosts={%s}, error=%s\n" 
        (String.concat ", " hosts) (Hsys.error e) ;
      exit 1
  in
  (Some hosts)

let set_inet_option name host =
  let host = 
    try Hsys.inet_of_string host with e -> 
      eprintf "ARGE:bad value for %s, exiting\n" name ;
      eprintf "  (host={%s}, error=%s\n" host (Hsys.error e) ;
      exit 1
  in
  (Some host)

let set_groupd properties name _ = true

let set_glue name s =
  try Glue.of_string s
  with Failure _ ->
    eprintf "ARGE:unknown glue '%s'\n" s ;
    exit 1

let set_time name t =
  let t = float_of_string t in
  let t = Time.of_float t in
  t

let set_port_range name ports =
  let ports = Util.string_split "-" ports in
  match ports with
  | [lo;hi] ->
      let lo = int_of_string lo in
      let hi = int_of_string hi in
      (Some(lo,hi))
  | _ -> 
      eprintf "ARGE:bad port range (usage: -%s lo-hi)\n" name ;
      exit 1

(**************************************************************)

let aggregate    = bool "aggregate" false "aggregate messages" set_ident
let alarm        = string "alarm" Alarm.Real "set alarm package to use" (set_f Alarm.id_of_string)
let force_modes  = bool "force_modes" false "disable transport modes checking" set_ident
let glue         = string "glue" Glue.Imperative "set layer glue to use" set_glue
let gossip_hosts = string "gossip_hosts" None "set hosts for gossip servers" set_inet_list_option
let gossip_port  = int "gossip_port" None "set port for gossip server" set_option
let group_name	 = string "group_name" "ensemble" "set default group name" set_ident
let groupd_balance = bool "groupd_balance" false "load balance groupd servers" set_ident
let groupd_hosts = string "groupd_hosts" None "set hosts for groupd servers" set_inet_list_option
let groupd_port  = int "groupd_port" None "set port for groupd server" set_option
let id           = string "id" "" "set 'user id' of process" set_ident
let key          = string "key" None "set security key" set_option
let log          = bool "log" false "use remote log server" set_ident
let log_host     = string "log_host" None "set hosts for log server" set_inet_option
let log_port     = int "log_port" None "set port for log server" set_option
let modes        = string "modes" [Addr.Udp;Addr.Tcp] "set default transport modes" set_modes
let nmembers     = int "nmembers" 2 "set number of members to use (for some applications)" set_ident
let port         = int "port" None "set port for UDP, TCP, Deering" set_option
let properties   = string "properties" Property.vsync "set default stack properties" set_properties
let groupd       = bool "groupd" false "use groupd server" (set_groupd properties)
let quiet        = bool "quiet" false "quiet execution" set_ident
let roots        = bool "roots" false "print information about Ensemble roots" set_ident
let sp2_suffixes = string "sp2_suffix" ["sw";"-hps"] "set hostname suffixes for SP2 fast interconnect" set_string_list
let port_range   = string "port_range" None "set range of ports Ensemble can bind to" set_port_range
let pgp          = string "pgp" None "set my id for using pgp" set_option
let pollcount    = int "pollcount" 10 "number of polling operations before blocking" set_ident
let refcount     = bool "refcount" false "use reference counting" set_ident
let multiread    = bool "multiread" false "read all data from sockets before delivering" set_ident
let sched_step   = int "sched_step" 200 "number of events to schedule before reading from network" set_ident
let udp_host     = string "udp_host" None "set host for udp communication" set_inet_option

let perturb      = bool "perturb" false "perturb this process" set_ident
let perturb_rate = string "perturb_rate" (0.1) "fraction of time asleep" (set_f float_of_string)
let perturb_doze = string "perturb_doze" (Time.of_float 0.1) "granularity of sleeping" set_time

(**************************************************************)

let set_properties prop =
  let prop = List.map Property.string_of_id prop in
  let prop = String.concat ":" prop in
  set_comline properties prop

let add_properties p =
  let p = string_split ":" p in
  let p = List.map Property.id_of_string p in
  set_properties (Lset.union (get properties) p)

let add_scale () =
  add_properties "Scale"

let add_total () =
  add_properties "Total"

let set_fifo () =
  set_properties Property.fifo

(**************************************************************)

let remove_properties p =
  let p = string_split ":" p in
  let p = List.map Property.id_of_string p in
  set_properties (Lset.subtract (get properties) p)

(**************************************************************)

let set_thread () =
  set_comline glue "Threaded" ;
  set_comline alarm "Threaded"		(* must use Threaded Alarm *)

(**************************************************************)

let trace name =
  let name = String.uppercase name in
  let f info dbg =
    if info = "" then 
      eprintf "%s\n" (String.concat ":" [name;dbg])
    else 
      eprintf "%s\n" (String.concat ":" [name;info;dbg])
  in
  Trace.add_log name f

(**************************************************************)

let logger server shed =
  Alarm.choose (get alarm) shed ;
  let alarm = Alarm.get () in
  let port = check name log_port in
  let host = check name log_host in
  if server then (
    Debug.server host port ;
    while true do 
      Alarm.block alarm
    done
  ) else (
    Debug.client host port
  )

(**************************************************************)
(* Seed the random number generator with the current time.
 *)

let seed () = 
  let time = Hsys.gettimeofday () in
  let time = sprintf "%60.30f" time in
  let time = Digest.string time in
  let len = String.length time in
  let seed = array_create name len 0 in
  for i = 0 to pred len do
    seed.(i) <- Char.code time.[i]
  done ;
  Random.full_init seed

(**************************************************************)
(* Print version and contact information.
 *)

let version () =
  eprintf "This application is using Ensemble version %s.\n"
    (Version.string_of_id Version.id) ;
  eprintf "  more information can be found at our web page:\n" ;
  eprintf "    http://www.cs.cornell.edu/Info/Projects/Ensemble/index.html\n" ;
  exit 0

(**************************************************************)

let stats () =
  Glue.stats ()

(**************************************************************)

let param_help debug cb s =
  let s = string_split "=" s in
  match s with
  | [name;value] ->
    cb name value
  | _ -> 
      eprintf "ARGE:bad format [-%s name=value], exiting\n" debug ;
      exit 1

let param_int =
  param_help "pint" (fun name value ->
    let value = int_of_string value in
    Param.default name (Param.Int value)
  )

let param_string =
  param_help "pstring" (fun name value ->
    Param.default name (Param.String value)
  )

let param_time =
  param_help "ptime" (fun name value ->
    let value = float_of_string value in
    let value = Time.of_float value in
    Param.default name (Param.Time value)
  )

let param_float =
  param_help "pfloat" (fun name value ->
    let value = float_of_string value in
    Param.default name (Param.Float value)
  )

let param_bool =
  param_help "pbool" (fun name value ->
    let value = 
      try bool_of_string value with Failure _ ->
      	eprintf "ARGE:bad value for parameter %s:'%s' not boolean, exiting\n" name value ;
	exit 1
    in
    Param.default name (Param.Bool value)
  )

(**************************************************************)

let args () = 
  (* We "zero" the args after the first use to eliminate
   * clutter int the major heap.
   *)
  let param_args = match !param_args with
  | None -> failwith "args used more than once"
  | Some a ->
      param_args := None ;
      a
  in

  let addtl_args = [
    "-add_prop",        Arg.String(add_properties), ": add protocol properties" ;
    "-fifo",            Arg.Unit set_fifo, ": set fifo stack" ;
(*
    "-log_server",      Arg.Unit(fun () -> logger true), ": run as log server" ;
*)
    "-pbool",           Arg.String(param_bool), ": set boolean parameter" ;
    "-pfloat",          Arg.String(param_float), ": set float parameter" ;
    "-pint",            Arg.String(param_int), ": set integer parameter" ;
    "-print_comments",	Arg.Unit(fun () -> Trace.print_comments () ; exit 0), ": print active comments in modules" ;
    "-print_config",	Arg.Unit(fun () -> Trace.print_config () ; exit 0), ": print configuration information" ;
    "-print_defaults",	Arg.Unit(fun () -> Param.print_defaults () ; exit 0), ": print default parameter settings" ;
    "-pstr",            Arg.String(param_string), ": set string parameter" ;
    "-ptime",           Arg.String(param_time), ": set time parameter" ;
    "-remove_prop",     Arg.String(remove_properties), ": remove protocol properties" ;
    "-scale",           Arg.Unit add_scale, ": add scalable property" ;
    "-secure",		Arg.Unit Transport.set_secure, ": prevent insecure communication" ;
    "-seed",		Arg.Unit(seed), ": seed the random number generator" ;
    "-test",		Arg.String(Trace.exec_test), ": run named test (-print_config lists tests)" ;
    "-thread",          Arg.Unit(set_thread), ": use threaded layers" ;
    "-time",            Arg.String(Timestamp.use), ": enable timers for performance testing" ;
    "-total",           Arg.Unit add_total, ": add total ordering property" ;
    "-trace",		Arg.String(trace), ": enable named trace messages" ;
    "-v",               Arg.Unit(version), ": print version" ;
    "-verbose",		Arg.Set Util.verbose, ": enable verbose messages"
  ] in
  let args = addtl_args @ param_args in
  let args = Sort.list (fun (a,_,_) (b,_,_) -> a < b) args in
  args

(**************************************************************)

let badarg name s =
  eprintf "%s: unknown argument '%s'\n" name s ;
  exit 1

let parse app_args badarg doc =
  let args = app_args @ (args ()) in
  let args = Sort.list (fun (a,_,_) (b,_,_) -> a < b) args in
  Arg.parse (app_args @ args) badarg doc

(**************************************************************)

let check_port name port = 
  match get port_range with
  | None -> ()
  | Some(lo,hi) ->
      if port < lo || port > hi then (
	eprintf "ARGE: %s port being allocated outside range\n" name ;
	eprintf "  (range is set with -port_range)\n" ;
	exit 1
      )

(**************************************************************)
