(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* APPL.ML *)
(* Author: Mark Hayden, 4/95 *)
(**************************************************************)
open Util
open View
(**************************************************************)
let name = Trace.source_file "APPL"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

let root_sched = Stacke.root_sched

(**************************************************************)

let addr modes =
  let addr = 
    List.map (fun mode ->
      let dom = Domain.of_mode mode in
      Domain.addr dom mode
    ) modes
  in 
  let addr = Array.of_list addr in
  Addr.set_of_array addr

(**************************************************************)

let full_info name groupd protocol modes key =
  let force_modes = Arge.get Arge.force_modes in
  let pgp = Arge.get Arge.pgp in
  let id = Arge.get Arge.id in

  (* If ENS_ID environment variable is set, add that as
   * prefix of name so that users do not interact with each
   * other unless using same id.  
   *)
  let group = 
    let name = 
      if id = "" then name 
      else sprintf "%s:%s" id name
    in
    Group.named name 
  in

  let modes = 
    if not (List.mem Addr.Deering modes) 
    || Hsys.has_ip_multicast ()
    then modes else (
      eprintf "APPL:warning:DEERING mode unavailable (removing from modes)\n" ;
      except Addr.Deering modes
    )
  in

  if not force_modes then (
    if list_filter Addr.has_mcast modes = [] then (
      eprintf "APPL:no mcast modes were selected\n" ;
      Addr.error (Array.of_list modes)
    ) ;
    
    if list_filter Addr.has_pt2pt modes = [] then (
      eprintf "APPL:no pt2pt modes were selected\n" ;
      Addr.error (Array.of_list modes)
    )
  ) ;
    
  let addr = addr modes in
  let addr = Addr.array_of_set addr in

  let addr = match Arge.get Arge.pgp with
  | None -> addr
  | Some nm ->
      let auth = Auth.lookup Addr.Pgp in
      let addr = Array.append [|Auth.principal auth Addr.Pgp nm|] addr in
      addr
  in

  let addr = Addr.set_of_array addr in
  let endpt = Endpt.id () in
      
  
  let (ls,vs) = View.singleton key protocol group endpt addr in
  let vs = View.set vs [Vs_groupd groupd] in
  (ls,vs)

(**************************************************************)

let default_info name =
(*
  Trace.log "CONFIG" "default_info" (fun () -> Arge.get Arge.alarm_str) ;
*)
  let groupd = Arge.get Arge.groupd in
  let modes = Arge.get Arge.modes in
  let properties = Arge.get Arge.properties in
  let key = Arge.get Arge.key in

  let key = match key with
  | None -> Security.NoKey
  | Some s -> Security.Common s
  in

  Alarm.choose (Arge.get Arge.alarm) root_sched ;

  let properties =
    if groupd then (
      if not (Arge.get Arge.quiet) then
        eprintf "APPL:warning:stripping unnecessary properties for '-groupd'\n" ;
      Property.strip_groupd properties
    ) else properties
  in      

  let protocol = Property.choose properties in
  full_info name groupd protocol modes key

(**************************************************************)

let exec names run =
  let appl = Sys.argv.(0) in		(* get executable's name *)
  let appl = Filename.basename appl in	(* chop directory name *)
  let appl =				(* chop extension for nt *)
    try Filename.chop_extension appl 
    with Invalid_argument _ -> appl
  in
  let appl = String.lowercase appl in
  if List.mem appl names then (
    try
(*
      Gc.full_major () ;
      List.iter (fun s ->
    	printf "APPL:exec:%s\n" s
      ) (string_list_of_gc_stat (Gc.stat ())) ;
*)
(*
      Sys.catch_break true ;
*)
      run () ;
      exit 0 
    with 
    | Sys.Break ->
	exit 0
    | exc ->
      let error_msg =
	try Hsys.error exc with _ ->
	  string_of_exn exc 
      in
      eprintf "ENSEMBLE:uncaught exception:%s\n" error_msg ;
      eprintf "  (Ensemble Version.id is %s)\n" (Version.string_of_id Version.id) ;
      exit 1
  )

(**************************************************************)

let init_groupd =
  let groupd = ref None in
  fun () ->
    match !groupd with 
    | None ->
	let port = Arge.check name Arge.groupd_port in
	if !verbose then
	  eprintf "APPL:init_groupd:using port %d\n" port ;
	let try_connect host =
	  let sock = Hsys.socket_stream () in
	  if !verbose then
	    eprintf "APPL:attempting connection to %s\n" (Hsys.string_of_inet host) ; 
	  try
	    Hsys.connect sock host port ;
	    if !verbose then
	      eprintf "APPL:connected to server\n" ;
	    Some sock
	  with e ->
	    if !verbose then
	      eprintf "APPL:connection failed (%s)\n" (Hsys.error e) ;
	    Hsys.close sock ;
	    None
	in

	let host = Hsys.inet_of_string (Hsys.gethostname ()) in
	let sock =
	  match try_connect host with
	  | Some sock -> sock
	  | None ->
	      let hosts = Arge.check name Arge.groupd_hosts in

	      (* Randomize the list of hosts for load balancing.
	       *)
	      let hosts = 
		if Arge.get Arge.groupd_balance then
		  Util.random_list hosts
		else 
		  hosts
	      in
	      let rec loop = function
		| [] ->
		    let hosts = List.map (Hsys.string_of_inet) (host::hosts) in
		    let hosts = string_of_list ident hosts in
		    eprintf "APPL:no groupd server found at %s, exiting\n" hosts ;
		    exit 1
		| hd::tl ->
		    match try_connect hd with
		    | None -> loop tl
		    | Some sock -> sock
	      in loop hosts
	in
	let m = Manage.proxy sock in
	groupd := Some m ;
	m
    | Some m -> m


let config interface (ls,vs) =
  if vs.groupd then (
    let member = Stacke.config_full interface (ls,vs) in
    Manage.config (init_groupd ()) (ls,vs) member
  ) else (
(*
    Gc.full_major () ;
    List.iter (fun s ->
      printf "APPL:config:b:%s\n" s
    ) (string_list_of_gc_stat (Gc.stat ())) ;
*)    
    Stacke.config interface (ls,vs) ;
(*
    Gc.full_major () ;
    List.iter (fun s ->
      printf "APPL:config:a:%s\n" s
    ) (string_list_of_gc_stat (Gc.stat ())) ;
*)
  )

(**************************************************************)
(* Randomly perturb this process.
 *)

let perturb alarm doze rate =
  let next = ref (Alarm.gettime alarm) in

  let disabled = ref false in
  let disable () =
    disabled := true
  in

  let handler_r = ref (fun _ -> ()) in
  let handler time = !handler_r time in
  let schedule = Alarm.alarm alarm handler in
  handler_r := (fun time ->
    if !disabled then (
      Alarm.disable schedule
    ) else (
      while !next <= Alarm.gettime alarm do
	next := Time.add !next doze ;
      	if Random.float 1.0 < rate then (
	  Hsys.select [] [] [] (Time.to_float doze) ;
	)
      done ;
      
      Alarm.schedule schedule !next 
    )
  ) ;
  
  handler (Alarm.gettime alarm) ;
  disable

(**************************************************************)

let start_monitor () =
  let log_gc = Trace.logl "GC" "" in
  let log_refcnt = Trace.logl "REFCNTM" "" in
  let alarm = Alarm.get () in
  let sched_r = ref None in
  let roots = Arge.get Arge.roots in

  let loop time =
    if time > Time.zero then (
      log_refcnt (fun () ->
	let pool = Mbuf.pool Mbuf.global in
	Refcnt.pool_debug pool
      ) ;

      log_gc (fun () ->
(*
	Gc.full_major () ;
*)
	string_list_of_gc_stat (Gc.stat ())
      ) ;

      if roots then
	Trace.print_roots () ;
    ) ;
    Alarm.schedule (some_of !sched_r) (Time.add time (Time.of_float 10.0))
  in 

  sched_r := Some (Alarm.alarm alarm loop) ;
  loop Time.zero

(**************************************************************)

let async id =
  let async = Async.find id in
  let async () = Sched.enqueue root_sched async in
  async

(**************************************************************)

let main_loop () =
  let sched_step = Sched.step root_sched in
  let iter_to_block = Arge.get Arge.pollcount in
  let count = Arge.get Arge.sched_step in
  let alarm = Alarm.get () in
  let counter = ref 0 in
  start_monitor () ;

  let rec loop0 () =
    (* Check for messages.
     *)
    let got_msgs = Alarm.poll alarm Alarm.SocksPolls in
(*
    let got_msgs = Alarm.poll alarm Alarm.OnlyPolls in
*)
    (* Schedule events in the layers.
     *)
    let got_events = sched_step count in

    (* BUG: perf *)
    if Alarm.check alarm || got_msgs || got_events then
(*
    if got_msgs || got_events || Alarm.check alarm then
*)
      loop0 ()
  in

  (* Called first because lower loop is a little
   * backwards.
   *)
  loop0 () ;

  let rec loop1 () =
    (* If no events, consider blocking.
     *)
    decr counter ;
    if !counter < 0 then (
      counter := iter_to_block ;
      (* One last try...
       *)
      if not (Alarm.poll alarm Alarm.SocksPolls) then (
        Alarm.block alarm ;
      )
    ) ;

    (* This goes after the blocking part to reduce
     * the reaction time when unblocking.
     *)
    loop0 () ;
    loop1 ()
  in
  
  loop1 ()

(**************************************************************)

let main_loop_opt () =
  let sched_step = Sched.step root_sched in
  let alarm = Alarm.get () in

  let f () =
    while true do
      (* Check for messages.
       *)
      Alarm.poll alarm Alarm.OnlyPolls ;

      (* Schedule some events in the layers.
       *)
      sched_step 200 ;

      Alarm.check alarm
    done
  in
  
  Printexc.catch f ()

(**************************************************************)

let _ =
  if Arge.get Arge.refcount then (
    if not (Arge.get Arge.quiet) then 
      eprintf "APPL:warning:using reference counted Iovecs\n" ;
    Refcnt.actually_use_counts := true ;
    Refcnt.force_major_gc := false ;
  )

(**************************************************************)
