import javax.swing.Icon;

/**
 * The <code>Piece</code> class manages all static information about
 * the pieces, such as their value, count, images.
 * Instances of piece can be created and used on the board.
 * @author  James Ezick
 * @version 2.0, 02/11/00
 */

class Piece {

  ///////////////////////////////////////////////
  // INSTANCE
  ///////////////////////////////////////////////

  /**
   * Store letter that this piece instance represents
   */
  protected char letter;

  /**
   * Creates a piece of a certain letter
   * @param char piece code
   */
  public Piece(char c) {
    // this might throw an ArrayIndexOutOfBounds exception
    int i=getIndex(c); 
    letter=(char)('A'+i);
  }

  /**
   * Must exist for subclasses to be happy
   */
  protected Piece() {}

  /**
   * Returns letter on this piece
   */
  public char getLetter() {
    return letter;
  }

  /**
   * Returns value of this piece
   */
  public int getValue() {
    return getValue(getLetter());
  }

  /**
   * Returns image of this piece
   */
  public Icon getIcon() {
    return getIcon(getLetter());
  }

  /**
   * Returns piece count/frequency
   */
  public int getCount() {
    return getCount(getLetter());
  }

  /**
   * Converts piece information to string
   */
  public String toString() {
    return ""+getLetter();
  }

  ///////////////////////////////////////////////
  // STATIC
  ///////////////////////////////////////////////

  /**
   * Value/score of each letter.
   */
  private static final int VALUE[] = {
    1, 3, 3, 2, 1, 4, 2, 4, 1, 8,
    5, 1, 3, 1, 1, 3,10, 1, 1, 1,
    1, 4, 4, 8, 4, 10
  };

  /**
   * Count/frequency of each letter.
   */
  private static final int COUNT[] = {
    9, 2, 2, 4,12, 2, 3, 2, 9, 1,
    1, 4, 2, 6, 8, 2, 1, 6, 4, 6,
    4, 2, 2, 1, 2, 1
  };

  /** 
   * Letter images
   */
  private static Icon IMAGES[];
  
  static {
    Icon tmpImages[];
    try {
      java.io.ObjectInputStream in=
        new java.io.ObjectInputStream(
        new java.util.zip.GZIPInputStream(
        new java.io.FileInputStream("IMAGES.bin")));
      byte pics[][]=(byte[][])in.readObject();
      tmpImages = new Icon[pics.length-1];
      for(int i=0;i<pics.length-1;i++)
        tmpImages[i]=new javax.swing.ImageIcon(pics[i]);
    }
    catch (Exception e) {
      tmpImages = null;
    }
    IMAGES = tmpImages;
  }

  /**
   * Returns value of a given piece
   */
  public static int getValue(char c) {
    return VALUE[getIndex(c)];
  }

  /**
   * Returns image of this piece
   */
  public static Icon getIcon(char c) {
    return IMAGES[getIndex(c)];
  }

  /**
   * Returns piece count/frequency
   */
  public static int getCount(char c) {
    return COUNT[getIndex(c)];
  }

  /**
   * Calculates internal array index 
   * Instances are created via the create methods
   * @param char piece code
   */
  protected static int getIndex(char c) {
    if(c>='a' && c<='z') return c-'a';
    if(c>='A' && c<='Z') return c-'A';
    throw new IndexOutOfBoundsException("Invalid letter: "+c);
  }

  /**
   * Converts an array of pieces to a string
   */
  public static String toString(Piece word[]) {
    StringBuffer s=new StringBuffer("");
    for(int i=0;i<word.length;i++)
      s.append(word[i].getLetter());
    return s.toString();
  }
}
