import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;

/**
 * The <code>BoardView</code> class displays Board information to the 
 * user. Instances of BoardView are grid-like components that observe 
 * Board objects.
 * @author  James Ezick
 * @version 2.0, 02/11/00
 */
class BoardView extends JComponent implements Observer {

  /**
   * Store references to buttons so that Icons can be later changed
   */
  private JButton grid[][];

  /**
   * Construct BoardView component
   */
  public BoardView(final Board board) {
    // set layout manager to grid
    setLayout(new GridLayout(Board.SIZE, Board.SIZE));

    // extract width, height of an Icon (they are all the same)
    Icon icon=(new WildcardPiece()).getIcon();
    Dimension d=new Dimension(icon.getIconWidth(), icon.getIconHeight());

    // add all the buttons
    grid=new JButton[Board.SIZE][];
    for(int row=0; row<Board.SIZE; row++) {
      grid[row]=new JButton[Board.SIZE];
      for(int col=0; col<Board.SIZE; col++) {
        // store reference to button in array, for later updates
        grid[row][col]=new JButton();
        // set size of button to fit icon exactly
        grid[row][col].setMinimumSize(d);
        grid[row][col].setMaximumSize(d);
        grid[row][col].setPreferredSize(d);
        // set color of square
        grid[row][col].setBackground(Board.getSquareColor(row,col));
        // create final copies of counters to be use in 
        // anonymous listener object
        final int r=row, c=col;
        // add action listener
        grid[row][col].addActionListener(new ActionListener() {
            /**
             * This method gets called when button is clicked
             */
            public void actionPerformed(ActionEvent e) {
              // print out row, col coordinate of button
              System.out.println("("+r+","+c+")");
              // cycle letter at that position on board
              char letter;
              if(board.getPiece(r,c)!=null)
                letter=board.getPiece(r,c).getLetter();
              else {
                board.setPiece(r,c,new Piece('a'));
                return;
              }
              if(Character.toLowerCase(letter)=='z')
                board.clearPiece(r,c);
              else
                board.setPiece(r,c,new Piece(++letter));
            }
          });
        
        // finally add this button to be displayed
        add(grid[row][col]);
      }
    }
    // add ourselves to the board's observer list
    board.addObserver(this);
    // update once to get the state of the board right now
    // (maybe it's not empty!)
    update(board,null);
  }

  /**
   * Called whenever Board data changes
   * Just set the squares to the correct icons
   */
  public void update(Observable o, Object arg) {
    setSquareLetters((Board)o);
  }

  /**
   * Set the icon on each square to that representing
   * the piece on the corresponding location on the board
   */
  private void setSquareLetters(Board b) {
    // for each row, col
    // set icon of each button (null == no piece)
    for(int row=0; row<Board.SIZE; row++)
      for(int col=0; col<Board.SIZE; col++)
        grid[row][col].setIcon( b.getPiece(row,col)==null ?
            null : b.getPiece(row,col).getIcon() );
  }

  /**
   * Construct complete user interface
   */
  public static JFrame createPlayerWindow(final Board board, 
                                          final JavlleClient jc) 
  {
    // create new window
    JFrame frame=new JFrame("cs202 - Javlle");
    // make it respond to close events
    frame.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent e) {
          System.exit(0);
        }
      });

    Container pane=frame.getContentPane();

    // create panel for Javvle labels at top
    JPanel label=new JPanel();
    label.add(new JLabel(Piece.getIcon('j')));
    label.add(new JLabel(Piece.getIcon('a')));
    label.add(new JLabel(Piece.getIcon('v')));
    label.add(new JLabel(Piece.getIcon('l')));
    label.add(new JLabel(Piece.getIcon('l')));
    label.add(new JLabel(Piece.getIcon('e')));

    // create go button for bottom, add action listener
    JButton button=new JButton("Go!");
    button.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          try {
            jc.boardSend(board);
          }
          catch (IOException e1) {
            System.err.println("Error sending board");
          }
        }
      });

    // put it all together, with BoardView in the CENTER
    pane.setLayout(new BorderLayout());
    pane.add(label, BorderLayout.NORTH);
    pane.add(new BoardView(board), BorderLayout.CENTER);
    pane.add(button, BorderLayout.SOUTH);

    // don't allow user to resize
    frame.setResizable(false);
    // pack components
    frame.pack();
    // display window
    frame.show();
    return frame;
  }
}
