/* uniform sampling kernel SVM */
#include <stdio.h>
#include <assert.h>
#include "svm_common.h"
#include "./SFMT-src-1.3.3/SFMT.h"


#define ALPHA_THRESHOLD 1E-10
#define IDLE_ITER 20
#define CLEANUP_CHECK 50

long kernel_cache_statistic=0;

/* mosek interface */
int mosek_qp_optimize(double**, double*, double*, long, double, double*);

void my_read_input_parameters(int argc, char* argv[], char *trainfile, char *testfile,
char *modelfile,
			      LEARN_PARM *learn_parm, KERNEL_PARM *kernel_parm);

void my_wait_any_key();

int resize_cleanup(int size_active, int m, long SAMPLE_SIZE, int *idle, 
		   double *alpha, double *delta, double **inner_prod_score, double **G_sample,
		   long **all_sample_idx, long *all_size_c);
void init_model(MODEL* model, int num_sv, long totdoc, long totwords, KERNEL_PARM *kernel_parm);


/* assume only one svector in doc */
double my_kernel(KERNEL_PARM *kernel_parm, DOC *a, DOC *b) {
  kernel_cache_statistic++;
  
  switch (kernel_parm->kernel_type) {
  /* linear kernel */
  case 0: return((double) sprod_ss(a->fvec,b->fvec));
  /* polynomial */
  case 1: return((double)pow(kernel_parm->coef_lin*sprod_ss(a->fvec,b->fvec)+kernel_parm->coef_const,(double)kernel_parm->poly_degree)); 
  /* RBF with gamma=0.05 */
  case 2: return((double) exp(-kernel_parm->rbf_gamma*(a->fvec->twonorm_sq-2*sprod_ss(a->fvec,b->fvec)+b->fvec->twonorm_sq)));
  default: printf("Unrecognized kernel_type %ld!\n", kernel_parm->kernel_type); exit(0);
  }
}


int main(int argc, char* argv[]) {
  DOC **X;
  double *Y;
  long totwords, totdoc, test_totwords, test_totdoc;
  double **G_sample, *G_sample_col;
  double *Xw, *alpha, *delta;
  double **inner_prod_score;
  long *sample_idx;
  long *ind;
  long i, j, k, m, temp_idx, size_c;
  int iter, r;
  double value, xi, temp;
  double C, epsilon;
  LEARN_PARM learn_parm;
  KERNEL_PARM kernel_parm;
  char trainfile[1024];
  char testfile[1024];
  char modelfile[1024];
  long MAX_ITER;
  long SAMPLE_SIZE;
  long **all_sample_idx;
  long *all_size_c;
  long runtime_start, runtime_end;

  /* for testing */
  DOC **test_X;
  double *test_Y;
  double *train_Xw, *test_Xw;
  long train_error, test_error;
  double margin_loss;
  /* for clean up */
  int *idle;
  int size_active;
  double dual_obj;
  /* stopping criterion */
  long p;
  long temp_size_c;
  long *temp_sample_idx;
  double *temp_inner_prod_score;
  MODEL *model=(MODEL *)my_malloc(sizeof(MODEL));
  
  double *beta;
  long num_sv;
  long *sv_idx;
  
  
  /* read input parameters */
  my_read_input_parameters(argc, argv, trainfile, testfile, modelfile, &learn_parm, &kernel_parm); 

  epsilon = learn_parm.eps;
  C = learn_parm.svm_c;
  MAX_ITER = learn_parm.maxiter;
  SAMPLE_SIZE = learn_parm.svm_maxqpsize;


  /* input in svm^light format */
  read_documents(trainfile, &X, &Y, &totwords, &totdoc);

  /* initialize the RNG */
  init_gen_rand(learn_parm.biased_hyperplane);

  /* start timer, exclude I/O */
  runtime_start = get_runtime();


  /* initialization */
  m = totdoc;
  xi = 0.0;
  G_sample = NULL;
  G_sample_col = NULL;
  delta = NULL;
  alpha = NULL;
  all_sample_idx = NULL;
  all_size_c = NULL;
  inner_prod_score = NULL;
  sample_idx = (long*) my_malloc(sizeof(long)*SAMPLE_SIZE);
  Xw = (double*) my_malloc(sizeof(double)*SAMPLE_SIZE);
  ind = (long*) my_malloc(sizeof(long)*SAMPLE_SIZE);
  idle = NULL;
  p=0;
  beta = (double*) my_malloc(sizeof(double)*m);
  sv_idx = (long*) my_malloc(sizeof(long)*m);


  iter = 0;
  size_active = 0;
  
  /* sample a constraint */
  size_c = 0;
  value = 0.0;
  for (i=0;i<SAMPLE_SIZE;i++) {
    temp_idx = (long) floor(genrand_res53()*m);
    sample_idx[i] = temp_idx;
    Xw[i] = 0.0;
  
    for (j=0;j<size_active;j++) {
      inner_prod_score[j][i] = 0.0;
	  for (k=0;k<all_size_c[j];k++) {
	    inner_prod_score[j][i] += Y[all_sample_idx[j][k]]*my_kernel(&kernel_parm, X[all_sample_idx[j][k]], X[temp_idx]);
	  }
	  inner_prod_score[j][i]/=SAMPLE_SIZE;
	  Xw[i] += alpha[j]*inner_prod_score[j][i];
    }
    if (1-Y[temp_idx]*Xw[i]>0) {
      ind[size_c] = i;
	  size_c++;
	  value += (1-Y[temp_idx]*Xw[i]);
    }
  }
  value/=SAMPLE_SIZE;


  while ((p<=learn_parm.remove_inconsistent)&&(iter<MAX_ITER)) {
    iter+=1;
	size_active+=1;
	/* allocate more space for inner_prod_score */
	inner_prod_score = (double**) realloc(inner_prod_score, sizeof(double*)*size_active);
	assert(inner_prod_score!=NULL);
	inner_prod_score[size_active-1] = (double*) my_malloc(sizeof(double)*SAMPLE_SIZE);

	/* allocate more space for alpha */
	alpha = (double*) realloc(alpha, sizeof(double)*size_active);
	assert(alpha!=NULL);
	alpha[size_active-1] = 0.0;

    /* update cleanup information */
    idle = (int*) realloc(idle, sizeof(int)*size_active);
    assert(idle!=NULL);
    idle[size_active-1] = 0;

    /* update G, delta*/
    delta = (double*) realloc(delta, sizeof(double)*size_active);
    assert(delta!=NULL);
    delta[size_active-1] = ((double) size_c)/SAMPLE_SIZE;
  
    G_sample_col = (double*) realloc(G_sample_col, sizeof(double)*size_active);
    assert(G_sample_col!=NULL);
    for (j=0;j<size_active-1;j++) {
      G_sample_col[j] = 0.0;
	  for (k=0;k<size_c;k++) {
	    G_sample_col[j] += Y[sample_idx[ind[k]]]*inner_prod_score[j][ind[k]];
	  }
	  G_sample_col[j]/=SAMPLE_SIZE;
    }

    G_sample_col[size_active-1] = 0.0;
    for (j=0;j<size_c;j++) {
	  for (k=j+1;k<size_c;k++) {
	    G_sample_col[size_active-1] += 2*Y[sample_idx[ind[j]]]*Y[sample_idx[ind[k]]]*my_kernel(&kernel_parm, X[sample_idx[ind[j]]], X[sample_idx[ind[k]]]);
	  }
	  G_sample_col[size_active-1] += my_kernel(&kernel_parm, X[sample_idx[ind[j]]], X[sample_idx[ind[j]]]);
	}
    G_sample_col[size_active-1]/=(SAMPLE_SIZE*SAMPLE_SIZE);
  
  
    G_sample = (double**) realloc(G_sample, sizeof(double*)*size_active); 
    assert(G_sample!=NULL);
	G_sample[size_active-1] = NULL;
    for (j=0;j<size_active;j++) {
      G_sample[j] = (double*) realloc(G_sample[j], sizeof(double)*size_active);
	  assert(G_sample[j]!=NULL);	
    }
    for (j=0;j<size_active-1;j++) {
	  G_sample[j][size_active-1] = G_sample_col[j];
	  G_sample[size_active-1][j] = G_sample_col[j];
    }
    G_sample[size_active-1][size_active-1] = G_sample_col[size_active-1];


    /* update all_size_c, all_sample_idx */
    all_size_c = (long*) realloc(all_size_c, sizeof(long)*size_active); 
    assert(all_size_c!=NULL); 
    all_size_c[size_active-1] = size_c;

    all_sample_idx = (long**) realloc(all_sample_idx, sizeof(long*)*size_active);
    assert(all_sample_idx!=NULL); 
    all_sample_idx[size_active-1] = (long*) my_malloc(sizeof(long)*SAMPLE_SIZE);
    assert(all_sample_idx[size_active-1]!=NULL);  
    for (i=0;i<size_c;i++) {
      all_sample_idx[size_active-1][i] = sample_idx[ind[i]]; 
    }
  
    /* solve QP to update alpha */
    r = mosek_qp_optimize(G_sample, delta, alpha, (long) size_active, C, &dual_obj);

    /* update xi */
    xi = 0.0;
    for (j=0;j<size_active;j++) {
      temp = 0.0;
	  for (k=0;k<size_active;k++) {
	    temp += G_sample[j][k]*alpha[k];
	  }
      if (delta[j]-temp>xi) {
        xi = delta[j]-temp; 
	  }
    }

	
    /* sample a constraint */
	p = 0;
	while ((p<=learn_parm.remove_inconsistent)&&((p==0)||(value<=xi+epsilon))) {
	  for (i=0;i<SAMPLE_SIZE;i++) {
	    sample_idx[i] = (long) floor(genrand_res53()*m);
	    Xw[i] = 0.0;
	  }
	  for (j=0;j<size_active;j++) {
	    temp_size_c = all_size_c[j];
	    temp_sample_idx = all_sample_idx[j];
	    temp_inner_prod_score = inner_prod_score[j];
	    for (i=0;i<SAMPLE_SIZE;i++) {
	      temp_inner_prod_score[i] = 0.0;
	      for (k=0;k<temp_size_c;k++) {
			temp_inner_prod_score[i] += Y[temp_sample_idx[k]]*my_kernel(&kernel_parm, X[sample_idx[i]], X[temp_sample_idx[k]]);
	      }
	      temp_inner_prod_score[i]/=SAMPLE_SIZE;
	      Xw[i] += alpha[j]*temp_inner_prod_score[i];
	    }
	  }
	  size_c=0;
	  value=0.0;
	  for (i=0;i<SAMPLE_SIZE;i++) {
	    if (1-Y[sample_idx[i]]*Xw[i]>0) {
	      ind[size_c]=i;
	      size_c++;
	      value += (1-Y[sample_idx[i]]*Xw[i]);
	    }
	  }
	  value/=SAMPLE_SIZE;

	  p++;
    }

    /* print iteration */
    printf("Iteration %d\n", iter);

    /* print out iteration statistics */
    printf("Margin Loss: %.8g\n", value);
    printf("Violation: %.8g\n", value - xi); 
	
    /* update cleanup information */
    for (i=0;i<size_active;i++) {
      if (alpha[i]<ALPHA_THRESHOLD*C) {
	    idle[i]++;
      } else {
	    idle[i]=0;
      }
    }
    /* cleanup */
    if (iter % CLEANUP_CHECK==0) {
      size_active = resize_cleanup(size_active, m, SAMPLE_SIZE, idle, alpha, delta, inner_prod_score, G_sample, all_sample_idx, all_size_c); 
    }
	
  }  /* end while */

  printf("ITER END\n"); fflush(stdout);
  
  /* end timer */
  runtime_end = get_runtime();

  printf("Training time in cpu seconds (excluding I/O): %.2f\n", ((float) runtime_end - (float) runtime_start)/100.0);  
  
  /* test on training set */
  train_error = 0;
  margin_loss = 0.0;
  train_Xw = (double*) my_malloc(sizeof(double)*m);
  for (i=0;i<m;i++) {
    train_Xw[i] = 0.0;
    for (j=0;j<size_active;j++) {
      temp = 0.0;
	  for (k=0;k<all_size_c[j];k++) {
	    temp += Y[all_sample_idx[j][k]]*my_kernel(&kernel_parm, X[all_sample_idx[j][k]], X[i]);
	  }
	  temp/=SAMPLE_SIZE;
	  train_Xw[i] += alpha[j]*temp;
    }

	if (Y[i]*train_Xw[i]<=0) {
		train_error += 1;
	}
	if (1-Y[i]*train_Xw[i]>0) {
		margin_loss += (1-Y[i]*train_Xw[i]);
	}
  }
  printf("Training Set Error: %.4f\n", ((double) train_error)/m);
  fflush(stdout);
  
  /* test on test set */
  read_documents(testfile, &test_X, &test_Y, &test_totwords, &test_totdoc);

  test_error = 0;
  margin_loss = 0.0;
  test_Xw = (double*) my_malloc(sizeof(double)*test_totdoc);
  for (i=0;i<test_totdoc;i++) {
    test_Xw[i] = 0.0;
	for (j=0;j<size_active;j++) {
	  temp = 0.0;
	  for (k=0;k<all_size_c[j];k++) {
	    temp += Y[all_sample_idx[j][k]]*my_kernel(&kernel_parm, X[all_sample_idx[j][k]], test_X[i]);
	  }
	  temp/=SAMPLE_SIZE;
	  test_Xw[i] += alpha[j]*temp;
	}

	if (test_Y[i]*test_Xw[i]<=0) {
	  test_error += 1;
	}
	if (1-test_Y[i]*test_Xw[i]>0) {
	  margin_loss += (1-test_Y[i]*test_Xw[i]);
	}
  }
  printf("Test Set Error: %.4f\n", ((double) test_error)/test_totdoc);
  fflush(stdout);
  
  
  /* save model */
  for (i=0;i<m;i++) {
    beta[i] = 0.0;
  }
  for (j=0;j<size_active;j++) {
    for (k=0;k<all_size_c[j];k++) {
	  beta[all_sample_idx[j][k]] += alpha[j]/SAMPLE_SIZE;
	}
  }
  num_sv = 0;
  for (i=0;i<m;i++) {
    beta[i]/=m; 
    if (beta[i]>1E-8) {
      sv_idx[num_sv] = i;
      num_sv++;
    }
  }
  
  /* write to model file */
  init_model(model,num_sv,totdoc,totwords,&kernel_parm);
  for (i=0;i<num_sv;i++) {
    model->supvec[i+1] = X[sv_idx[i]];
	model->alpha[i+1] = Y[sv_idx[i]]*beta[sv_idx[i]];
  }
  write_model(modelfile, model);

  
  /* free the memory */
  for (i=0;i<size_active;i++) {
    free(G_sample[i]);
  }
  free(G_sample);
  free(G_sample_col);
  free(alpha);
  free(delta);
  free(Xw);
  free(sample_idx);
  free(idle);
  
  for (i=0;i<size_active;i++) {
    free(all_sample_idx[i]);
	free(inner_prod_score[i]);
  }
  free(all_sample_idx);
  free(inner_prod_score);
  free(all_size_c);
  free(ind);
  
  /* free examples and labels */
  for (i=0;i<m;i++) {
	free_example(X[i],1);
  }
  free(X);
  free(Y);
  
  for (i=0;i<test_totdoc;i++) {
    free_example(test_X[i],1);
  }
  free(test_X);
  free(test_Y);
  
  free(train_Xw);
  free(test_Xw);
  
  return(0);
}






void my_read_input_parameters(int argc, char *argv[], char *trainfile, char *testfile, char *modelfile,
			   LEARN_PARM *learn_parm, KERNEL_PARM *kernel_parm) {
  
  long i;

  /* set default */
  learn_parm->maxiter=300;
  learn_parm->svm_maxqpsize=100;
  learn_parm->svm_c=100.0;
  learn_parm->eps=0.001;
  learn_parm->biased_hyperplane=12345; /* store random seed */
  learn_parm->remove_inconsistent=10; 
  kernel_parm->kernel_type=0;
  kernel_parm->rbf_gamma=0.05;
  kernel_parm->coef_lin=1;
  kernel_parm->coef_const=1;
  kernel_parm->poly_degree=3;
  strcpy(kernel_parm->custom,"empty");

  for(i=1;(i<argc) && ((argv[i])[0] == '-');i++) {
    switch ((argv[i])[1]) {
    case 'c': i++; learn_parm->svm_c=atof(argv[i]); break;
    case 'e': i++; learn_parm->eps=atof(argv[i]); break;
    case 's': i++; learn_parm->svm_maxqpsize=atol(argv[i]); break; /* use svm_maxqpsize to store SAMPLE_SIZE */
    case 'g': i++; kernel_parm->rbf_gamma=atof(argv[i]); break;
    case 'd': i++; kernel_parm->poly_degree=atol(argv[i]); break;
    case 'r': i++; learn_parm->biased_hyperplane=atol(argv[i]); break; /* random seed */
    case 't': i++; kernel_parm->kernel_type=atol(argv[i]); break;
    case 'n': i++; learn_parm->maxiter=atol(argv[i]); break;
	case 'p': i++; learn_parm->remove_inconsistent=atol(argv[i]); break; /* number of consecutive non-violation before stopping */
    default: printf("\nUnrecognized option %s!\n\n",argv[i]);
      exit(0);
    }


  }

  if(i>=argc) {
    printf("\nNot enough input parameters!\n\n");
    my_wait_any_key();
    exit(0);
  }
  strcpy (trainfile, argv[i]);

  if((i+1)<argc) {
    strcpy (testfile, argv[i+1]);
  }
  if((i+2)<argc) {
    strcpy (modelfile, argv[i+2]);
  }

}

void my_wait_any_key()
{
  printf("\n(more)\n");
  (void)getc(stdin);
}

int resize_cleanup(int size_active, int m, long SAMPLE_SIZE, int *idle, 
		   double *alpha, double *delta, double **inner_prod_score, double **G_sample,
		   long **all_sample_idx, long *all_size_c) {
  int i,j,new_size_active;
  long k;

  i=0;
  while ((i<size_active)&&(idle[i]<IDLE_ITER)) i++;
  j=i;
  while ((j<size_active)&&(idle[j]>=IDLE_ITER)) j++;
  
  while (j<size_active) {
    /* copying */
    alpha[i] = alpha[j];
    delta[i] = delta[j];
    all_size_c[i] = all_size_c[j];
    for (k=0;k<size_active;k++) {
      G_sample[i][k] = G_sample[j][k];
    }

	for (k=0;k<all_size_c[j];k++) {
		all_sample_idx[i][k] = all_sample_idx[j][k];
	}		

	for (k=0;k<SAMPLE_SIZE;k++) {
	  inner_prod_score[i][k] = inner_prod_score[j][k];
	}
    i++;
    j++;
    while ((j<size_active)&&(idle[j]>=IDLE_ITER)) j++;
  }
  new_size_active = i;
  alpha = (double*) realloc(alpha, sizeof(double)*new_size_active);
  delta = (double*) realloc(delta, sizeof(double)*new_size_active);
  all_size_c = (long*) realloc(all_size_c, sizeof(long)*new_size_active);
  for (k=i;k<size_active;k++) {
    free(G_sample[k]);
    free(all_sample_idx[k]);
	free(inner_prod_score[k]);
  }

  inner_prod_score = (double**) realloc(inner_prod_score, sizeof(double*)*new_size_active);
  
  G_sample = (double**) realloc(G_sample, sizeof(double*)*new_size_active);
  all_sample_idx = (long**) realloc(all_sample_idx, sizeof(long*)*new_size_active);
  /* second pass for idle and G_sample */
  i=0;
  while ((i<size_active)&&(idle[i]<IDLE_ITER)) i++;
  j=i;
  while ((j<size_active)&&(idle[j]>=IDLE_ITER)) j++;

  while (j<size_active) {
    idle[i] = idle[j];
    for (k=0;k<new_size_active;k++) {
      G_sample[k][i] = G_sample[k][j];
    }
    i++;
    j++;
    while ((j<size_active)&&(idle[j]>=IDLE_ITER)) j++;
  }  
  idle = (int*) realloc(idle, sizeof(int)*new_size_active);
  for (k=0;k<new_size_active;k++) {
    G_sample[k] = (double*) realloc(G_sample[k], sizeof(double)*new_size_active);
  }
  return(new_size_active);
}


void init_model(MODEL* model, int num_sv, long totdoc, long totwords, KERNEL_PARM *kernel_parm) {

  model->supvec = (DOC **)my_malloc(sizeof(DOC *)*(num_sv+2));
  model->alpha = (double *)my_malloc(sizeof(double)*(num_sv+2));

  model->at_upper_bound=0;
  model->b=0;	       
  model->supvec[0]=0;  /* element 0 reserved and empty for now */
  model->alpha[0]=0;
  model->lin_weights=NULL;
  model->totwords=totwords;
  model->totdoc=totdoc;
  model->kernel_parm=(*kernel_parm);
  model->sv_num=1+num_sv;
  model->loo_error=-1;
  model->loo_recall=-1;
  model->loo_precision=-1;
  model->xa_error=-1;
  model->xa_recall=-1;
  model->xa_precision=-1;

}
